/******************************************************************************/
/* psx-blk.c                                   Block Management Interface     */
/******************************************************************************/
/** @file psx-blk.c Block Management Interface - Source Code File
 * Functions supporting the handling of memory block datastructures.
 */
 
#include "psx.h"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** validate size and reallocate memory for block blk
 * @param blk pointer to block structure
 * @param s total (new) size of block
 */

static BOOL blk_validate (PSX_BLK * blk,NUM s)
{
 PTR * p;

 if (blk == NULL)
  return (FALSE);

 if (blk -> siz >= s)                      // Block is valid
  return (TRUE);

 if ((p = realloc (blk -> ptr,s)) == NULL) // fit memory to new size
  return (FALSE);

 blk -> ptr = p;                           // new values
 blk -> siz = s;

 return (TRUE);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** allocate space for structure blk, initialize
 * @param blk pointer to block structure
 */

BOOL blk_create (PSX_BLK ** blk)
{
 PSX_BLK * b;

 if (blk == NULL)
  return (FALSE);

 if ((b = SYS_ALLOCATE (PSX_BLK)) == NULL)
  return (FALSE);

 b -> ptr = NULL;
 b -> len = 0;
 b -> siz = 0;
 b -> idx = 0;

 *blk = b;

 return (TRUE);
}

/******************************************************************************/
/** free structure space, free data space
 * @param blk pointer to block structure
 */

BOOL blk_delete (PSX_BLK ** blk)
{
 if (blk == NULL)
  return (FALSE);

 if (*blk == NULL)
  return (FALSE);

 blk_clr (*blk);

 free (*blk);
 *blk = NULL;

 return (TRUE);
}

/******************************************************************************/
/** clear block
 * @param blk pointer to block structure
 */

BOOL blk_clr (PSX_BLK * blk)
{
 if (blk == NULL)
  return (FALSE);

 if (blk -> ptr)
 {
  free (blk -> ptr);
  blk -> ptr = NULL;
 }

 memset (blk,0,sizeof (*blk));

 return (TRUE);
}

/******************************************************************************/
/** copy block to block 
 * @param blk pointer to block structure containing copy of src after invocation
 * @param src pointer to source block
 */

BOOL blk_copy (PSX_BLK * blk,PSX_BLK * src)
{
 if (blk == NULL || src == NULL)
  return (FALSE);

 if (!blk_validate (blk,src -> len))
  return (FALSE);

 memcpy (blk -> ptr,src -> ptr,src -> len);   // copy block
 blk -> len = src -> len;

 return (TRUE);
}

/******************************************************************************/
/** read from block
 * @param blk pointer to block source
 * @param p pointer to copy of block contents (return)
 * @param n size of copied memory
 */

BOOL blk_read (PSX_BLK * blk,PTR p,NUM n)
{
 if (blk == NULL || p == NULL)
  return (FALSE);

 if (blk -> idx + n > blk -> len)
  return (FALSE);

 memcpy (p,blk -> ptr,n);
 blk -> idx += n;

 return (TRUE);
}

/******************************************************************************/
/** write block and validate
 * @param blk pointer to block (return)
 * @param p pointer to string that has t be written in blk
 * @param n size of memory to be copied
 */

BOOL blk_write (PSX_BLK * blk,PTR p,NUM n)
{
 if (blk == NULL || p == NULL)
  return (FALSE);

 if (!blk_validate (blk,blk -> idx + n))
  return (FALSE);

 memcpy (PTR_OFS (blk -> ptr,blk -> idx),p,n); // PTR_OFS(p,o)sets to Offset
 blk -> idx += n;

 if (blk -> idx > blk -> len)
  blk -> len = blk -> idx;

 return (TRUE);
}

/******************************************************************************/
/** set index position
 * @param blk pointer to block structure
 * @param pos new block index
 */

BOOL blk_setpos (PSX_BLK * blk,IDX pos)
{
 if (blk == NULL)
  return (FALSE);

 if (pos >= blk -> len)
  return (FALSE);

 blk -> idx = pos;

 return (TRUE);
}

/******************************************************************************/
/** get index position
 * @param blk pointer to block structure
 * @param pos index of current block
 */

BOOL blk_getpos (PSX_BLK * blk,IDX * pos)
{
 if (blk == NULL || pos == NULL)
  return (FALSE);

 *pos = blk -> idx;

 return (TRUE);
}

/******************************************************************************/
/** validate and put block from src to target after Offset
 * @param blk pointer to block structure
 * @param src pointer to src string
 * @param n number of characters to copy
 */

BOOL blk_put (PSX_BLK * blk,const char * src,NUM n) //was PTR instead of char *
{
 if (!blk_validate (blk,blk -> idx + n)) // check block, set new size
  return (FALSE);

 memcpy (PTR_OFS (blk -> ptr,blk -> idx),src,n);

 blk -> idx += n;                        // set index
 blk -> len = blk -> idx;                // set block length

 return (TRUE);
}

/******************************************************************************/
/** put block without validation and without Offset
 * @param blk pointer to block structure
 * @param src source string
 */

BOOL blk_put_s (PSX_BLK * blk,const char * src)
{
 if (blk == NULL || src == NULL)
  return (FALSE);

 if (!blk_put (blk,src,strlen (src)))   // write src in block
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** get block to destination
 * @param blk pointer to block
 * @param dst string pointer to destination of copying
 */

BOOL blk_get_s (PSX_BLK * blk,char ** dst)
{
 if (blk == NULL || dst == NULL)
  return (FALSE);

 if ((*dst = (char *) malloc ((blk -> len + 1) * sizeof (char))) == NULL)
  return (FALSE);

 memcpy (*dst,blk -> ptr,blk -> len);         // copy block
 (*dst) [blk -> len] = 0;

 return (TRUE);
}

/******************************************************************************/
#define BLK_PRINTF_MAX 10000
/******************************************************************************/
/** print block
 * @param blk pointer to block structure
 * @param fmt format string
 * @param ... variable argument list
 */
 
BOOL blk_printf (PSX_BLK * blk,char * fmt,...)
{
 char * p;
 va_list l;
 int r;

 if (blk == NULL || fmt == NULL)
  return (FALSE);

 if ((p = (char *) malloc (BLK_PRINTF_MAX * sizeof (char))) == NULL)
  return (FALSE);

 va_start (l,fmt);
 r = vsnprintf (p,BLK_PRINTF_MAX,fmt,l);
 va_end (l);

 if (r == -1)
 {
  free (p);
  return (FALSE);
 }

 if (!blk_put_s (blk,p))
 {
  free (p);
  return (FALSE);
 }

 free (p);

 return (TRUE);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


