/******************************************************************************/
/* cci-lex.c                                  Compiler Construction Interface */
/******************************************************************************/
/** @file cci-lex.c Compiler Construction Interface - Source Code File
 * Functions implementing an interface to a lex generated scanner.
 * It features the scan of a file checking on syntactical errors.
 */
 
#include "psx.h"

#define FAIL(s) { printf ("%s\n",s); exit (-1); }
#define yylval (*lex -> yylval)

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** call lex scanner procedure for token t
 * @param lex pointer to lex structure
 * @param cci_tok compiler token
 * @param t token
 */

static int lex_put (CCI_LEX * lex,int cci_tok,CCI_TOKEN * t)
{
 if (lex == 0 || lex -> scnprc == NULL)
  return (0);

 lex -> scnprc (cci_tok,t);   // scan t

 return (t -> id);
}

/******************************************************************************/
/** scan input comment block
 * @param lex pointer to lex structure
 */

static int lex_cmt_blk (CCI_LEX * lex)
{
 register int c,sts,level;

 sts = 0;
 level = 1;	                      // we came here by a"/*"

 while ((c = lex -> input ()) != EOF)
 {
  if (c == '\n')
  {
   lex_nl (lex,0);                /* return found                  */
   continue;
  }

  switch (sts)
  {
   case 0:                         /* scanning for '*' '/' 	      */
   {
    switch (c)
    {
     case '*':
      sts = 1;
      break;

     case '/':
      sts = 2;
      break;
    }
    break;
   }

   case 1:                       /* got '/' '*' ... '*', awaiting '/'  */
   {
    switch (c)
    {
     case '/':
      level--;
      if (level == 0)
       return (0);
      sts = 0;
      break;

     case '*':                   /* next try			      */
      break;

     default:
      sts = 0;
      break;
    }
    break;
   }

   case 2:                      /* got '/', looking for '*'	      */
   {
    switch (c)
    {
     case '*':
      level++;
      sts = 0;
      break;

     case '/':                  /* next try			      */
      break;

     default:
      sts = 0;
      break;
    }
    break;
   }

   default:
   {
    FAIL ("illegal scanner state");
    return (-1);
   }
  }
 }

#if 0
  switch (c)
  {
   case '*':
   {
    c = input ();

    switch (c)
    {
     case '/': return 0;
     case EOF: return (TOK_ERROR);
     case '\n': scn_nl (); break;
    }
    break;
   }

   case '/':
   {
    break;
   }

   default:
   {
    break;
   }
  }
 }
#endif

 return (0);
}

/******************************************************************************/
/** skip comment line and call lex_nl()
 * @param lex pointer to lex structure
 */

static int lex_cmt_lin (CCI_LEX * lex)
{
 register int c;

 while ((c = lex -> input ()) != '\n' && c != EOF)
  ;          /* eat up text of comment */

 if (c == '\n')
  lex_nl (lex,0);

 return (0);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** scan new line
 * @param lex pointer to lex structure
 * @param id number of identifier
 */

int lex_nl (CCI_LEX * lex,int id)
{
 memset (&yylval,0,sizeof (yylval));
 lex_put (lex,CCI_TOK_NL,&yylval);  // callback lex

 return (id);
}

/******************************************************************************/
/** scan identifier
 * @param lex pointer to lex structure
 * @param id number of identifier
 */ 

int lex_id (CCI_LEX * lex,int id)
{
 memset (&yylval,0,sizeof (yylval));
 strcpy (yylval.txt,*lex -> yytext);
 yylval.id = id;
 lex_put (lex,CCI_TOK_ID,&yylval);  // callback lex

 return (yylval.id);
}

/******************************************************************************/
/** scan string
 * @param lex pointer to lex structure
 * @param id number of identifier
 */ 

int lex_str (CCI_LEX * lex,int id)
{
 char * txt = *lex -> yytext;
 int l = strlen (txt);

 strcpy (yylval.txt,"");

 if (l < 2)
  return (0);

 if (!(txt [0] == '"' && txt [l - 1] == '"'))
  return (0);

 memcpy (yylval.txt,txt + 1,l - 2);
 yylval.txt [l-2] = 0;

 yylval.id = id;
 return (id);

}

/******************************************************************************/
/** scan decimal
 * @param lex pointer to lex structure
 * @param id number of identifier
 */

int lex_dec (CCI_LEX * lex,int id)
{
 long      l;

 sscanf (*lex -> yytext,"%lu",&l);
 memset (&yylval,0,sizeof (yylval));
 yylval.itg = l;
 yylval.id = id;
 strcpy (yylval.txt,*lex -> yytext);

 return (id);
}

/******************************************************************************/
/** scan integer
 * @param lex pointer to lex structure
 * @param id number of identifier
 */ 

int lex_int (CCI_LEX * lex,int id)
{
 int       i;

 sscanf (*lex -> yytext,"%d",&i);
 memset (&yylval,0,sizeof (yylval));
 yylval.itg = i;
 yylval.id = id;
 strcpy (yylval.txt,*lex -> yytext);

 return (id);
}

/******************************************************************************/
/** scan hex
 * @param lex pointer to lex structure
 * @param id number of identifier
 */

int lex_hex (CCI_LEX * lex,int id)
{
 unsigned  u;

 sscanf ((*lex -> yytext)+2,"%x", &u);
 memset (&yylval,0,sizeof (yylval));
 yylval.itg = u;
 yylval.id = id;
 strcpy (yylval.txt,*lex -> yytext);

 return (id);
}

/******************************************************************************/
/** scan bin
 * @param lex pointer to lex structure
 * @param id number of identifier
 */

int lex_bin (CCI_LEX * lex,int id)
{
 int    i = 0;
 char * yp;

 for (yp = (*lex -> yytext) + 2;*yp;yp++)
 {
  i <<= 1;
  i += *yp - '0';
 }

 memset (&yylval,0,sizeof (yylval));
 yylval.itg = i;
 yylval.id = id;
 strcpy (yylval.txt,*lex -> yytext);

 return (id);
}

/******************************************************************************/
/** scan octal
 * @param lex pointer to lex structure
 * @param id number of identifier
 */

int lex_oct (CCI_LEX * lex,int id)
{
 unsigned  u;

 sscanf ((*lex -> yytext),"%o", &u);
 memset (&yylval,0,sizeof (yylval));
 yylval.itg = u;
 yylval.id = id;

 return (id);
}

/******************************************************************************/
/** scan real
 * @param lex pointer to lex structure
 * @param id number of identifier
 */

int lex_real (CCI_LEX * lex,int id)
{
 float f;

 sscanf ((*lex -> yytext),"%f", &f);
 memset (&yylval,0,sizeof (yylval));
 yylval.flt = f;
 yylval.id = id;

 return (id);
}

/******************************************************************************/
/** scan character
 * @param lex pointer to lex structure
 * @param id number of identifier
 */

int lex_chr (CCI_LEX * lex,int id)
{
 int c,d;

 c = lex -> input (); 
 d = lex -> input ();

 if (d != 39)	    // '
  return (0);

 memset (&yylval,0,sizeof (yylval));
 yylval.chr = c;
 yylval.id = id;

 return (id);
}

/******************************************************************************/
/** scan relational operator
 * @param lex pointer to lex structure
 * @param id number f identifier
 */

int lex_rop (CCI_LEX * lex,int id)
{
 int opr;

 if (strcmp (*lex -> yytext,"==") == 0)
  opr = CCI_ROP_EQ;
 else
  if (strcmp (*lex -> yytext,"!=") == 0)
   opr = CCI_ROP_NE;
  else
   if (strcmp (*lex -> yytext,"<") == 0)
    opr = CCI_ROP_LS;
   else
    if (strcmp (*lex -> yytext,"<=") == 0)
     opr = CCI_ROP_LE;
    else
     if (strcmp (*lex -> yytext,">") == 0)
      opr = CCI_ROP_GR;
     else
      if (strcmp (*lex -> yytext,">=") == 0)
       opr = CCI_ROP_GE;
      else
       return (0);

 yylval.id = id;
 yylval.opr = opr;

 return (id);
}

/******************************************************************************/

#if 0
new:

int scn_comment_c (void)
{
 register int c;

 for ( ; ; )
 {
  while ((c = input ()) != '*' && c != EOF)
   ;						  /* eat up text of comment */

  if (c == '*')
  {
   while ((c = input ()) == '*')
    ;
   if (c == '/')
    break;				/* found the end */
  }

  if (c == EOF)
  {
   // error("EOF in comment" );
   break;
  }
 }

 return (0);
}

#endif

/******************************************************************************/

/******************************************************************************/
/** scan comments
 * @param lex pointer to lex structure
 * @param id umber of identifier
 */

int lex_cmt (CCI_LEX * lex,int id)
{
//printf ("lex: yytext = '%s'\n",*lex -> yytext);

 if (strcmp (*lex -> yytext,"/*") == 0)
  return (lex_cmt_blk (lex));

 if (strcmp (*lex -> yytext,"//") == 0)
  return (lex_cmt_lin (lex));

 return (0);
}

/******************************************************************************/
/** decide lex type
 * @param lex pointer to lex structure
 * @param ltp lex type
 * @param id number of identifier
 */
 
int lex (CCI_LEX * lex,int ltp,int id)
{
 int r;

 switch (ltp)
 {
  case LTP_NL:		r = lex_nl  (lex,id);	break;
  case LTP_ID:		r = lex_id  (lex,id);	break;
  case LTP_STR:		r = lex_str (lex,id);	break;
  case LTP_DEC:		r = lex_dec (lex,id);	break;
  case LTP_INT:		r = lex_int (lex,id);	break;
  case LTP_HEX:		r = lex_hex (lex,id);	break;
  case LTP_BIN:		r = lex_bin (lex,id);	break;
  case LTP_OCT:		r = lex_oct (lex,id);	break;
  case LTP_REAL:	r = lex_real(lex,id);	break;
  case LTP_CHR:		r = lex_chr (lex,id);	break;
  case LTP_ROP:		r = lex_rop (lex,id);	break;
  case LTP_CMT:		r = lex_cmt (lex,id);	break;
  default:
   r = 0;

 }

 return (r);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


