/******************************************************************************/
/* psx-cgi.c                         Common Gateway Interface                 */
/******************************************************************************/
/** @file psx-cgi.c Common Gateway Interface - Source Code File
 * Functions implementing the Common Gateway Interface including functions
 * for conversion of query strings.
 * The CGI manages the construction and processing of the PSX Programme via
 * cgi. By calling a CGI-Script (psx-cgi) the programme can be started in cgi
 * mode.
 */


#include "psx.h"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

CGI cgi =
{
 NULL,0,NULL,FALSE        // par, par_num, stg, cnt
};

static BOOL cgi_active = FALSE;

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** convert hexadecimal number, given as string, to decimal, also string
 * @param s pointer to string
 */

static char cgi_str_x2c (char * s)
{
 register char digit;
  
 digit = (s [0] >= 'A' ? ((s [0] & 0xdf) - 'A')+10 : (s [0] - '0'));
 digit *= 16;
 digit += (s [1] >= 'A' ? ((s [1] & 0xdf) - 'A')+10 : (s [1] - '0'));

 return (digit);
}

/******************************************************************************/
/** unescape "%"-escaped characters from cgi string
 * @param s pointer to string
 */

static void cgi_str_unescape (char * s)
{
 register int x,y,len;
  
 len = strlen (s);
  
 for (x = 0,y = 0;s [y];++x,++y)
 {
  if ((s [x] = s [y]) == '%' && y < len - 2)
  {
   s [x] = cgi_str_x2c (&s [y+1]);
   y+=2;
  }
 }

 s [x] = '\0';
}

/******************************************************************************/
/** convert pluses back to spaces
 * @param s pointer to string
 */

static void cgi_str_plustospace (char * s)
{
 register int x;
  
 for (x = 0;s [x];x++)
  if (s [x] == '+')
   s [x] = ' ';
}

/******************************************************************************/
/** convert cgi string to 'regular' string
 * @param s string pointer to cgi string, contains converted string after
 *          invocation
 */

static BOOL cgi_str_convert (char * s)
{
 if (s == NULL)
  return (FALSE);

 cgi_str_unescape (s);        // unescape escaped characters
 cgi_str_plustospace (s);     // plus back to space

 return (TRUE);
}

/******************************************************************************/
/** get cgi parameter at position i in query string
 * @param q string pointer to query string
 * @param i int pointer to position in query string
 * @param sym pointer to string containing name of parameter after invocation
 * @param val pointer to string containing value of parameter after invocation
 */
 
static BOOL cgi_str_get_par (char * q,int * i,char ** sym,char ** val)
{
 char * t, * p;

 if (q == NULL || i == NULL || sym == NULL || val == NULL)
  return (FALSE);

 if ((t = (char*) strchr (q + *i,'=')) == NULL) // get symbol after '='
  return (FALSE);

 *sym = q + *i;   /* sym points to current label                              */ 
 *t++ = 0;        /* overwrite equality sign with "0" (string delimiter),
                      t points to first character in reststring  after "="    */
 *val = t;        /* val contains reststring                                  */

 if ((p = (char*) strchr (t,'&')) != NULL)  // looking for '&' in reststring
 {
  *p++ = 0;       /* overwrite ampersand with "0" (string delimiter),
                     p points to first character in reststring  after "&"    */
  *i = p - q;
 }
 else
 {
  *i = t - q + strlen (t);    // no '&' found
 }

 return (TRUE);
}

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** get environment variable REQUEST_METHOD
 *  @return integer encoding request method (1="GET", 2="POST")
 */

static int cgi_getRequestMethod ()
{
 char * e;

 if ((e = getenv ("REQUEST_METHOD")) == NULL)
  return (CGI_REQ_ZERO);

 if (strcmp (e,"GET") == 0)
  return (CGI_REQ_GET);

 if (strcmp (e,"POST") == 0)
  return (CGI_REQ_POST);
  
 return (CGI_REQ_ZERO); 
}

/******************************************************************************/
/** get environment variable CONTENT_TYPE
 * @return integer encoding content type (1="application/x-www-form-urlencoded", 0 else)
 */

static int cgi_getContentType ()
{
 char * t;

 if ((t = getenv("CONTENT_TYPE")) == NULL)
  return (CGI_CNT_ZERO);

 if (strcmp (t,"application/x-www-form-urlencoded") == 0)
  return (CGI_CNT_FORM);

 return (CGI_CNT_ZERO);
}

/******************************************************************************/
/** get length of content
 * @param l pointer to length
 */

static BOOL cgi_getContentLength (int * l)
{
 char * t;

 if (l == NULL)

  return (FALSE);

 if ((t = getenv("CONTENT_LENGTH")) == NULL)  // check existence
 {                                            // if no environment variable
  if (l)
   *l = -1;
  return (FALSE);
 }

 if (sscanf (getenv ("CONTENT_LENGTH"),"%d",l) != 1)   // getlength
 {
  if (l)
   *l = -1;
  return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** add parameter to (extern) cgi structure
 * @param sym string pointer to name of parameter
 * @param val string pointer to value
 */

static BOOL cgi_par_add (char * sym,char * val)
{
 CGI_PAR * t;

//printf ("p: '%s' -> '%s'<br>\n",sym,val);

 if (sym == NULL || val == NULL)
  return (FALSE);


 if ((t = (CGI_PAR *) realloc (cgi.par,(cgi.par_num + 1) * sizeof (CGI_PAR))) == NULL)
  return (FALSE);

 cgi.par = t;
 cgi.par [cgi.par_num].sym = sym;
 cgi.par [cgi.par_num].val = val;
 cgi.par_num++;

 return (TRUE);
}

/******************************************************************************/

/** parse cgi query string q and add parameters to (extern) cgi structure
 * @param q string pointer to cgi query string
 */
 
static BOOL cgi_parse (char * q)
{
 int i;
 char * s,* v;
 
 cgi_str_convert (q);      // convert cgi string to 'regular' string

 i = 0;

 // get parameters of query string
 while (cgi_str_get_par (q,&i,&s,&v)) // get parameter name and value at pos i in query string q
 {
  cgi_par_add (s,v);                  // add s and v to cgi struct
  /* Logging for debugging purposes. To use this line, the calls to psx_cgi() and
   * psx_bnr() in psx_init() have to be moved to the end, after initialization of
   * the log file.
   */
  //psx_log_m ("cgi", "retreived parameter '%s' -> '%s'", s, v);
 }
 return (TRUE);
}

/******************************************************************************/
/** parse query string (method get) and add parameters to (extern) cgi structure
 */
 
static BOOL cgi_parse_get ()
{
 char * t;
 
 if ((t = getenv("QUERY_STRING")) == NULL)  
  return (FALSE);
 
 if (!cgi_parse (t))              
  return (FALSE);

 return (TRUE);

}

/******************************************************************************/
/** parse query string (method post) and add parameters to (extern) cgi structure
 */

static BOOL cgi_parse_post ()
{
 int i,n;
 char * t;

 if (cgi_getContentType () != CGI_CNT_FORM)
  return (FALSE);
  
 if (!cgi_getContentLength (&n))
  return (FALSE);

 if ((t = malloc (n + 1)) == NULL)
  return (FALSE);
 
 fgets (t,n + 1,stdin);           // get content from stdin

 cgi.stg = t;                     // strlen(t) <= n+1 (+1 because '\0')
 
 if (!cgi_parse (cgi.stg)) 
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/* Private                                                             Access */
/******************************************************************************/
/** get access time
 * @param tsp time specification string
 * @param dat date string
 * @param tim time string
 */

static BOOL cgi_acc_get_time (PSX_STR tsp,PSX_STR dat,PSX_STR tim)
{
 time_t      t;
 struct tm * u;

 t = time (NULL);
 u = gmtime (&t);

 strftime (tsp,100,"%Y-%m-%d-%H-%M-%S",u);
 strftime (dat,100,"%d. %m. %Y",u);
 strftime (tim,100,"%H:%M:%S",u);

 return (TRUE);
}

/******************************************************************************/
/** get remote address, remote port and host
 * @param adr address string
 * @param hst host string
 * @param prt port string
 */

static BOOL cgi_acc_get_remote (PSX_STR adr,PSX_STR hst,PSX_STR prt)
{
 PSX_STR a,p,h;

 if (adr == NULL || hst == NULL)
  return (FALSE);

 strcpy (adr,"");
 strcpy (hst,"");
 strcpy (prt,"");

 if (!sys_env_get ("REMOTE_ADDR",a))
  return (FALSE);

 if (!sys_env_get ("REMOTE_PORT",p))
  return (FALSE);

 sprintf (adr,"%s",a);

 if (nci_lookup (a,h))
  sprintf (hst,"%s",h);
 else
  strcpy (hst,"?");

 strcpy (prt,p);

 return (TRUE);
}


/******************************************************************************/

/* Public                                                                     */
/******************************************************************************/
/** retrieve and parse cgi information, store in (extern) cgi structure 
 */

BOOL cgi_retrieve ()
{
printf ("Content-type:text/html\n\n");

 switch (cgi_getRequestMethod ())
 {
  case CGI_REQ_GET:
   cgi_parse_get ();
   break;

  case CGI_REQ_POST:
   cgi_parse_post ();
   cgi_parse_get ();
   break;

  default:
   printf ("REQ: ?<br>\n");
   return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/

BOOL cgi_init ()
{
 if (cgi_active)
  return (TRUE);

 cgi.par     = NULL;
 cgi.par_num = 0;
 cgi.stg = NULL;

 cgi_retrieve ();   // get cgi information and parse input variables
 cgi_active = TRUE;

 return (TRUE);
}

/******************************************************************************/

BOOL cgi_exit ()
{
 if (!cgi_active)
  return (FALSE);

 if (cgi.par)
 {
  free (cgi.par);
  cgi.par = NULL;
 }

 if (cgi.stg)
 {
  free (cgi.stg);
  cgi.stg = NULL;
 }

 cgi_active = FALSE;

 return (TRUE);
}

/******************************************************************************/
/** get value of cgi parameter                                
 * @param atr pointer to string containing parameter name (e.g. "sts")                                         
 * @param val pointer to string containing value after invocation (e.g. "[0|1]")
 */
 
BOOL cgi_get (const char * atr,char ** val)
{

 IDX i;

 if (atr == NULL)
  return (FALSE);


 CGI_PAR_ITERATE (i)

 {
  if (strcmp (CGI_SYM (i),atr) == 0)
  {
   if (val)
    *val = CGI_VAL (i);

   return (TRUE);
  }
 }

 return (FALSE);
}

/******************************************************************************/
/** print list to stdout (cgi output)                                     
 * @param fmt constant string pointer to output format (e.g. "%s")                                        
 * @param l list of values to print out (variable argument list)
 */
 
BOOL cgi_put_l (const char * fmt,va_list l)
{
 if (fmt == NULL)
  return (FALSE);

 if (!cgi.cnt)
 {
  //printf ("Content-type:text/html\n\n");
  cgi.cnt = TRUE;
 }

 vfprintf (stdout,fmt,l);

 return (TRUE);
}

/******************************************************************************/
/** do cgi output
 * @param fmt constant pointer to string containing output format
 * @param ... values to be put (variable parameter list)
 */
 
BOOL cgi_put (const char * fmt,...)
{
 va_list l;

 if (fmt == NULL)
  return (FALSE);

 va_start (l,fmt);
 cgi_put_l (fmt,l);
 va_end (l);

 return (TRUE);
}

/******************************************************************************/
/** get access
 * @param acc pointer to access information
 */
 
BOOL cgi_acc_get (CGI_ACC * acc)
{
 if (acc == NULL)
  return (FALSE);

 if (!sys_env_get ("REMOTE_USER",acc -> usr))
  strcpy (acc -> usr,"unknown");

 if (!cgi_acc_get_time (acc -> tsp,acc -> dat,acc -> tim))
  return (FALSE);

 cgi_acc_get_remote (acc -> adr,acc -> hst,acc -> prt);

 return (TRUE);
}

/******************************************************************************/
/** environment information structure
 */
 
static const struct
{
 const char * sym;
 const char * exp;
} cgi_env [] =
{

 { "SSL_CLIENT_S_DN"      ,"SSL_CLI"   },


 { "SSL_CLIENT_S_DN_C"    ,"SSL_CLI_C" },
 { "SSL_CLIENT_S_DN_CN"   ,"SSL_CLI_N" },
 { "SSL_CLIENT_S_DN_Email","SSL_CLI_E" },
 { "SSL_CLIENT_S_DN_L"    ,"SSL_CLI_L" },
 { "SSL_CLIENT_S_DN_O"    ,"SSL_CLI_O" },
 { "SSL_CLIENT_S_DN_OU"   ,"SSL_CLI_U" },
 { "SSL_CLIENT_S_DN_ST"   ,"SSL_CLI_S" }
};

/******************************************************************************/
/** get dictionary
 * @param dic pointer to dictionary
 */
BOOL cgi_dic_get (PSX_DIC * dic)
{
 CGI_ACC acc;
 NUM n;
 IDX i;

 if (dic == NULL)
  return (FALSE);

 if (!cgi_acc_get (&acc))
  return (FALSE);

 if (!dic_put (dic,"USR",acc.usr))	return (FALSE);
 if (!dic_put (dic,"TSP",acc.tsp))	return (FALSE);
 if (!dic_put (dic,"DATE",acc.dat))	return (FALSE);
 if (!dic_put (dic,"TIME",acc.tim))	return (FALSE);
 if (!dic_put (dic,"ADR",acc.adr))	return (FALSE);
 if (!dic_put (dic,"HST",acc.hst))	return (FALSE);
 if (!dic_put (dic,"PRT",acc.prt))	return (FALSE);

 n = sizeof (cgi_env) / sizeof (cgi_env [0]);

 for (i = 0;i < n;i++)
 {
  PSX_STR v;

  if (!sys_env_get (cgi_env [i].sym,v))
   strcpy (v,"?");

  if (!dic_put (dic,cgi_env [i].exp,v))
   return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


