/******************************************************************************/
/* psx-cio.c                                                     Console I/O  */
/******************************************************************************/
/** @file psx-cio.c Console I/O - Source Code File
 * Functions and Structures implementing the Console I/O service,
 * which provides reading from and writing in a file using specified
 * prefixes and suffixes for every string.
 */
 
#include "psx.h"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

static struct
{
 BOOL cnt;
 int  pad;
} cio = { FALSE,0 };

#define CIO_FMT_PFX "%>"   /**< prefix */
#define CIO_FMT_SFX "%<"   /**< suffix */

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** parse format
 * @param fmt string pointer to format
 * @param l pointer to variable parameter list
 * @param f string pointer to format after invocation
 * @param p string pointer to padding information
 * @param cnt pointer to boolean, TRUE if content is valid and closed by "%<"
 */
 
static BOOL cio_fmt_parse (const char * fmt,va_list * l,char * f,char * p,BOOL * cnt)
{
 int    k,n,fn;
 char *t,* f0;

 if (fmt == NULL || f == NULL || p == NULL || cnt == NULL)
  return (FALSE);

 k = strlen (CIO_FMT_PFX);		// '%>'

 if (strncmp (fmt,CIO_FMT_PFX,k) == 0)	// "%>%<"
 {
  t = va_arg (*l,char *);		// nchstes Argument der Argumentenliste

  if (t)
   strcpy (p,t);
  else
  {
   memset (p,' ',cio.pad);
   p [cio.pad] = 0;
  }

  f0 = (char *) fmt + k;		// next character
 }
 else
 {
  strcpy (p,"");
  f0 = (char *) fmt;
 }

 n = strlen (f0);			// char after '%>'
 k = strlen (CIO_FMT_SFX);		// "%<"

 if (n >= k)				// more character than "%>%<"
  if (strncmp (f0 + n - k,CIO_FMT_SFX,k) == 0)	// last character "%<"
  {
   fn = n - k;		// fn: end of string
   *cnt = TRUE;
  }
  else
  {
   fn = n;		// end not closed by "%<" 
   *cnt = FALSE;
  }

 strncpy (f,f0,fn);	// extract content ('%>' Inhalt '%<')
 f [fn] = 0;

 return (TRUE);
}

/******************************************************************************/
/** print idented output, if input string contains newline character ouput will be idented at that position
 * @param f file pointer to output file
 * @param i number of spaces to insert
 * @param s pointer to string to be printed
 */
 
static BOOL cio_print_i (FILE * f,int i,char * s)
{
 char pad [SYS_STR_MAX];
 char t   [SYS_STR_MAX];

 if (s == NULL)
  return (FALSE);

 memset (pad,' ',i); pad [i] = 0; // pad now contains i spaces

 while (*s != '\0')
 {
  char * p = strchr (s,'\n');

  if (p)                          // if newline character found
  {
   int k = p - s;                 // k = numbers of characters before newline

   assert (k < SYS_STR_MAX);

   strncpy (t,s,k); t [k] = 0;    // t now contains characters up to next newline
   fprintf (f,"%s\n%s",t,pad);    // print t to file f and intend next line
   s += k + 1;
  }
  else
  {
   fprintf (f,"%s",s);
   break;
  }
 }

 return (TRUE);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** get value by attribute from file
 * @param atr string pointer to attribute
 * @param val string pointer to value after invocation
 */
 
BOOL cio_get (char * atr,char * val)
{
 char t [1000];
 int k;

 if (atr)
  printf ("%s:> ",atr);

 if (fgets (t,999,stdin) == NULL)
  return (FALSE);

 if ((k = strlen (t)) > 0)
 {
  if (t [k - 1] == '\n' || t [k - 1] == '\r')
   t [k - 1] = 0;
 }

 if (val)
  strcpy (val,t);

 return (TRUE);
}

/******************************************************************************/
/** put a line to file
 * @param fil pointer to destination file
 * @param fmt string pointer to format information
 * @param l variable argument list
 */
 
BOOL cio_put_l (FILE * fil,const char * fmt,va_list l)
{
 BOOL    r,cnt;
 SYS_STR f,p;
 char s [10000];
 int n;

 if (fmt == 0)
  return (FALSE);

 if (!cio_fmt_parse (fmt,&l,f,p,&cnt))
  return (FALSE);

 if (!cio.cnt)		// cio_fmt_parse: kein Stringende mit "%<" gefunden
 {
  if (p [0] != 0)
  {
   fprintf (fil,"%s",p);	// schreibe p in Datei: Argument aus va_list
   n = strlen (p);
  }
  else
   n = cio.cnt ? cio.pad : 0;
 }
 else
  n = cio.pad;

 vsnprintf (s,10000,f,l);	// in Argumenten Liste l

 r = cio_print_i (fil,n,s);	// in logfile schreiben

 if (cnt)
  cio.pad = strlen (p);
 else
  fprintf (fil,"\n");

 cio.cnt = cnt;
 cio.pad = n;		// cio.pad = strlen(p) wird berschrieben !

 return (TRUE);
}

/******************************************************************************/
/** put lines while there are parameters in list
 * @param f pointer to destination file
 * @param fmt string pointer to format
 * @param ... variable parameter list
 */
 
BOOL cio_put (FILE * f,const char * fmt,...)
{
 BOOL r;
 va_list l;

 va_start (l,fmt);
 r = cio_put_l (f,fmt,l);
 va_end (l);

 return (r);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


