/******************************************************************************/
/* psx-cmd-cfg.inl                                          Command Interface */
/******************************************************************************/
/** @file psx-cmd-cfg.inl Command Interface: Configuration Command - Inline Source Code File
 * functions and datatype definitions implementing the processing of the config
 * command.
 * Typical for any PSX Command the processing consists of an initialisation
 * routine, an executing procedure and an exiting routine.
 * The configuration command invoces the output of the configuration.
 */
 
/** configuration context */
typedef struct _CTX_CFG
{
 PSX_SCH * sch;          /**< pointer to schema */
 PSX_REC * rec;          /**< pointer to record */
 MTM       mtm;          /**< matchmode         */
 /** options */
 struct
 {
  BOOL sch;              /**< schema            */
  BOOL mat;              /**< match             */
 } opt;
} CTX_CFG;

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** print schema
 * @param ctx configuration context
 */
 
static BOOL cfg_mat_sch_print (CTX_CFG * ctx)
{
 IDX i;

 if (ctx == NULL)
  return (FALSE);

 if (ctx -> sch == NULL)
  return (FALSE);

 printf ("> Schema\n");

 for (i = 0;i < ctx -> sch -> fsp -> n;i++)
 {
  PSX_STR cps;

  cpi_cps_get (ctx -> sch -> fsp -> itm [i].cps,cps);

  printf ("  > %s (%s)\n",ctx -> sch -> fsp -> itm [i].sym,cps);

 }

 return (TRUE);
}

/******************************************************************************/
/* Private                                                            cfg sch */
/******************************************************************************/
/** initialize configuration process
 * @param ctx pointer to configuration context after invication
 * @param csp pointer to command specification
 */
 
static BOOL cfg_init (CTX_CFG * ctx,PSX_CSP * csp)
{
 if (ctx == NULL || csp == NULL)
  return (FALSE);

 memset (ctx,0,sizeof (*ctx));

 if (!pdi_open ())
  return (FALSE);

 if (!pdi_sch_get (&ctx -> sch))
  return (FALSE);

 ctx -> mtm = csp -> ctx.cfg.mtm;

 ctx -> opt.sch = csp -> ctx.cfg.sch;
 ctx -> opt.mat = csp -> ctx.cfg.mat;

 return (TRUE);
}

/******************************************************************************/
/** exit configuration
 * @param ctx pointer to configuration context structure
 */
 
static BOOL cfg_exit (CTX_CFG * ctx)
{
 BOOL r = TRUE;

 if (ctx == NULL)
  return (FALSE);

 if (!pdi_close ())
  r = FALSE;

 return (r);
}

/******************************************************************************/
/** run configuration
 * @param ctx pointer to configuration context structure
 */
 
static BOOL cfg_run (CTX_CFG * ctx)
{
 if (ctx == NULL)
  return (FALSE);


 if (!ctx -> opt.sch && !ctx -> opt.mat)
 {
  psx_put ("pseudonymization database is active");
  return (TRUE);
 }

 if (ctx -> opt.sch)
 {
  printf ("Configuration/Schema\n\n");
  sch_print (ctx -> sch);
 }

 if (ctx -> opt.mat)
 {
  printf ("Configuration/Matching\n\n");
  cfg_mat_sch_print (ctx);
  mti_print (ctx -> sch,ctx -> mtm);
 }

 return (TRUE);
}


/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** invoke configuration process
 * @param csp pointer to command specification
 */
 
static int cmd_cfg (PSX_CSP * csp)
{
 CTX_CFG ctx;
 int r = 0;

 if (csp == NULL)
  return (-1);

 if (!cfg_init (&ctx,csp))
  return (-1);

 if (!cfg_run (&ctx))
  r = -1;

 if (!cfg_exit (&ctx))
  r = -1;

 return (r);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/



