/******************************************************************************/
/* psx-cmd-ctn.inl                                          Command Interface */
/******************************************************************************/
/** @file psx-cmd-ctn.inl Command Interface: Cryptographic Transformation Command - Source Code File
 * Functions and definitions implementing the processing of the cryptographic
 * transformation command.
 * Structures:
 * - CTX_CTN_ENC
 * - CTX_CTN_BAT
 * - CTX_CTN_CVT
 * The cryptographic transformation command invokes the process of the
 * transformation of data.
 * Among the usual command processing functions like init, run and exit there
 * are additional Procedures for encoding and decoding. The main management
 * function is cmd_ctn_cvt(), it invokes the process.
 */
 
/* Note: cmd_ctn_enc(void) can be found in psx-int.c, line 120                */

/** cryptographic transformation encipher information  */
typedef struct _CTX_CTN_ENC   
{
 PSX_CMX * cmx;               /**< Cipher management context */
} CTX_CTN_ENC;

/**************************************/
/** cryptographic transformation: batch mode */
typedef struct _CTX_CTN_BAT
{
 PSX_CMX * cmx;               /**< Cipher management context */
} CTX_CTN_BAT;

/**************************************/
/** cryptographic transformation: convert */
typedef struct _CTX_CTN_CVT
{
 PSX_FSP * fmt;       /**< format                */
 PSX_REC * rec;       /**< record                */
 FILE *    idf;       /**< input data file       */
 FILE *    odf;       /**< output data file      */
 CPE       sen;       /**< source encoding       */
 CPE       den;       /**< destination encoding  */
 char* del; /**< :BG: delimiter for csv input */
} CTX_CTN_CVT;

/******************************************************************************/
/* Private                                                            ctn enc */
/******************************************************************************/
/** initialize cryptographic transformation encipherment
 * @param ctx contains context after invocation
 * @param csp complete command specification
 */
 
static BOOL ctn_enc_init (CTX_CTN_ENC * ctx,PSX_CSP * csp)
{
 CMI_CFG cmc;

 if (ctx == NULL || csp == NULL)
  return (FALSE);

 memset (ctx,0,sizeof (*ctx));
 memset (&cmc,0,sizeof (cmc));

 cmc.cmd = CMI_CMD_ENC;
 cmc.bat = csp -> ctx.ctn.enc.bat;
 cmc.iff = csp -> ctx.ctn.enc.ifs;
 cmc.off = csp -> ctx.ctn.enc.ofs;
 cmc.enc = csp -> ctx.ctn.enc.enc;
 strcpy(cmc.del, csp -> ctx.ctn.enc.del);                  /*:BG: added delimiter*/
 cmc.idf = str_isnil (csp -> ctx.ctn.enc.idf) ? NULL : csp -> ctx.ctn.enc.idf;
 cmc.odf = str_isnil (csp -> ctx.ctn.enc.odf) ? NULL : csp -> ctx.ctn.enc.odf;
 // cmc.pos = -1;
 cmc.pos = csp -> ctx.ctn.enc.pos;
 cmc.hdn = csp -> ctx.ctn.enc.hdn;
 cmc.ics = csp -> ctx.ctn.enc.ics;  //added 11.11.2004
 strcpy (cmc.key,csp -> ctx.ctn.enc.key);

 if (!cmi_create (&ctx -> cmx))
  return (FALSE);

 if (!cmi_init (ctx -> cmx,&cmc))
 {
  if (ctx -> cmx -> msg)
   psx_put ("%s",ctx -> cmx -> msg);
  return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/** exit encipherment
 * @param ctx pointer to cryptographic transformation encipherment context
 */
 
static BOOL ctn_enc_exit (CTX_CTN_ENC * ctx)
{
 BOOL r;

 if (ctx == NULL)
  return (FALSE);

 r = TRUE;

 if (ctx -> cmx)
 {
  if (!cmi_exit (ctx -> cmx))
  {
   if (ctx -> cmx -> msg)
    psx_put ("%s",ctx -> cmx -> msg);
   r = FALSE;
  }

  cmi_delete (&ctx -> cmx);
 }

 return (r);
}

/******************************************************************************/
/** run encipherment
 * @param ctx pointer to cryptographic transformation encipherment context
 */
 
static BOOL ctn_enc_run (CTX_CTN_ENC * ctx)
{
 if (ctx == NULL)
  return (FALSE);

 if (!cmi_run (ctx -> cmx))                  /* run ciphermanagement */
 {
  if (!str_isnil (ctx -> cmx -> msg))
   psx_put ("%s",ctx -> cmx -> msg);

  return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/* Private                                                            ctn dec */
/******************************************************************************/
/** initialize decryption process
 * @param ctx pointer to cryptographic transformation encipherment context after
 *            invocation
 * @param csp pointer to command specification
 */
 
static BOOL ctn_dec_init (CTX_CTN_ENC * ctx,PSX_CSP * csp)
{
 CMI_CFG cmc;

 if (ctx == NULL || csp == NULL)
  return (FALSE);

 memset (ctx,0,sizeof (*ctx));
 memset (&cmc,0,sizeof (cmc));

 cmc.cmd = CMI_CMD_DEC;

 cmc.iff = csp -> ctx.ctn.dec.ifs;
 cmc.off = csp -> ctx.ctn.dec.ofs;
 cmc.enc = csp -> ctx.ctn.dec.enc;
 cmc.idf = str_isnil (csp -> ctx.ctn.dec.idf) ? NULL : csp -> ctx.ctn.dec.idf;
 cmc.odf = str_isnil (csp -> ctx.ctn.dec.odf) ? NULL : csp -> ctx.ctn.dec.odf;

 cmc.pos = -1;
 // cmc.pos = csp -> ctx.ctn.dec.pos;
 cmc.hdn = csp -> ctx.ctn.dec.hdn;
 cmc.ics = csp -> ctx.ctn.dec.ics;
 
 strcpy (cmc.key,csp -> ctx.ctn.dec.key);

 cmc.bat = TRUE;

 if (!cmi_create (&ctx -> cmx))
  return (FALSE);

 if (!cmi_init (ctx -> cmx,&cmc))
 {
  if (ctx -> cmx -> msg)
   psx_put ("%s",ctx -> cmx -> msg);
  return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/** exit decryption process
 * @param ctx pointer to cryptographic transformation encipherment context
 */
 
static BOOL ctn_dec_exit (CTX_CTN_ENC * ctx)
{
 BOOL r;

 if (ctx == NULL)
  return (FALSE);

 r = TRUE;

 if (ctx -> cmx)
 {
  if (!cmi_exit (ctx -> cmx))
   r = FALSE;

  cmi_delete (&ctx -> cmx);
 }

 return (r);
}

/******************************************************************************/
/** run decryption
 * @param ctx pointer to cryptographic transformation encipherment context
 */
 
static BOOL ctn_dec_run (CTX_CTN_ENC * ctx)
{
 if (ctx == NULL)
  return (FALSE);

 if (!cmi_run (ctx -> cmx))
 {
  if (!str_isnil (ctx -> cmx -> msg))
   psx_put ("%s",ctx -> cmx -> msg);

  return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/* Private                                                            ctn cvt */
/******************************************************************************/
/** initialize  conversion process
 * @param ctx pointer to cryptographic transformation conversion context after
 *            invocation
 * @param csp pointer to command specification
 */
 
static BOOL ctn_cvt_init (CTX_CTN_CVT * ctx,PSX_CSP * csp)
{
 if (ctx == NULL || csp == NULL)
  return (FALSE);

 memset (ctx,0,sizeof (*ctx));

 /* storage */

 if (!fsp_create (&ctx -> fmt))               /* create format specification*/
  return (FALSE);

 if (!rec_create (&ctx -> rec))
  return (FALSE);

 /* format */

 psx_put_v ("format specification: %s",csp -> ctx.ctn.cvt.ifm);

 if (!fsp_load (ctx -> fmt,csp -> ctx.ctn.cvt.ifm))    /* load specification */
 {
  psx_put ("> invalid format specification\n");
  return (FALSE);
 }

 /* files */

 if ((ctx -> idf = fopen (csp -> ctx.ctn.cvt.ifn,"rt")) == NULL)
 {
  psx_put ("file open error [%s]",csp -> ctx.ctn.cvt.ifn);
  return (FALSE);
 } 

 if ((ctx -> odf = fopen (csp -> ctx.ctn.cvt.ofn,"wt")) == NULL)
 {
  psx_put ("file open error [%s]",csp -> ctx.ctn.cvt.ofn);
  return (FALSE);
 } 


 return (TRUE);
}

/******************************************************************************/
/** exit conversion process
 * @param ctx pointer to cryptographic transformation conversion context 
 */
 
static BOOL ctn_cvt_exit (CTX_CTN_CVT * ctx)
{
 if (ctx == NULL)
  return (FALSE);

 if (ctx -> fmt)
  fsp_delete (&ctx -> fmt);

 if (ctx -> rec)
  rec_delete (&ctx -> rec);


 return (TRUE);
}

/******************************************************************************/
/** run conversion process
 * @param ctx pointer to cryptographic transformation conversion context 
 */
 
static BOOL ctn_cvt_run (CTX_CTN_CVT * ctx)
{
 while (rec_read (ctx -> rec,ctx -> fmt,ctx -> idf, NULL,ctx->del))     /*:BG: added 0 for delimiter*/
 {
  rec_print (ctx -> rec);        /* Does nothing but read and print ??? */
 } 

 return (TRUE);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** invoke data encryption
 * @param csp pointer to command specification
 */
 
static int cmd_ctn_enc (PSX_CSP * csp)
{ 
 CTX_CTN_ENC ctx;
 int r = 0;

 if (csp == NULL)
  return (-1);

 if (!ctn_enc_init (&ctx,csp))   /* copy values into ctx */
  return (-1);

 if (!ctn_enc_run (&ctx))        /* run encryption */
  r = -1;

 if (!ctn_enc_exit (&ctx))
  r = -1;

 return (r);
} 

/******************************************************************************/
/** invoke data decryption
 * @param csp pointer to command specification
 */
 
static int cmd_ctn_dec (PSX_CSP * csp)
{
 CTX_CTN_ENC ctx;
 int r = 0;

 if (csp == NULL)
  return (-1);

 if (!ctn_dec_init (&ctx,csp))
  return (-1);

 if (!ctn_dec_run (&ctx))
  r = -1;

 if (!ctn_dec_exit (&ctx))
  r = -1;

 return (r);
}

/******************************************************************************/
/** invoke cryptographic transformation
 * @param csp pointer to command specification
 */
 
static int cmd_ctn_cvt (PSX_CSP * csp)
{
 CTX_CTN_CVT ctx;
 int r;
 
 if (csp == NULL)
  return (1);

 if (!ctn_cvt_init (&ctx,csp))
  return (1); 

 r = 0;

 if (!ctn_cvt_run (&ctx))
  r = -1;

 ctn_cvt_exit (&ctx);
 
 return (r);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/



