/******************************************************************************/
/* psx-cmd-dat.inl                                          Command Interface */
/******************************************************************************/
/** @file psx-cmd-dat.inl Command Interface: Data Generation Command - Source Code File
 * The Data Generation Command invokes the process of the generation of a random
 * data record, able of being used as testdata.
 * Similar to the other command interface processes in PSX this one consists of
 * a main management procedure (cmd_dat_gen), which calls an initialisation
 * procedure, a run procedure and an exiting routine.
 * The main context structure is defined as a CTX_DAT_GEN structure.
 */


/** Data generation context */
typedef struct _CTX_DAT_GEN
{
 PSX_FMT * fsp;             /**< format specification */
 FILE    * out;             /**< output file          */
 NUM       num;             /**< number               */
 PSX_REC * rec;             /**< record               */
 clock_t   clk_start;       /**< starttime            */
 clock_t   clk_end;         /**< endtime              */
 clock_t   clk_run;         /**< runtime              */
} CTX_DAT_GEN;

/**************************************/

/******************************************************************************/
/* Private                                                            ctn gen */
/******************************************************************************/
/**  initiaalize data generation
 * @param ctx pointer to data generation context
 * @param csp command specification
 */
 
static BOOL dat_gen_init (CTX_DAT_GEN * ctx,PSX_CSP * csp)
{
 if (ctx == NULL || csp == NULL)
  return (FALSE);

 if (!fsp_create (&ctx -> fsp))
  return (FALSE);

 psx_put_v ("format specification: %s",csp -> ctx.dat.gen.fsp);

 if (!fsp_load (ctx -> fsp,csp -> ctx.dat.gen.fsp))  /* load format specification */
 {
  psx_put ("> invalid format specification\n");
  return (FALSE);
 }
                                                      /* open output file        */
 if ((ctx -> out = fopen (csp -> ctx.dat.gen.out,"wt")) == NULL)
 {
  psx_put ("> file open error (%s)\n",csp -> ctx.dat.gen.out);
  return (FALSE);
 }

 ctx -> num = csp -> ctx.dat.gen.num;

 if (!rec_create (&ctx -> rec))
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** exit data generation process
 * @param ctx pointer to data generation contex
 */
 
static BOOL dat_gen_exit (CTX_DAT_GEN * ctx)
{
 if (ctx == NULL)
  return (FALSE);

 if (ctx -> fsp)
  fsp_delete (&ctx -> fsp);

 if (ctx -> out)
  fclose (ctx -> out);

 if (ctx -> rec)
  rec_delete (&ctx -> rec);

 return (TRUE);
}

/******************************************************************************/
/** run data generation
 * @param pointer to data generation context
 */
 
static BOOL dat_gen_run (CTX_DAT_GEN * ctx)
{
 IDX i;

 if (ctx == NULL)
  return (FALSE);

 ctx -> clk_start = clock() / CLOCKS_PER_SEC;

 printf ("\n");

 for (i = 0;i < ctx -> num;i++)
 {
  printf ("\r%05d\r",i);

  rec_clear (ctx -> rec);

  if (!rec_gen_rnd (ctx -> rec,ctx -> fsp))     /* generate a random record */
   return (FALSE);

  if (!rec_write (ctx -> rec,ctx -> fsp,ctx -> out))
   return (FALSE);
 }

 printf ("\r%05d\n",i);

 ctx -> clk_end = clock() / CLOCKS_PER_SEC;
 ctx -> clk_run = ctx -> clk_end - ctx -> clk_start;

 psx_put ("rec: %d, time: %d sec\n",i,ctx -> clk_run);

 psx_put ("ready");


 return (TRUE);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** invoke random data generetion
 */
 
static int cmd_dat_gen (PSX_CSP * csp)
{
 CTX_DAT_GEN ctx;
 int r = 0;

 if (csp == NULL)
  return (-1);

 if (!dat_gen_init (&ctx,csp))
  return (-1);

 if (!dat_gen_run (&ctx))
  r = -1;

 if (!dat_gen_exit (&ctx))
  r = -1;

 if (r != 0)
  psx_put ("data generation error");
 
 return (r);
} 

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/




