/******************************************************************************/
/* psx-cmd-dbs.inl                                          Command Interface */
/******************************************************************************/
/** @file psx-cmd-dbs.inl Command Interface: Database Generation Command - Source Code File
 * The Database Generation Command can be used to generate a postgres database.
 * In opposite to the other command processes this does not contain a run
 * function, instead the functionality is splitted in different functions,
 * managed by cmd_dbs_gen(). There are functions for checking the status of
 * the database, checking the consistency and deleting a database.
 * The generation of the database is based on the specified schema and the pdi
 * module.
 * Note: The command does not work with ODBC.
 */
 

/** database generation context */
typedef struct _CTX_DBS_GEN
{
 PSX_FMT * fmt;      /**< format */
 PSX_SCH * sch;      /**< schema */
 BOOL      del;      /**< delete */
} CTX_DBS_GEN;

/**************************************/

/******************************************************************************/
/* Private                                                            ctn gen */
/******************************************************************************/
/** initialize database generation
 * @param ctx pointer to database generation context
 * @param csp pointer to command specification
 */
 
static BOOL dbs_gen_init (CTX_DBS_GEN * ctx,PSX_CSP * csp)
{
 if (ctx == NULL || csp == NULL)
  return (FALSE);

 /* storage */

 if (!fsp_create (&ctx -> fmt))
  return (FALSE);

 if (!sch_create (&ctx -> sch))
  return (FALSE);

 /* format */

 psx_put_v ("schema specification: %s",csp -> ctx.dbs.gen.sch);

 if (!sch_parse (ctx -> sch,csp -> ctx.dbs.gen.sch))
 {
  psx_put ("> invalid schema specification\n");
  return (FALSE);
 }

 ctx -> del = csp -> ctx.dbs.gen.del;

 return (TRUE);
}

/******************************************************************************/
/** exit database generetion process
 * @param ctx pointer to database generation context
 */
 
static BOOL dbs_gen_exit (CTX_DBS_GEN * ctx)
{
 if (ctx == NULL)
  return (FALSE);

 if (ctx -> fmt)
  fsp_delete (&ctx -> fmt);

 if (ctx -> sch)
  sch_delete (&ctx -> sch);

 return (TRUE);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** generate database and delete old database, if specified
 * @param ctx pointer to database generation context
 */
 
static int cmd_dbs_gen (PSX_CSP * csp)
{
 CTX_DBS_GEN ctx;
 int r = 0;
 PSX_STR dbs;

 if (csp == NULL)
  return (-1);

 if (!dbs_gen_init (&ctx,csp))
  return (-1);

 if (!psx_cfg_get (PSX_CFG_DBS,dbs))             /* get configuration */
  return (FALSE);

 if (dbi_test (dbs) && csp -> ctx.dbs.gen.del)   
 {
  psx_put ("database delete");

  if (!dbi_delete (dbs))                         /* delete database */
   return (FALSE);
 }

 psx_put ("database create");

 if (!pdi_create ())                             /* create database */
  return (-1);

 psx_put ("database register");

 if (!pdi_register (ctx.sch))                    /* register schema */
 {
  psx_put ("schema registration error");
  r = -1;
 }

 if (!dbs_gen_exit (&ctx))                       /* exit database generation */
  r = -1;

 if (r == 0)
  psx_put ("database ready");
 else
  psx_put ("database error");
 
 return (r);
} 

/******************************************************************************/
/** check database consistency
 */
 
static BOOL cmd_dbs_sts_consistency (void)
{
 NUM cnt;
 PIX pix;
 IDX i;
 BOOL r;

 r = TRUE;

 if (!pdi_pix_get (&pix))
 {
  psx_put ("pix retrieval failed");
  return (FALSE);
 }

 for (i = 1;i < pix;i++)
 {
  char * pid;

  if ((pid = PIDgen (i)) == NULL)
  {
   psx_put ("pid gen failed");
   return (FALSE);
  }

  if (!pdi_lookup (NULL,pid))
  {
   psx_put ("missing record for pid '%s [%d]'",pid,i);
   r = FALSE;
  }

  if (r)
   psx_put ("record integrity test succeeded");
  else
   psx_put ("record integrity test failed");

 }

 return (TRUE);
}

/******************************************************************************/
/** check database status
 * @param csp pointer to command specification structure
 */
 
static int cmd_dbs_sts (PSX_CSP * csp)
{
 PDI_STS sts;
 int r;

 if (!pdi_open ())
 {
  psx_put ("database open error");
  return (-1);
 }

 if (!pdi_sts (&sts))
 {
  psx_put ("status retrieval error");
  return (-1);
 }

 psx_put ("Database Status\n");

 psx_put ("record count: %u",sts.num_rec);


 cmd_dbs_sts_consistency ();

 if (!pdi_close ())
  r = -1;

 return (r);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/




