/******************************************************************************/
/* psx-cmd-pid.inl                                          Command Interface */
/******************************************************************************/

/******************************************************************************/
/** @file psx-cmd-pid.inl Command Interface - command PID inline code
 * Functions supporting the command execution process.
 * Commands assigned to this file are the PID request command, PID cgi command
 * and the PID check command.
 *
 * PID request - this command invokes the retrieval or the generation of a PID.
 * It consists of an initialisation function, where its context is set, a run
 * function, that invokes the further processing, an exit routine to free allocated
 * memory and a function for asserting the correct handling of the command.
 *
 * PID cgi - though assigned to this file cgi is more related to the gui command
 * because of the functionality of graphical user interface using a cgi script
 * for execution of PSX.
 *
 * PID chk - Every PID can be checked on its validity by using the chk command.
 * the hierarchy is quite similar to the PID request command, though it is not
 * that complex in its execution. The control function pid_chk invokes the checking
 * process by calling the initialisation, run and exit function. By making a difference
 * between single and batch mode the run function calls either the single mode run
 * function or the batch mode run function, which handles batch mode by calling
 * the single run function for every PID in batch input file.
 */
 
/** PID request context  */
typedef struct _CTX_PID_REQ
{
 PSX_PMX * pmx;   /**< PMI_CTX context    */
} CTX_PID_REQ;

/** PID over CGI context */
typedef struct _CTX_PID_CGI
{
 PSX_REC * rec;   /**< pointer to record */
} CTX_PID_CGI;

/** PID check context     */
typedef struct _CTX_PID_CHK
{
 PSX_STR inp;      /**< input PID               */
 PSX_STR out;      /**< output PID              */
 FILE    * idf;    /**< input datafile pointer  */
 FILE    * odf;    /**< output datafile pointer */
 BOOL      bat;    /**< batchmode               */
 NUM       pos;    /**< start position in file  */
 PSX_STR   msg;    /**< message                 */
} CTX_PID_CHK;

/******************************************************************************/
/*                                                                            */
/******************************************************************************/
/** store message for programmer in file
 * @param msg string pointer to message text
 */
 
static BOOL pid_notify (char * msg)
{
 FILE * f;

 // enter your own file here:
 if ((f = fopen ("/home/moormann/tmp/psx-not.txt","wt")) == NULL) 
  return (FALSE);

 fprintf (f,"%s",msg);
 fclose (f);

 // notify programmer, enter your own path here
 system ("/home/moormann/scr/psx-notify");

 return (TRUE);
}

/******************************************************************************/
/* Private                                                                ctn */


/******************************************************************************/
/* Private                                                          pid req/s */
/******************************************************************************/

/******************************************************************************/
/* Private                                                          pid req/m */
/******************************************************************************/

/******************************************************************************/
/* Private                                                            pid req */
/******************************************************************************/
/** Initialize PID request.
 * Takes the command specification structure csp as input and
 * returns PID request structure in parameter ctx.
 * @param ctx pointer to PID request structure
 * @param csp Command specification.
 */
 
static BOOL pid_req_init (CTX_PID_REQ * ctx,PSX_CSP * csp)
{
 PMI_CFG pmc;                           // pid management configuration

 if (ctx == NULL || csp == NULL)
  return (FALSE);

 memset (ctx,0,sizeof (*ctx));
 memset (&pmc,0,sizeof (pmc));

 pmc.cmd = PMI_CMD_REQ;
 pmc.idf = csp -> ctx.pid.req.dat;      // input data file
 pmc.iff = csp -> ctx.pid.req.fsp;      // format specification
 pmc.bat = csp -> ctx.pid.req.bat;      // batch mode
 pmc.pos = csp -> ctx.pid.req.pos;      // position
 pmc.otf = csp -> ctx.pid.req.otf;      // output trace file
 pmc.sur = csp -> ctx.pid.req.sur;      // sureness
 pmc.frc = csp -> ctx.pid.req.frc;      // force PID generation
 pmc.ics = csp -> ctx.pid.req.ics;      // input character set
 strcpy(pmc.del,csp-> ctx.pid.req.del); // record input delimiter           /*:BG:added 24.02.05*/
 
 if (!pmi_create (&ctx -> pmx))
  return (FALSE);

 if (!pmi_init (ctx -> pmx,&pmc))
 {
  if (ctx -> pmx -> msg)
   psx_put ("%s",ctx -> pmx -> msg);
  return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/** exit request
 * @param ctx pointer to PID request structure
 */
 
static BOOL pid_req_exit (CTX_PID_REQ * ctx)
{
 BOOL r = TRUE;

 if (ctx == NULL)
  return (FALSE);

 if (ctx -> pmx)
 {
  if (!pmi_exit (ctx -> pmx))
  {
   if (ctx -> pmx -> msg)
    psx_put ("%s",ctx -> pmx -> msg);
   r = FALSE;
  }

  pmi_delete (&ctx -> pmx);
 }

 return (r);
}

/******************************************************************************/
/* Private                                                            pid get */
/******************************************************************************/
/** Run request.
 * Delegates request to pmi_run().
 * @param ctx pointer to PID request structure
 */
 
static BOOL pid_req_run (CTX_PID_REQ * ctx)
{
 if (ctx == NULL)
  return (FALSE);

 if (!pmi_run (ctx -> pmx))
 {
  if (ctx -> pmx -> msg)
   psx_put ("%s",ctx -> pmx -> msg);      // message output
  return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/* Private                                                            pid bat */
/******************************************************************************/



/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** invoke request processing
 * @param csp pointer to command specification
 */
 
static int cmd_pid_req (PSX_CSP * csp)
{ 
 CTX_PID_REQ ctx;
 int r = 0;

 if (csp == NULL)
  return (-1);

 if (!pid_req_init (&ctx,csp))       /* init request  */
 {
  psx_put ("pid init failed");
  r = -1;
 }

 if (r == 0)
  if (!pid_req_run (&ctx))            /* run request  */
   r = -1;

 if (!pid_req_exit (&ctx))            /* exit request */
  r = -1;

 return (r);
}

/******************************************************************************/
//added 23.11.2004
/** correct PID for output
 * @param ctx pointer to PID check context
 * @param res result of PIDcheck function, specifies which correction method
 *            is chosen
 * @param a character which replaces the nth (specified by res) character of
 *          the PID using one correction method
 */

BOOL pid_correct(CTX_PID_CHK * ctx, int res, PSX_STR a)
{  
  PSX_STR temp ; // temporary PID
  PSX_STR temp2;
  PSX_STR m;

  if (ctx == NULL || a == NULL)
    return (FALSE);

  strcpy(temp, ctx->inp);

  
  if ( (1 <= res) && (res <= 8))
  {   
    temp[res-1] = a[0];
    strcpy(ctx->out, temp);
    
  } else if ( 9 <= res && res <= 15)
  {
    temp2[0] = temp[res-9];
    temp[res-9] = temp[res-8];
    temp[res-8] = temp2[0];
    strcpy(ctx->out, temp);
  }

  sprintf (m,"COR: %s\n", ctx->out);
  strcat (ctx->msg,m);

  return (TRUE);
  
}

/******************************************************************************/
//added 23.11.2004
/** initialize PID check context structure
 * @param ctx pointer to PID check context
 * @param csp command specification
 */
 
static BOOL pid_chk_init (CTX_PID_CHK * ctx,PSX_CSP * csp)
{
 
 if (ctx == NULL || csp == NULL)
  return (FALSE);

 memset (ctx,0,sizeof (*ctx));
 
 //pmc.cmd = PMI_CMD_REQ;

 if( ctx->bat = csp -> ctx.pid.chk.bat)      // batch mode
 {
  if((ctx->idf = fopen(csp -> ctx.pid.chk.idf, "r")) == NULL)      // input data file
    return(FALSE);
 }
 else
 {
  ctx->idf = NULL;
 }
   
 
 ctx->pos = csp -> ctx.pid.chk.pos;      // position

 if(!(strcmp(csp->ctx.pid.chk.odf,"")== 0))
 {
  if((ctx->odf = fopen(csp -> ctx.pid.chk.odf, "w")) == NULL)      // output trace file
    return(FALSE);
 }
 else
  ctx->odf = stdout;
       
 strcpy(ctx->inp, csp -> ctx.pid.chk.inp);      // input PID
 strcpy(ctx->out, "");
 strcpy(ctx->msg, "");
 
 return (TRUE);
}

/******************************************************************************/
//added 23.11.2004
/** exit PID check context by closing all files
 * @param ctx pointer to PID check context
 */
 
static BOOL pid_chk_exit (CTX_PID_CHK * ctx)
{
 if (ctx == NULL)
  return (FALSE);

 if(ctx->idf)
  fclose(ctx->idf);

 if(ctx->odf)
  fclose(ctx->odf);
  
    
 return (TRUE);
}


/******************************************************************************/
//added 23.11.2004
/** invoke PID check procedure for one single PID
 * @param ctx pointer to PID check context
 */
 
static BOOL pid_chk_run_s (CTX_PID_CHK * ctx)
{
 int res;
 PSX_STR a;       // output of PIDcheck function
 PSX_STR tmp;    // temporary PID
 PSX_STR m;

 if (ctx == NULL)
  return (FALSE);

 res = PIDcheck(ctx->inp, a);

 switch (res)
 {
   case -1:                         // PID is valid
   {
    strcpy(ctx->out,ctx->inp);
    sprintf(m, "VAL: %s\n", ctx->out);
    strcat(ctx->msg, m);
    break;
   }

   case 0:
   {                                // PID is invalid and cannot be corrected
    strcpy(ctx-> out,"");
    strcat(ctx->msg, "INV\n");
    break;
   }

   case 1: case 2: case 3: case 4: case 5: case 6: case 7:case 8: case 9: case 10: case 11: case 12: case 13: case 14: case 15:
   {
    if (!pid_correct(ctx, res, a) )  // PID is invalid and can be corrected
      return (FALSE);
    break;
   }

   default:
   {                                
    return (FALSE);
    break;
   }
 }
   fprintf(ctx->odf, ctx->msg);     // result output (stdout or file)
   
 return (TRUE);
}

/******************************************************************************/
//added 23.11.2004
/** invoke PID check procedure for batch mode
 * @param ctx pointer to PID check context
 */
static BOOL pid_chk_run_m (CTX_PID_CHK * ctx)
{
 PSX_STR t;
 NUM n;

 if (ctx == NULL)
  return (FALSE);
  
 //? assert (ctx -> idf != NULL && ctx -> ids != NULL);

 n = 0;  // number of checks
  
 if (ctx -> pos != 0)     // skip pos lines
 {
  IDX i;

  psx_put ("skipping %d records",ctx -> pos - 1);

  for (i = 1;i < ctx -> pos;i++)
  {
   if (! fgets (t, 1000, ctx->idf))   // skip line
   {
    psx_put ("error while skipping");
    return (FALSE);
   }
  }
 }

 
 while((fgets(t, 1000, ctx->idf)) != NULL)
 {   
   str_trim(t);
   strcpy(ctx->inp, t);

   sprintf (ctx->msg,"%04d",++n);     // n = record number in dataset
   strcat (ctx->msg,": ");
    
   if(!pid_chk_run_s(ctx))           // PID check for each single PID
   {
     psx_put("PID check failed\n");
     return(FALSE);
   }
 }

 psx_put("ready\n");
  
 return(TRUE);
}
/******************************************************************************/
//added 23.11.2004
/** the decision, whether to use single or batch mode PID check is being made here
 * @param ctx pointer to PID check context
 */
 
static BOOL pid_chk_run (CTX_PID_CHK * ctx)
{
 
 if (ctx == NULL)
  return (FALSE);

 if(ctx->bat)
 {
  if(!pid_chk_run_m(ctx))
    return(FALSE);
 }
 else
 {
   if(!pid_chk_run_s(ctx))
    return(FALSE);
 }

 return (TRUE);
}


/******************************************************************************/
//added 23.11.2004
/** check PID for correctness and correct it, if possible
 * @param csp pointer to command specification
 */

static int cmd_pid_chk (PSX_CSP * csp)
{ 
 CTX_PID_CHK ctx;
 int r = 0;

 if (csp == NULL)
  return (-1);

 if (!pid_chk_init (&ctx,csp))          // initialize PID check context
 {
  psx_put ("pid check init failed\n");
  return(-1);
 }
  
 if (!pid_chk_run (&ctx))               // run check
   return(-1);
    
 if (!pid_chk_exit (&ctx))              // close files
  r = -1;

 return (r);
}  


/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


