/******************************************************************************/
/* psx-cmd.c                                                Command Interface */
/******************************************************************************/
/** @file psx-cmd.c Command Interface  - Source Code File
 * Functions supporting the systematic programme flow respective execution of
 * functionalities featured by the PSX Pseudonymisation Service. In this file
 * functions for looking up and executing commands can be found.
 * The method of executing a command starts with getting the command, retrieving
 * the referring function and then invoking the command execution process.
 */
 
#include "psx.h"

#define MOD PSX_MOD_CMD
#include "psx-cmd-ctn.inl"
#include "psx-cmd-pid.inl"
#include "psx-cmd-gui.inl"
#include "psx-cmd-dat.inl"
#include "psx-cmd-dbs.inl"
#include "psx-cmd-cfg.inl"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** array of pairs of commands(represented as BYTEs) and addresses of
 * respective procedures
 */
static struct     
{
 CMD     cmd; /**< BYTE representation of command */
 CMD_PRC prc; /**< procedure associated with command */
} cmd_idx [] =
{
 { CMD_CTN_ENC,&cmd_ctn_enc },   /* encode   */
 { CMD_CTN_DEC,&cmd_ctn_dec },   /* decipher */
 { CMD_CTN_CVT,&cmd_ctn_cvt },   /* convert  */

 { CMD_PID_REQ,&cmd_pid_req },   // pid request

 { CMD_PID_CHK,&cmd_pid_chk },   // pid check

 { CMD_DAT_GEN,&cmd_dat_gen },   // generate new pid

 { CMD_DBS_GEN,&cmd_dbs_gen },   // generate database
 { CMD_DBS_STS,&cmd_dbs_sts },   // database status

 { CMD_GUI_CGI,&cmd_gui_cgi },   // get gui options

 { CMD_CFG    ,&cmd_cfg }        // command config
};

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** get help            <<< under construction >>>
 */
 
static int cmd_hlp ()
{
 return (0);
}

/******************************************************************************/
/** get procedure assigned to command cmd and store in prc
 * @param cmd command name
 * @param prc pointer to procedure
 */
 
static BOOL cmd_lookup (CMD cmd,CMD_PRC * prc)
{
 IDX i;

 for (i = 0;i < sizeof (cmd_idx) / sizeof (cmd_idx [0]);i++)
  if (cmd_idx [i].cmd == cmd)
  {
   if (prc)
    *prc = cmd_idx [i].prc;

   return (TRUE);
  }

 return (FALSE);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** reset command specification
 * @param csp pointer to command specification
 */

BOOL cmd_csp_reset (PSX_CSP * csp)
{
 if (csp == NULL)
  return (FALSE);

 memset (csp,0,sizeof (*csp));

 return (TRUE);
}

/******************************************************************************/
/** Execute command procedure.
 * The command is given by the field csp->cmd, which takes as value one of
 * the byte constants given in psx-cmd.h. The corresponding command function
 * is determined by cmd_lookup() and executed. Command execution is delegated
 * to the following functions:
 * <ul>
 *   <li>cmd_ctn_enc(): encode
 *   <li>cmd_ctn_dec(): decipher
 *   <li>cmd_ctn_cvt(): convert
 *   <li>cmd_pid_req(): pid request
 *   <li>cmd_pid_chk(): pid check
 *   <li>cmd_dat_gen(): generate new pid
 *   <li>cmd_dbs_gen(): generate database
 *   <li>cmd_dbs_sts(): database status
 *   <li>cmd_gui_cgi(): get gui options
 *   <li>cmd_cfg(): command config
 * </ul>
 * @param csp pointer to command specification
 */
 
int cmd_execute (PSX_CSP * csp)
{
 CMD_PRC prc;
 int r = -1;

 if (csp == NULL)
  return (-1);

 if (!cmd_lookup (csp -> cmd,&prc))   /* lookup command procedure */
  return (-1);

 assert (prc != NULL);

 r = prc (csp);                       /* execute command procedure */

 return (r);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/

