/******************************************************************************/
/* psx-cmd.h                                                Command Interface */
/******************************************************************************/
/**@file psx-cmd.h Command Interface - Header File
 * Definitions of commands supporting the systematic programme flow for every
 * feature of the PSX Pseudonymisation Service. 
 * For each functionality specified by adding options and parameters to the
 * executable file ther is a command in defined in this module.
 * A command consists of
 * - an identification, which is given by a byte representation,
 * - a procedure for retrieving and parsing the respective input parameters,
 * - an own inline file.
 * For each command there is an own entry in the command specification (_PSX_CSP)
 * structure defined in this file and an own context structure.
 */


#ifndef _psx_cmd_
#define _psx_cmd_

/******************************************************************************/
/* Interface                                                                  */
/******************************************************************************/

#include "psx.h"

/******************************************************************************/
/* Definition                                                                 */
/******************************************************************************/

/**************************************/
/* Attributes                         */
/**************************************/
/** attribute definitions             */
#define	CSP_ZERO	0x00
#define	CSP_CMD		0x01    /**< command         */
#define	CSP_MOD		0x02    /**< mode            */
#define	CSP_CTL		0x03    /**< control         */

/**************************************/
/* Command                            */
/**************************************/
/** command representation by BYTEs   */
typedef BYTE CMD;

#define	CMD_ZERO	0x00

#define	CMD_CTN_ENC	0x11   /**< cryptographic transformation: encipher     */
#define	CMD_CTN_DEC	0x12   /**< cryptographic transformation: decipher     */
#define	CMD_CTN_CVT	0x13   /**< cryptographic transformation: convert      */

#define	CMD_PID_REQ	0x21   /**< PID request                                */
#define	CMD_PID_CGI	0x23   /**< PID request over CGI                       */

#define CMD_PID_CHK 0x24   /**< check PId for correctness                  */

#define	CMD_DAT_GEN	0x31   /**< generate random data                       */

#define	CMD_DBS_GEN	0x41   /**< generate database                          */
#define	CMD_DBS_STS	0x42   /**< get status of database                     */

#define	CMD_GUI_CGI	0x51   /**< invoke graphical user interface over cgi   */

#define	CMD_CFG		0x61     /**< configuration                              */

#define	CMD_HLP		0xF1     /**< show help                                  */

#define	CMD_TEST	0xFF     

/**************************************/
/* Mode                               */
/**************************************/
/** mode definitions                  */
#define	MOD_ZERO	0x00
#define	MOD_WEB		0x01  /**< over web   */
#define	MOD_BAT		0x02  /**< batch mode */



/******************************************************************************/
/* Structure                                                                  */
/******************************************************************************/

/**************************************/
/* Command                            */
/**************************************/
/** PSX Command specification         */
typedef struct _PSX_CSP
{
 BYTE cmd;
 /** Context */
 union
 {
  /****************/
  /** Control numbers                                                     */
  union
  {
   struct                /* ENCIPHERMENT */
   {
    PSX_STR ifs;         /**< input format specification                  */
    PSX_STR ofs;         /**< output format specification                 */
    PSX_STR idf;         /**< input data file                             */
    PSX_STR odf;         /**< output data file                            */
    NUM     pos;         /**< start position                              */
    CPE     enc;         /**< encoding                                    */
    BOOL    bat;         /**< batch mode                                  */
    BOOL    hdn;         /**< print heading                               */
    PSX_STR key;         /**< AES key                                     */
    ICS     ics;         /**< input character set[din 66003| iso 8859-1]  */ //added 25.10.2004
    PSX_STR del; /**< :BG: delimiter for csv input */
   } enc;

   /** Decipherment                                                       */
   struct                
   {
    PSX_STR ifs;         /**< input format specification                  */
    PSX_STR ofs;         /**< output format specification                 */
    PSX_STR idf;         /**< input data file                             */
    PSX_STR odf;         /**< output data file                            */
    CPE     enc;         /**< encoding                                    */
    BOOL    bat;         /**< batch mode                                  */
    BOOL    hdn;         /**< print heading                               */
    PSX_STR key;         /**< AES key                                     */
    ICS     ics;         /**< input character set[din 66003| iso 8859-1]  */ //added 25.10.2004
    PSX_STR del; /**< :BG: delimiter for csv input */
   } dec;

   /** Conversion                                                         */
   struct                
   {
    PSX_STR ifm;         /**< input format                                */
    PSX_STR ofm;         /**< output format                               */
    PSX_STR ifn;         /**< input file name                             */
    PSX_STR ofn;         /**< output file name                            */
    BYTE    iec;         /**< input                                       */
    BYTE    oec;         /**< output                                      */
    ICS     ics;         /**< input character set[din 66003| iso 8859-1]  */ //added 25.10.2004
    PSX_STR del; /**< :BG: delimiter for csv input */
   } cvt;
  } ctn;

  /****************/
  /** Pseudonymisation Identificator                                      */
  union                  
  {
   /** PID Request                                                        */ 
   struct                
   {
    BOOL    bat;         /**< batch mode                                  */
    PSX_STR fsp;         /**< only for batch                              */
    PSX_STR dat;         /**< data                                        */
    PSX_STR otf;         /**< output trace file                           */
    NUM     pos;         /**< start position                              */
    int     sur;         /**< sureness: 0,1,*                             */
    BOOL    frc;         /**< force                                       */
    ICS     ics;         /**< input character set[din 66003| iso 8859-1]  */
    PSX_STR del; /**< :BG: delimiter for csv input */
   } req;

   /** PID CHECK                                                          */
   struct                
   {
     PSX_STR inp;        /**< input PID                                   */
     PSX_STR idf;        /**< input data file                             */
     PSX_STR odf;        /**< output data file                            */
     BOOL bat;           /**< batch mode                                  */
     NUM pos;            /**< position in file (line)                     */
   } chk;

    /** CGI                                                               */
   struct               
   {
    PSX_STR fmt;         /**< format                                      */
   } cgi;
  } pid;

  /****************/
  /** Data                                                                */
  union
  {
   /** Generation                                                         */
   struct
   {
    PSX_STR fsp;         /**< format specification                        */
    PSX_STR out;         /**< output string                               */
    NUM     num;         /**< number                                      */
   } gen;

  } dat;

  /****************/
  /** Database                                                            */
  union
  {
   /** Generation                                                         */ 
   struct
   {
    PSX_STR fmt;         /**< format                                      */
    PSX_STR sch;         /**< database schema                             */
    BOOL    del;         /**< delete                                      */
   } gen;

   /** Status                                                             */
   struct
   {
    PSX_STR dummy;       /**< dummy                                       */
   } sts;
  } dbs;

  /****************/

  /*union
  {
   struct
   {
    int dummy;
   } sch;

   struct
   {
    //MXS mxs;
    MTM mtm;
   } mat;

   struct
   {
    PSX_STR psx;
    PSX_STR cgi;
    PSX_STR cfg;
    PSX_STR spc;
    BOOL    mak;
   } ins;
  } cfg1;
  */

  /** Configuration                                                       */
  struct
  {
   BOOL sch;            /**< database schema                              */
   BOOL mat;            /**< matching                                     */
   MTM  mtm;            /**< matchmode                                    */
  } cfg;


  /****************/
  /** Graphical User Interface                                            */
  union
  {
   struct
   {
    BOOL sys;           /**< system                                       */
    BOOL dbg;           /**< debugmode                                    */
   } cgi;
  } gui;

  /****************/

 } ctx;
} PSX_CSP;

typedef int (* CMD_PRC) (PSX_CSP * csp);
#define PSX_COMMANDPROCEDURE(prc) int prc (PSX_CSP * csp)

/******************************************************************************/
/* Service                                                                    */
/******************************************************************************/

BOOL	cmd_csp_reset	(PSX_CSP * csp);

int	cmd_execute	(PSX_CSP * csp);

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/

#endif // _psx_cmd_

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/

