/******************************************************************************/
/* psx-cmi.c                                      Cipher Management Interface */
/******************************************************************************/
/** @file psx-cci.c Cipher Management Interface - Source Code File
 * The Cipher Management Interface provides functions for the encipherment of data.
 * If the input data is given in DIN format a conversion to ASCII is made. 
 */
 
#include "psx.h"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** get cipher management interface output format specification
 * @param ctx pointer to cmi context structure
 */

static BOOL cmi_ofs_get (CMI_CTX * ctx)
{
 PSX_FSP * ofm;
 BOOL r = TRUE;

 if (ctx == NULL)
  return (FALSE);

 if (ctx -> ofs == NULL)
  return (FALSE);

 if (!fsp_create (&ofm))			// create output format spec structure
  return (FALSE);

 if (ctx -> pos != -1)
 {
  FMT_ITM itm;

  strcpy (itm.sym,"idx");
  itm.pos = 1;
  itm.len = 10;
  itm.tfm = 0;

  if (!fsp_add (ofm,&itm))			// add itm to format spec
   r = FALSE;
 }

 if (!cpi_fsp_map (ctx -> ifs,ofm,ctx -> enc))	// map fsp
  r = FALSE;						

 if (!r)
 {
  fsp_delete (&ofm);
  return (FALSE);
 }

 fsp_copy (ctx -> ofs,ofm);						// store ofm
 fsp_delete (&ofm);

 return (TRUE);
}

/******************************************************************************/
/** run single record file mode
 * get input record, encipher, write output record
 * @param ctx pointer to cmi context structure
 */

static BOOL cmi_run_s (CMI_CTX * ctx)
{
 // CPI_REC_ITM cnr;
 int i;
 FILE * out;           // pointer to output file

 if (ctx == NULL)
  return (FALSE);

 if (ctx -> ids)
 {
  psx_put_v ("loading record from file"); // %s",csp -> ctx.ctn.gen.dat);

  if (!rec_read_s (ctx -> idr,ctx -> ifs,ctx -> ids)) // load record
  {
   printf ("> record load failure\n");
   return (FALSE);
  }

  if ( TEST (ctx -> ics,ICS_DIN) )  // added 15.11.2004
    rec_cvt_din (ctx -> idr);  // convert record: DIN 66003 to ISO 8859-1   
 } 
 else  // get record from stdin
 {
  if (!rec_def_fmt (ctx -> idr,ctx -> ifs))       // clear idr and add ifs
   return (FALSE);

  psx_put ("data acquisition\n");

  if (!rec_input (ctx -> idr,2))                  // read record from stdin
   return (FALSE);

  printf ("\n");
 }

 psx_put ("result\n");

 out = ctx -> ods != NULL ? ctx -> ods : stdout;		// output data stream


 cpi_set_key (ctx -> key);								// set cpi_key to ctx->key

 // encipher data record
 if (!cpi_gen (ctx -> idr,ctx -> odr,ctx -> ifs,ctx -> enc))	
  return (FALSE);

 // write output data record 
 for (i = 0;i < ctx -> odr -> n;i++)
 {
  if (i > 0)
   fprintf (out,"\n");

  fprintf (out,"%s",ctx -> odr -> itm [i].val);  // write result to "out"
 }

 fprintf (out,"\n");

 return (TRUE);
}

/******************************************************************************/
/** run multiple record batch mode
 * @param ctx pointer to cmi context structure
 */

static BOOL cmi_run_b (CMI_CTX * ctx)
{
 PSX_STR t;

 if (ctx == NULL)
  return (FALSE);

//printf ("cmi_run_b: off: %s\n",ctx -> off);

 tmi_tmg_start (&ctx -> tmg);

 if (ctx -> ids)
  fseek (ctx -> ids,0,SEEK_SET);

 if (ctx -> ods)
  fseek (ctx -> ods,0,SEEK_SET);

 if (!cmi_ofs_get (ctx))
  return (FALSE);

 assert (ctx -> ofs != NULL);

 if (!str_isnil ((char *) ctx -> off))
 {
  if (!fmt_save (ctx -> ofs,ctx -> off))
  {
   sprintf (ctx -> msg,"error writing output format specification [%s]",ctx -> off);
   return (FALSE);
  }
 }

 if (ctx -> hdn)
  rec_write_m (ctx -> ofs,ctx -> ods);

 cpi_set_key (ctx -> key);

 if (ctx -> pos != 0)        //skip records if neccessary
 {
  IDX i;
	
  psx_put ("skipping %d records",ctx -> pos - 1);

  for (i = 1;i < ctx -> pos;i++)
  {
   if (!rec_read (ctx -> idr,ctx -> ifs,ctx -> ids, NULL,""))		// skip records        /*:BG:added 0 for delimiter*/
   {
    psx_put ("error while skipping");
    return (FALSE);
   }

  }
 }
//printf ("cmi_run_b:\n-----------------------------\n");

 while (rec_read (ctx -> idr,ctx -> ifs,ctx -> ids, NULL,ctx->del)) // ctx -> dat))   /*:BG:added ctx->del*/
 {
  rec_clear (ctx -> odr);

  if ( TEST (ctx -> ics, ICS_DIN) ) // added 15.11.2004
    rec_cvt_din (ctx -> idr);  // convert record: DIN 66003 to ISO 8859-1   
 
//rec_print (ctx -> idr);

  if (!cpi_gen (ctx -> idr,ctx -> odr,ctx -> ifs,ctx -> enc))
   return (FALSE);

  if (!rec_write (ctx -> odr,ctx -> ofs,ctx -> ods))
   return (FALSE);

  //rec_write_m (ctx -> ofs,ctx -> odf);

  ctx -> cnt++;
 }

 tmi_tmg_stop (&ctx -> tmg);

 tmi_tmg_fmt (&ctx -> tmg,t);

 printf ("%s sec\n",t);

 return (TRUE);
}

/******************************************************************************/
/** decide input type, run
 * @param ctx pointer to cmi context structure
 */


static BOOL cmi_run_enc (CMI_CTX * ctx)
{
 BOOL r;

 if (ctx == NULL)
  return (FALSE);

 if (!ctx -> bat)
  r = cmi_run_s (ctx);
 else
  r = cmi_run_b (ctx);

 return (r);
}

/******************************************************************************/
/* syn: run single    <<< under construction >>>                              */
/*
static BOOL cmi_run_dec_s (CMI_CTX * ctx)
{
 printf ("not implemented\n");
 exit (0);
 return (TRUE);
}
*/
/******************************************************************************/
/* run batch           <<< under construction >>>                             */
/*
static BOOL cmi_run_dec_b (CMI_CTX * ctx)
{
 printf ("not implemented\n");
 exit (0);
 return (TRUE);
}
*/
/******************************************************************************/
/** run reverse fsp
 * @param ctx pointer to cmi context structure
 */

static BOOL cmi_run_dec (CMI_CTX * ctx)
{
 BOOL r;

 if (ctx == NULL)
  return (FALSE);

 if (ctx -> ifs)
  fsp_reverse (ctx -> ifs);       // reverse input format specification

 if (!ctx -> bat)
  r = cmi_run_s (ctx);            // decide if single or batch mode
 else
  r = cmi_run_b (ctx);

 return (r);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** create new ctx
 * @param ctx pointer to pointer to ctx structure
 */


BOOL cmi_create (CMI_CTX ** ctx)
{
 if (ctx == NULL)
  return (FALSE);

 if ((*ctx = (CMI_CTX *) malloc (sizeof (**ctx))) == NULL)
  return (FALSE);

 memset (*ctx,0,sizeof (*ctx));

 return (TRUE);
}

/******************************************************************************/
/** delete ctx
 * @param ctx pointer to pointer to ctx structure
 */

BOOL cmi_delete (CMI_CTX ** ctx)
{
 if (ctx == NULL)
  return (FALSE);

 if (*ctx == NULL)
  return (FALSE);

 free (*ctx);

 return (TRUE);
}

/******************************************************************************/
/** initialize ctx with config structure
 * @param ctx pointer to cmi context structure
 * @param cfg pointer to cmi configuration structure
 */

BOOL cmi_init (CMI_CTX * ctx,CMI_CFG * cfg)
{
 if (ctx == NULL || cfg == NULL)
  return (FALSE);

 memset (ctx,0,sizeof (*ctx));
 strcpy (ctx -> msg,"");

 ctx -> cmd = cfg -> cmd;

 /* input format specification */

 if (1)
 {
  if (str_isnil (cfg -> iff))
  {
   strcpy (ctx -> msg,"missing input format specification");
   return (FALSE);
  }

  if (!fsp_create (&ctx -> ifs))
   return (FALSE);

  if (!fsp_load (ctx -> ifs,cfg -> iff))
  {
   strcpy (ctx -> msg,"invalid input format specification");
   fsp_delete (&ctx -> ifs);
   return (FALSE);
  }
 }

 /* output format specification */

 if (cfg -> bat)
 {
  ctx -> off = cfg -> off;

  if (!fsp_create (&ctx -> ofs))
   return (FALSE);
 }


 /* input data file */

 if (!str_isnil (cfg -> idf))
 {
  if ((ctx -> ids = fopen (cfg -> idf,"rt")) == NULL)
  {
   fprintf (stderr,"file r open error: '%s'\n",cfg -> idf);
   return (FALSE);
  }
 }

 /* output data file */

 if (!str_isnil (cfg -> odf))
 {
  if ((ctx -> ods = fopen (cfg -> odf,"wt")) == NULL)
  {
   fprintf (stderr,"file open error: '%s'\n",cfg -> odf);
   return (FALSE);
  }
 }

 /* input record */

 if (1)
 {
  if (!rec_create (&ctx -> idr))
   return (FALSE);
 }

 /* output record */

 if (1)
 {
  if (!rec_create (&ctx -> odr))
   return (FALSE);
 }

 /* encoding */

 ctx -> enc = cfg -> enc;
 ctx -> pos = cfg -> pos;
 ctx -> bat = cfg -> bat;
 ctx -> hdn = cfg -> hdn;
 ctx -> ics = cfg -> ics;
 strcpy(ctx->del,cfg->del);
 strcpy (ctx -> key,cfg -> key);

 return (TRUE);
}

/******************************************************************************/
/** exit module, delete fsp and rec, close streams
 * @param ctx pointer to cmi context structure
 */

BOOL cmi_exit (CMI_CTX * ctx)
{
 BOOL r;

 if (ctx == NULL)
  return (FALSE);

 if (ctx == NULL)
  return (FALSE);

 strcpy (ctx -> msg,"");

 r = TRUE;

 if (ctx -> ifs) if (!fsp_delete (&ctx -> ifs)) r = FALSE;
 if (ctx -> ofs) if (!fsp_delete (&ctx -> ofs)) r = FALSE;
 if (ctx -> idr) if (!rec_delete (&ctx -> idr)) r = FALSE;
 if (ctx -> odr) if (!rec_delete (&ctx -> odr)) r = FALSE;
 if (ctx -> ids) if (fclose (ctx -> ids) == EOF) r = FALSE;
 if (ctx -> ods) if (fclose (ctx -> ods) == EOF) r = FALSE;

 return (r);
}

/******************************************************************************/
/** run cmi enc or dec
 * @param ctx pointer to cmi context structure
 */

BOOL cmi_run (CMI_CTX * ctx)
{
 BOOL r;

 if (ctx == NULL)
  return (FALSE);

 switch (ctx -> cmd)


 {
  case CMI_CMD_ENC:
   r = cmi_run_enc (ctx);
   break;

  case CMI_CMD_DEC:
   r = cmi_run_dec (ctx);
   break;

  default:
   return (FALSE);

 }

 return (r);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/



