/******************************************************************************/
/* psx-cpi.c                                                Cipher Interface  */
/******************************************************************************/
/** @file psx-cpi.c Cipher Interface - Source Code File
 * Function supporting the transformation of data into control numbers.
 */
 
#include "psx.h"

#define MOD PSX_MOD_CPI
/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

static const char * cpi_sfx [] =
{
 "C1",
 "C2",
 "C3",
 "PC",
 "PH"
};

static char cpi_key [1000];

#include "psx-cpi-map.inl"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** normalize data (as of now only for datatype day)
 * @param dtp datatype
 * @param val value string
 */

static BOOL cpi_itm_normalize (DTP dtp,PSX_STR val) // modify data here, in psi check only
{
 PSX_STR old;

 if (val == NULL)
  return (FALSE);

 strcpy (old,val);                                  // store old val

 switch (dtp) 
 {
  case DTP_DD:                                      // if date day
  {
   str_fmt_num (val,2);                             // new val
   break;
  }

  default:
   break;
 }

 LOG (MOD,"normalize: '%s' -> '%s'",old,val); // log changes

 return (TRUE);
}

/******************************************************************************/
/** convert cipher encoding to text format
 * @param cpe cipher encoding structure
 * @param string pointer to textfield
 */

static BOOL cpe_to_txf (CPE cpe,BYTE * txf)
{
 if (txf == NULL)
  return (FALSE);

 switch (cpe)

 {
  case CPE_RAW: *txf = TXF_RAW; break;
  case CPE_HEX: *txf = TXF_HEX; break;
  default: return (FALSE);
 }

 return (TRUE);
}



/******************************************************************************/
/** set component structure
 * @param cmp pointer to cpi cipher record component structure
 * @param src pointer to source string
 * @param len size of copied block from src
 * @param cnt component (?)
 */

static BOOL cpi_cmp_assign (CPI_REC_CMP * cmp,const BYTE * src,NUM len,BYTE cnt)
{
 if (cmp == NULL)
  return (FALSE);

 if (src)
  memcpy (cmp -> stg,src,len);    // copy src to cmp->stg
 else
  memset (cmp -> stg,0,len);

 cmp -> stg [len] = 0;

 cmp -> cnt = cnt;

 return (TRUE);
}

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** convert cipher encoding to text format
 * @param enc cipher encoding
 * @param txf pointer to textfield
 */

static BOOL enctotxf (CPE enc,BYTE * txf)
{
 if (enc == CPE_RAW)
  *txf = TXF_RAW;
 else
  if (enc == CPE_HEX)
   *txf = TXF_HEX;
  else
  {
   *txf = 0xFF;
   return (FALSE);
  }

 return (TRUE);
}

#include "psx-cpi-tfm.inl"


/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** copy component with number cmp to sym
 */

BOOL cpi_cmp_enc (CMP cmp,PSX_STR sym)
{
 if (cmp >= CMP_N)
  return (FALSE);

 strcpy (sym,cpi_sfx [cmp]);

 return (TRUE);
}

/******************************************************************************/
/** get right cmp
 * @param cmp pointer to component
 * @param sym symbol string after invocation
 */

BOOL cpi_cmp_dec (CMP * cmp,PSX_STR sym)
{
 IDX c;

 if (cmp == NULL || sym == NULL)
  return (FALSE);

 for (c = 0;c < CMP_N;c++)
  if (strcmp (sym,cpi_sfx [c]) == 0)
  {
   *cmp = c;
   return (TRUE);
  }

 *cmp = CMP_NIL;

 return (FALSE);
}

/******************************************************************************/
/** split up sym and get cps
 * @param cps pointer to component set after invocation
 * @param sym symbol string 
 */

BOOL cpi_cps_set (CMP * cps,PSX_STR sym)
{
 int i;
 PSX_STR t;

 if (cps == NULL || sym == NULL)
  return (FALSE);

 *cps = 0;
 i = 0;

 while (str_tok (sym,&i,":",t))
 {
  CMP c;

  if (!cpi_cmp_dec (&c,t))
   return (FALSE);

  *cps |= CPS_CMP (c);
 }

 return (TRUE);
}

/******************************************************************************/
/** lookup cps
 */

BOOL cpi_cps_get (CMP cps,PSX_STR sym)
{
 IDX c;

 if (sym == NULL)
  return (FALSE);

 strcpy (sym,"");

 for (c = 0;c < CMP_N;c++)
  if (TEST (cps,CPS_CMP (c)))
  {
   PSX_STR t;

   if (!cpi_cmp_enc (c,t))


    return (FALSE);

   if (sym [0] != 0)
    strcat (sym,":");

   strcat (sym,t);
  }

 return (TRUE);
}

/******************************************************************************/
/** map text format
 */

BOOL cpi_cps_map_tfm (CPS * cps,TFM tfm)
{
 if (cps == NULL)
  return (FALSE);

 *cps = CPS_ZERO;

 if (!TEST (tfm,TFM_COL))
  return (TRUE);

 if (TEST (tfm,TFM_DEC))
  *cps |= CPS_CMP (CMP_N1) | CPS_CMP (CMP_N2) | CPS_CMP (CMP_N3);
 
 if (TEST (tfm,TFM_PHO))
 {
  if (TEST (tfm,TFM_PHO_C))
   *cps |= CPS_CMP (CMP_PC);

  if (TEST (tfm,TFM_PHO_H))
   *cps |= CPS_CMP (CMP_PH);
 }

 return (TRUE);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** get length n of enc
 */

BOOL cpi_gap_get_len (CPE enc,NUM * n)
{
 BYTE txf;

 if (n == NULL)
  return (FALSE);

 if (!cpe_to_txf (enc,&txf))	// get txf
  return (FALSE);

 if (!cni_get_len (txf,n))		// get length
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/

static BOOL cpi_itm_gen_msg (CPI_REC_ITM * itm,PSX_STR sym)
{
 SYS_STR s;
 IDX     i;
 BOOL    t;

 if (itm == NULL || sym == NULL)
  return (FALSE);

 strcpy (s,"");

 t = FALSE;

 for (i = 0;i < CMP_N;i++)
  if (TEST (itm -> cnc,BIT (i)))

  {
   PSX_STR p;

   sprintf (p,"%s:%s",cpi_sfx [i],itm -> cmp [i].stg);

   if (t)
    strcat (s,",");

   strcat (s,p);
   t = TRUE;
  }

 strcpy (sym,s);

 return (TRUE);
}

/******************************************************************************/
/** generate cipher item
 * @param fsp pointer to format specification
 * @param idx index of item
 * @param txt item value
 * @param enc BYTE represenation of encoding specification ([1=raw, 2=hex])                                           
 * @param itm output cipher item
 */

BOOL cpi_itm_gen (PSX_FSP * fsp,IDX idx,PSX_STR txt,CPE enc, CPI_REC_ITM * itm)
{
 PSX_STR t;
 TFM tfm = fsp -> itm [idx].tfm;		// transformation code from format specification

 if (itm == NULL)
  return (FALSE);

 LOG (MOD,"generating cipher item [%s]",fsp -> itm [idx].sym);

 //fsp_tfm_enc (tfm,t); // get transformation symbol from format specification and store in t
                      // JM: is this needed at all ??? t gets overwriten!
									
 strcpy (t,txt);	// JM: overwrites transformation symbol from format specification! 
 str_trim (t);

 cpi_itm_normalize (fsp -> itm [idx].dtp,t);	// norm data (datatype day only)

 LOG (MOD,"> input: '%s' -> '%s'",txt,t);

 memset (itm,0,sizeof (*itm));
 strcpy (itm -> sym,fsp -> itm [idx].sym);

 if (tfm == TFM_ZERO)
  return (FALSE);

 if (str_isnil (txt))							// check if item is empty
 {
  cpi_cmp_assign (&itm -> cmp [CMP_N1],"",0,CPI_CNT_TXT);  // assign empty string
  itm -> cnc |= BIT (CMP_N1);

  if (TEST (tfm,TFM_DEC))
  {
   itm -> cnc |= BIT (CMP_N2) | BIT (CMP_N3);
  }

  if (TEST (tfm,TFM_PHO))
  {
   if (TEST (tfm,TFM_PHO_C))
    itm -> cnc |= BIT (CMP_PC);
   if (TEST (tfm,TFM_PHO_H))

    itm -> cnc |= BIT (CMP_PH);
  }

  LOG (MOD,"> empty -> return");
  return (TRUE);
 } // end if (isnil (txt) )

 /*************************************/

 /* if this line is reached, the item is not empty and is processed: */
 itm -> cnc = 0;

 if (TEST (tfm,TFM_COL))
 {
  // assign item to first component (itm -> cmp [CMP_N1])
  cpi_cmp_assign (&itm -> cmp [CMP_N1],t,strlen (t),CPI_CNT_TXT);
  itm -> cnc |= BIT (CMP_N1);
 }
 else
  return (FALSE);

 assert (itm -> cmp [CMP_N1].stg [0] != 0);
 /* test for: generate MD5 control number (TFM_CTN (?), decompose (TFM_DEC) */
 if (TEST (tfm,TFM_CTN) || TEST (tfm,TFM_DEC))
 {
  txt_trim_r (itm -> cmp [CMP_N1].stg);       // remove trailing spaces
  itm -> cmp [CMP_N1].len = strlen (itm -> cmp [CMP_N1].stg);  // adjust length
 }

 /*************************************/
 /* decomposition                     */

 if (TEST (tfm,TFM_DEC))					  // check if item will be decomposed
 {
  char s [1000];
  BYTE dec1,dec;

  dec1 = TEST (tfm,TFM_DEC_N) ? TXT_DEC_N : TXT_DEC_F;

  dec = dec1;

  txt_trim_r (itm -> cmp [CMP_N1].stg);             // remove trailing spaces
  strup_g (itm -> cmp [CMP_N1].stg);				// transform to upper case

  txt_cvt_uml (itm -> cmp [CMP_N1].stg);			// convert umlaute
  /* for every component, set a bit in cnc */
  itm -> cnc = BIT (CMP_N1) + BIT (CMP_N2) + BIT (CMP_N3); // + BIT (CMP_P1);
  itm -> cmp [CMP_N1].len = strlen (itm -> cmp [CMP_N1].stg);

  if (!cni_itm_prc_dec (itm,dec))					// decompose item

   return (FALSE);

 }

 /*************************************/
 /* phonetics                         */

 if (TEST (tfm,TFM_PHO))
 {
  /* check for Cologne phonetics */
  if (TEST (tfm,TFM_PHO_C))
  {
   if (!cni_itm_prc_pho (itm,PHO_COL)) // generate Cologne phonetics
    return (FALSE);

   itm -> cnc |= BIT (CMP_PC);
  }

  /* check for Hannover phonetics */
  if (TEST (tfm,TFM_PHO_H))
  {
   if (!cni_itm_prc_pho (itm,PHO_HAN)) // generate Hannover phonetics
    return (FALSE);

   itm -> cnc |= BIT (CMP_PH);
  }
 }

 /*************************************/
 /* control number (MD5 hash)         */

 if (TEST (tfm,TFM_CTN))
  if (!cni_itm_prc_ctn (itm))
   return (FALSE);

 /*************************************/
 /* encryption                        */

 if (TEST (tfm,TFM_ENC))
  if (!cni_itm_prc_enc (itm,tfm,enc))
   return (FALSE);

 /*************************************/
 /* encoding (raw or hex)             */
 /* This does not change anything in  */
 /* case of encrypted fields, which   */
 /* are converted in cni_itm_prc_enc()*/
 /*************************************/

 if (!cni_itm_prc_cvt (itm,enc))
  return (FALSE);

 cpi_itm_gen_msg (itm,t);
 LOG (MOD,"cipher item: %s",t);

 return (TRUE);
}

/******************************************************************************/
/** get list of cmp strings from itm->cmp
 */

BOOL cpi_itm_get (CPI_REC_ITM * itm,char * sym)
{
 SYS_STR s;
 IDX     i;
 BOOL    t;

 if (itm == NULL || sym == NULL)
  return (FALSE);

 strcpy (s,"");

 t = FALSE;

 for (i = 0;i < CMP_N;i++)
  if (TEST (itm -> cnc,BIT (i)))
  {
   if (t)
    strcat (s," ");

   strcat (s,itm -> cmp [i].stg);
   t = TRUE;
  }

 strcpy (sym,s);

 return (TRUE);
}

/******************************************************************************/
/** print record item to file
 */

BOOL cpi_itm_fprint (CPI_REC_ITM * itm,FILE * f)
{
 IDX  i;
 BOOL t;
 SYS_STR s;

 if (itm == NULL || f == NULL)
  return (FALSE);

 if (!cpi_itm_get (itm,s))
  return (FALSE);

 fprintf (f,"%s",s);

 return (TRUE);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** create new record item
 */

BOOL cpi_rec_create (CPI_REC ** cnr)
{
 if (cnr == NULL)
  return (FALSE);

 if ((*cnr = (CPI_REC *) malloc (sizeof (**cnr))) == NULL)
  return (FALSE);

 memset (*cnr,0,sizeof (**cnr));

 return (TRUE);
}

/******************************************************************************/
/** delete record item after clearing data
 */


BOOL cpi_rec_delete (CPI_REC ** cnr)
{
 if (cnr == NULL)
  return (FALSE);

 cpi_rec_clear (*cnr);

 free (*cnr);
 *cnr = NULL;


 return (TRUE);
}

/******************************************************************************/
/** clear data
 */

BOOL cpi_rec_clear (CPI_REC * cnr)
{
 if (cnr == NULL)
  return (FALSE);

 if (cnr -> n > 0)
 {
  int i;

  assert (cnr -> itm != NULL);
  free (cnr -> itm);
  cnr -> itm = NULL;
 }

 assert (cnr -> itm == NULL);

 cnr -> n = 0;


 return (TRUE); 
}

/******************************************************************************/
/** add record item to list
 */

BOOL cpi_rec_add (CPI_REC * cnr,CPI_REC_ITM * itm)
{
 CPI_REC_ITM * t;
 int k;

 if (cnr == NULL)
 return (FALSE);

 k = cnr -> n;

 if ((t = (CPI_REC_ITM *) realloc (cnr -> itm,(k + 1) * sizeof (CPI_REC_ITM))) == NULL)
  return (FALSE);

 cnr -> itm = t;
 cnr -> itm [k] = *itm;			// add item
 cnr -> n++;

 return (TRUE);
}

/******************************************************************************/
/** generate cipher records
 * @param cpr
 * @param rec original input record
 * @param fsp format specification
 * @param enc cipher encoding specification
 */

BOOL cpi_rec_gen (CPI_REC * cpr,PSX_REC * rec,PSX_FSP * fsp,CPE enc)
{
 IDX i;

 if (cpr == NULL || rec == NULL || fsp == NULL)
  return (FALSE);

 LOG (MOD,"generate cipher record");

 cpi_rec_clear (cpr);								// clear record

 for (i = 0;i < rec -> n;i++)						// loop over record items
 {
  CPI_REC_ITM itm;
    
  cpi_itm_gen (fsp,i,rec -> itm [i].val,enc,&itm);	// generate item

  if (!cpi_rec_add (cpr,&itm))						// add to list
   return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/** print every item record to file


 */

BOOL cpi_rec_fprint (CPI_REC * cnr,FILE * f)
{
 IDX i;

 if (cnr == NULL || f == NULL)
  return (FALSE);

 for (i = 0;i < cnr -> n;i++)
 {
  cpi_itm_fprint (&cnr -> itm [i],f);
  fprintf (f,"\n");
 }

 return (TRUE);
}

/******************************************************************************/
/** fill psx record dst with cpi information
 * @param src pointer to source record
 * @param dst pointer to destination record, contains ciphered record after invocation
 * @param fmt format specification
 * @param enc code for encoding mode ([1=raw|2=hex])
 */

BOOL cpi_gen (PSX_REC * src,PSX_REC * dst,PSX_FSP * fmt,CPE enc)
{
 CPI_REC * cpr;
 BOOL r;
 IDX i;

 if (src == NULL || dst == NULL || fmt == NULL)
  return (FALSE);

 r = TRUE;

 if (!cpi_rec_create (&cpr))			// create record
  return (FALSE);

 if (!cpi_rec_gen (cpr,src,fmt,enc))	// perform encipherment, result in cpr
 {
  cpi_rec_delete (&cpr);
  return (FALSE);
 }

 for (i = 0;i < cpr -> n;i++)			// loop over record items (lname, aname etc)
 {
  int j;
  // loop over components (C1,C2,C3,PC,PH)
  for (j = 0;j < CMP_N;j++)       // CMP_N = number of components in type name (5)
  {
   
   /* Test if component j is used for item i. This is always true for j==1
    * (every item has at least 1 component)
    */
   if (TEST (cpr -> itm [i].cnc,BIT (j))) // BIT(j) = bitshift j to left
   {

    PSS_STR lbl;     // label = name of field

    strcpy (lbl,cpr -> itm [i].sym); // copy field name to lbl
	
	/* Test if component j is not the only one of item i. This is the case,
	 * if other bits than 1 << j are set in cpr->itm[i].cnc. If there are several
	 * components, add suffix (_C1, _C2 etc.) to field name.
	 */
    if (cpr -> itm [i].cnc != BIT (j))  
    {
     strcat (lbl,"_");
     strcat (lbl,cpi_sfx [j]);
    }

    if (!rec_add1 (dst,lbl,cpr -> itm [i].cmp [j].stg))		// add item to output structure dst
    {
     r = FALSE;
     break;
    }
   }
  }
 }

 if (!cpi_rec_delete (&cpr))		// delete cpr
  r = FALSE;

 return (r);
}


/******************************************************************************/
/** map fsp encode/decode
 */

BOOL cpi_fsp_map (PSX_FSP * ifs,PSX_FSP * ofs,CPE enc)
{
 PSX_FSP * f;
 IDX pos,i;
 NUM gap;
 BOOL r;

 if (ifs == NULL || ofs == NULL)
  return (FALSE);

 if (!fsp_create (&f))			// create fsp
  return (FALSE);

 fsp_copy (f,ofs);				// copy fsp to f

 cpi_gap_get_len (enc,&gap);	// get gap length

 if (f -> n > 0)
 {
  pos = f -> itm [f -> n - 1].pos + f -> itm [f -> n - 1].len + 1;
 }
 else
  pos = 1;


 for (i = 0;i < ifs -> n;i++)
 {
  if (!cpi_fsp_itm_map (ifs,i,f,enc,&pos,MFM_SYM | MFM_CMP))		// encode/decode

  {
   fsp_delete (&f);
   return (FALSE);
  }
 }

 r = TRUE;


 if (!fsp_copy (ofs,f))
  return (FALSE);

 fsp_delete (&f);

 return (TRUE);
}

/******************************************************************************/
/** set cti_key
 */

BOOL cpi_set_key (const PSX_STR key)
{
 strcpy (cpi_key,key);
 return (TRUE);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


