/******************************************************************************/
/* psx-cpi.h                                                 Cipher Interface */
/******************************************************************************/
/** @file psx-cpi.h Cipher Interface - Header File
 * Definitions supporting the Transformation of data into control number
 * components.
 */
 
#ifndef _psx_cpi_
#define _psx_cpi_

/******************************************************************************/
/*                                                                            */
/******************************************************************************/

#include "psx.h"

/******************************************************************************/
/* Definition                                                                 */
/******************************************************************************/

/**************************************/
/* Cipher Component                   */
/**************************************/

typedef BYTE CMP;
/** @name Constants for cipher components
 * These constants identify field components (decomposed names
 * and phonetic code). They are used in bit patterns by doing
 * a bitshift <code>1 << CMP_XY</code>.
 */
//@{
#define CMP_N1    0x00
#define CMP_N2    0x01
#define CMP_N3    0x02
#define CMP_PC    0x03
#define CMP_PH    0x04
//@}

/** Number of components to encipher.
 * Includes name components and phonetic code.
 */
#define CMP_N		0x05
#define CMP_NIL		0xFF

/**************************************/
/* Cipher Composition                 */
/**************************************/
/*                                    */
/* |7|6|5|4|3|2|1|0|                  */
/*  | | | | | | | |                   */
/*  | | | | | | | +--- C1             */
/*  | | | | | | +----- C2             */
/*  | | | | | +------- C3             */
/*  | | | | +--------- PC             */
/*  | | | +----------- PH             */
/*  | | +-------------                */
/*  | +---------------                */
/*  +-----------------                */
/*                                    */
/**************************************/

//typedef	BYTE CPS; // -> FSP

#define CPS_ZERO      0x00
#define CPS_CMP(cmp)  BIT (cmp)

/**************************************/
/* Cipher Encoding                    */
/**************************************/

typedef BYTE CPE;

#define CPE_ZERO  0x00
#define CPE_RAW   0x01
#define CPE_HEX   0x02

/**************************************/
/* Map Format                         */
/**************************************/

typedef BYTE MFM;

#define MFM_ZERO	0x00
#define MFM_SYM		0x01
#define MFM_CMP		0x02

/******************************************************************************/
/* Structure                                                                  */
/******************************************************************************/

#define CPI_STG_MAX 1024

#define	CPI_CNT_ZERO	0x00
#define	CPI_CNT_TXT	0x01	/* free text */
#define	CPI_CNT_CTN	0x02	/* control numbers */
//#define	CPI_CNT_ENC	0x04	/* */
#define	CPI_CNT_DMP	0x05	/* final dump */

/** cpi record component                                           */
typedef struct _CPI_REC_CMP
{
 BYTE stg [CPI_STG_MAX];          /**< string, maximum length 1024 */
 NUM  len;                        /**< length                      */
 BYTE cnt;                        /**< component                   */
} CPI_REC_CMP;

/**************************************/
/** Control Number Record Item        */
/**************************************/

typedef struct _CPI_REC_ITM
{
 PSX_STR sym;
 BYTE    cnc;
 CPI_REC_CMP cmp [CMP_N];
} CPI_REC_ITM;

#define CNI_ITM_PUT(itm,cnc,src) strcpy ((itm).cmp [cnc],src);

/**************************************/
/** Cipher Record                     */
/**************************************/

typedef struct _CPI_REC
{
 CPI_REC_ITM * itm;
 NUM           n;
} CPI_REC;

/******************************************************************************/
/* Service                                                                    */
/******************************************************************************/

LINKAGE	BOOL	cpi_cmp_enc	(CMP cmp,PSX_STR sym);
LINKAGE	BOOL	cpi_cmp_dec	(CMP * cmp,PSX_STR sym);

LINKAGE	BOOL	cpi_cps_set	(CMP * cps,PSX_STR sym);
LINKAGE	BOOL	cpi_cps_get	(CMP cps,PSX_STR sym);
LINKAGE	BOOL	cpi_cps_map_tfm	(CPS * cps,TFM tfm);

LINKAGE	BOOL	cpi_gap_get_len	(CPE enc,NUM * n);

LINKAGE	BOOL	cpi_itm_gen	(PSX_FSP * fsp,IDX idx,PSX_STR txt,CPE enc,CPI_REC_ITM * itm);
LINKAGE	BOOL	cpi_itm_get	(CPI_REC_ITM * itm,char * sym);
LINKAGE	BOOL	cpi_itm_fprint	(CPI_REC_ITM * itm,FILE * f);

LINKAGE	BOOL	cpi_rec_create	(CPI_REC ** cnr);
LINKAGE	BOOL	cpi_rec_delete	(CPI_REC ** cnr);
LINKAGE	BOOL	cpi_rec_clear	(CPI_REC * cnr);
LINKAGE	BOOL	cpi_rec_add	(CPI_REC * cnr,CPI_REC_ITM * itm);
LINKAGE	BOOL	cpi_rec_gen	(CPI_REC * cpr,PSX_REC * rec,PSX_FSP * fsp,CPE enc);
LINKAGE	BOOL	cpi_rec_fprint	(CPI_REC * cnr,FILE * f);

LINKAGE	BOOL	cpi_gen		(PSX_REC * src,PSX_REC * dst,PSX_FSP * fmt,CPE enc);

LINKAGE	BOOL	cpi_fsp_map	(PSX_FSP * ifs,PSX_FSP * ofs,CPE enc);

LINKAGE	BOOL	cpi_set_key	(const PSX_STR key);

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/

#endif // _psx_cpi_

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


