/*****************************************************************************/
/* psx-cti-idea.c                     Cryptographic Transformation Interface */
/*****************************************************************************/
/** @file psx-cti-idea.c Cryptographic Transformation Interface: IDEA - Source Code File
 * Functions and definitions using MD5 and the IDEA Algorithm to encrypt data.
 *
 * The module is based on md5.c, idea.c and randpool.c from pgp source,
 * these are the original comment:
 *
 * - MD5 message-digest algorithm (md5.c from pgp source)
 * This code implements the MD5 message-digest algorithm.
 * The algorithm is due to Ron Rivest.  This code was
 * written by Colin Plumb in 1993, no copyright is claimed.
 * This code is in the public domain; do with it what you wish.
 *
 * Equivalent code is available from RSA Data Security, Inc.
 * This code has been tested against that, and is equivalent,
 * except that you don't need to include two pages of legalese
 * with every copy.
 *
 * To compute the message digest of a chunk of bytes, declare an
 * MD5Context structure, pass it to MD5Init, call MD5Update as
 * needed on buffers full of bytes, and then call MD5Final, which
 * will fill a supplied 16-byte array with the digest.
 *
 * The core of the MD5 algorithm, this alters an existing MD5 hash to
 * reflect the addition of 16 longwords of new data.  MD5Update blocks
 * the data and converts bytes into longwords for this routine.
 *
 * - True random number computation and storage (randpool.c from pgp source)
 * (c) Copyright 1990-1994 by Philip Zimmermann.  All rights reserved.
 * The author assumes no liability for damages resulting from the use
 * of this software, even if the damage results from defects in this
 * software.  No warranty is expressed or implied.
 *
 * Note that while most PGP source modules bear Philip Zimmermann's
 * copyright notice, many of them have been revised or entirely written
 * by contributors who frequently failed to put their names in their
 * code.  Code that has been incorporated into PGP from other authors
 * was either originally published in the public domain or is used with
 * permission from the various authors.
 *
 * PGP is available for free to the public under certain restrictions.
 * See the PGP User's Guide (included in the release package) for
 * important information about licensing, patent restrictions on
 * certain algorithms, trademarks, copyrights, and export controls.
 *
 * Written by Colin Plumb.
 *
 * - C source code for IDEA block cipher (idea.c from pgp source)
 * IDEA (International Data Encryption Algorithm), formerly known as
 * IPES (Improved Proposed Encryption Standard).
 * Algorithm developed by Xuejia Lai and James L. Massey, of ETH Zurich.
 * This implementation modified and derived from original C code
 * developed by Xuejia Lai.
 * Zero-based indexing added, names changed from IPES to IDEA.
 * CFB functions added.  Random number routines added.
 *
 * Extensively optimized and restructured by Colin Plumb.
 *
 * There are two adjustments that can be made to this code to
 * speed it up.  Defaults may be used for PCs.  Only the -DIDEA32
 * pays off significantly if selectively set or not set.
 * Experiment to see what works best for your machine.
 *
 * Multiplication: default is inline, -DAVOID_JUMPS uses a
 *            different version that does not do any conditional
 *            jumps (a few percent worse on a SPARC), while
 *            _DSMALL_CACHE takes it out of line to stay
 *            within a small on-chip code cache.
 * Variables: normally, 16-bit variables are used, but some
 *            machines (notably RISCs) do not have 16-bit registers,
 *            so they do a great deal of masking.  -DIDEA32 uses "int"
 *            register variables and masks explicitly only where
 *            necessary.  On a SPARC, for example, this boosts
 *            performace by 30%.
 *
 *      The IDEA(tm) block cipher is covered by patents held by ETH and a
 *      Swiss company called Ascom-Tech AG.  The Swiss patent number is
 *      PCT/CH91/00117, the European patent number is EP 0 482 154 B1, and
 *      the U.S. patent number is US005214703.  IDEA(tm) is a trademark of
 *      Ascom-Tech AG.  There is no license fee required for noncommercial
 *      use.  Commercial users may obtain licensing details from Dieter
 *      Profos, Ascom Tech AG, Solothurn Lab, Postfach 151, 4502 Solothurn,
 *      Switzerland, Tel +41 65 242885, Fax +41 65 235761.
 *
 *      The IDEA block cipher uses a 64-bit block size, and a 128-bit key
 *      size.  It breaks the 64-bit cipher block into four 16-bit words
 *      because all of the primitive inner operations are done with 16-bit
 *      arithmetic.  It likewise breaks the 128-bit cipher key into eight
 *      16-bit words.
 *
 *      For further information on the IDEA cipher, see the book:
 *        Xuejia Lai, "On the Design and Security of Block Ciphers",
 *        ETH Series on Information Processing (ed. J.L. Massey) Vol 1,
 *        Hartung-Gorre Verlag, Konstanz, Switzerland, 1992.  ISBN
 *        3-89191-573-X.
 *
 *      This code runs on arrays of bytes by taking pairs in big-endian
 *      order to make the 16-bit words that IDEA uses internally.  This
 *      produces the same result regardless of the byte order of the
 *      native CPU.
 */
 /* Transformation in ideaCipher()[line 858] */
 
#include "psx.h"

/******************************************************************************/
/* usuals.h from pgp source                                                   */
/******************************************************************************/

/* usuals.h - The usual typedefs, etc.
*/
#ifndef USUALS /* Assures no redefinitions of usual types...*/
#define USUALS

typedef unsigned char boolean;	/* values are TRUE or FALSE */
typedef unsigned char byte;	/* values are 0-255 */
typedef byte *byteptr;	/* pointer to byte */
typedef char *string;	/* pointer to ASCII character string */
typedef unsigned short word16;	/* values are 0-65535 */
#ifdef __alpha
typedef unsigned int word32;	/* values are 0-4294967295 */
#else
typedef unsigned long word32;	/* values are 0-4294967295 */
#endif

#ifndef TRUE
#define FALSE 0
#define TRUE (!FALSE)
#endif	/* if TRUE not already defined */

#ifndef min	/* if min macro not already defined */
#define min(a,b) (((a)<(b)) ? (a) : (b) )
#define max(a,b) (((a)>(b)) ? (a) : (b) )
#endif	/* if min macro not already defined */

/* void for use in pointers */
#ifndef NO_VOID_STAR
#define	VOID	void
#else
#define	VOID	char
#endif

	/* Zero-fill the byte buffer. */
#define fill0(buffer,count)	memset( buffer, 0, count )

	/* This macro is for burning sensitive data.  Many of the
	   file I/O routines use it for zapping buffers */
#define burn(x) fill0((VOID *)&(x),sizeof(x))

#endif	/* if USUALS not already defined */

/******************************************************************************/
/* md5.h from pgp source                                                      */
/******************************************************************************/

// copied from below for MD5Transform <begin>
#ifdef __alpha
typedef unsigned int uint32;
#else
typedef unsigned long uint32;
#endif
// copied from below for MD5Transform <end>

#if 0 // MW

#ifndef MD5_H
#define MD5_H

#ifdef __alpha
typedef unsigned int uint32;
#else
typedef unsigned long uint32;
#endif

struct MD5Context {
	uint32 buf[4];
	uint32 bits[2];
	unsigned char in[64];
};

void MD5Init(struct MD5Context *context);
void MD5Update(struct MD5Context *context, unsigned char const *buf,
	       unsigned len);
void MD5Final(unsigned char digest[16], struct MD5Context *context);
void MD5Transform(uint32 buf[4], uint32 const in[16]);

/*
 * This is needed to make RSAREF happy on some MS-DOS compilers.
 */
typedef struct MD5Context MD5_CTX;

#endif /* !MD5_H */

/******************************************************************************/
/* md5.c from pgp source                                                      */
/******************************************************************************/

/*
 * This code implements the MD5 message-digest algorithm.
 * The algorithm is due to Ron Rivest.  This code was
 * written by Colin Plumb in 1993, no copyright is claimed.
 * This code is in the public domain; do with it what you wish.
 *
 * Equivalent code is available from RSA Data Security, Inc.
 * This code has been tested against that, and is equivalent,
 * except that you don't need to include two pages of legalese
 * with every copy.
 *
 * To compute the message digest of a chunk of bytes, declare an
 * MD5Context structure, pass it to MD5Init, call MD5Update as
 * needed on buffers full of bytes, and then call MD5Final, which
 * will fill a supplied 16-byte array with the digest.
 */
#include <string.h>		/* for memcpy() */
//#include "md5.h"

#ifndef HIGHFIRST
#define byteReverse(buf, len)	/* Nothing */
#else
void byteReverse(unsigned char *buf, unsigned longs);

#ifndef ASM_MD5
/*****************************************************************************/
/** reverse byte
 * Note: this code is harmless on little-endian machines.
 * @param buf pointer to string buffer
 * @param longs number of characters
 */
void byteReverse(unsigned char *buf, unsigned longs)
{
    uint32 t;
    do {
	t = (uint32) ((unsigned) buf[3] << 8 | buf[2]) << 16 |
	    ((unsigned) buf[1] << 8 | buf[0]);
	*(uint32 *) buf = t;
	buf += 4;
    } while (--longs);
}
#endif
#endif

/*****************************************************************************/
/**
 * Start MD5 accumulation.  Set bit count to 0 and buffer to mysterious
 * initialization constants.
 * @param ctx pointer to context
 */
 
void MD5Init(struct MD5Context *ctx)
{
    ctx->buf[0] = 0x67452301;
    ctx->buf[1] = 0xefcdab89;
    ctx->buf[2] = 0x98badcfe;
    ctx->buf[3] = 0x10325476;

    ctx->bits[0] = 0;
    ctx->bits[1] = 0;
}

/*****************************************************************************/
/**
 * Update context to reflect the concatenation of another buffer full
 * of bytes.
 * @param ctx pointer to context
 * @param buf pointer to string buffer
 * @param len length
 */
 
void MD5Update(struct MD5Context *ctx, unsigned char const *buf, unsigned len)
{
    uint32 t;

    /* Update bitcount */

    t = ctx->bits[0];
    if ((ctx->bits[0] = t + ((uint32) len << 3)) < t)
	ctx->bits[1]++;		/* Carry from low to high */
    ctx->bits[1] += len >> 29;

    t = (t >> 3) & 0x3f;	/* Bytes already in shsInfo->data */

    /* Handle any leading odd-sized chunks */

    if (t) {
	unsigned char *p = (unsigned char *) ctx->in + t;

	t = 64 - t;
	if (len < t) {
	    memcpy(p, buf, len);
	    return;
	}
	memcpy(p, buf, t);
	byteReverse(ctx->in, 16);
	MD5Transform(ctx->buf, (uint32 *) ctx->in);
	buf += t;
	len -= t;
    }
    /* Process data in 64-byte chunks */

    while (len >= 64) {
	memcpy(ctx->in, buf, 64);
	byteReverse(ctx->in, 16);
	MD5Transform(ctx->buf, (uint32 *) ctx->in);
	buf += 64;
	len -= 64;
    }

    /* Handle any remaining bytes of data. */

    memcpy(ctx->in, buf, len);
}

/*****************************************************************************/
/**
 * Final wrapup - pad to 64-byte boundary with the bit pattern
 * 1 0* (64-bit count of bits processed, MSB-first)
 * @param digest[16] array of unsigned character, 16 bit word
 * @param ctx pointer to context 
 */
 
void MD5Final(unsigned char digest[16], struct MD5Context *ctx)
{
    unsigned count;
    unsigned char *p;

    /* Compute number of bytes mod 64 */
    count = (ctx->bits[0] >> 3) & 0x3F;

    /* Set the first char of padding to 0x80.  This is safe since there is
       always at least one byte free */
    p = ctx->in + count;
    *p++ = 0x80;

    /* Bytes of padding needed to make 64 bytes */
    count = 64 - 1 - count;

    /* Pad out to 56 mod 64 */
    if (count < 8) {
	/* Two lots of padding:  Pad the first block to 64 bytes */
	memset(p, 0, count);
	byteReverse(ctx->in, 16);
	MD5Transform(ctx->buf, (uint32 *) ctx->in);

	/* Now fill the next block with 56 bytes */
	memset(ctx->in, 0, 56);
    } else {
	/* Pad block to 56 bytes */
	memset(p, 0, count - 8);
    }
    byteReverse(ctx->in, 14);

    /* Append length in bits and transform */
    ((uint32 *) ctx->in)[14] = ctx->bits[0];
    ((uint32 *) ctx->in)[15] = ctx->bits[1];

    MD5Transform(ctx->buf, (uint32 *) ctx->in);
    byteReverse((unsigned char *) ctx->buf, 4);
    memcpy(digest, ctx->buf, 16);
    memset(ctx, 0, sizeof(ctx));	/* In case it's sensitive */
}

#endif // 0 [MW]

#ifndef ASM_MD5

/* The four core functions - F1 is optimized somewhat */

/* #define F1(x, y, z) (x & y | ~x & z) */
#define F1(x, y, z) (z ^ (x & (y ^ z)))
#define F2(x, y, z) F1(z, x, y)
#define F3(x, y, z) (x ^ y ^ z)
#define F4(x, y, z) (y ^ (x | ~z))

/* This is the central step in the MD5 algorithm. */
#define MD5STEP(f, w, x, y, z, data, s) \
	( w += f(x, y, z) + data,  w = w<<s | w>>(32-s),  w += x )

/*****************************************************************************/
/** The core of the MD5 algorithm, this alters an existing MD5 hash to
 *  reflect the addition of 16 longwords of new data.  MD5Update blocks
 *  the data and converts bytes into longwords for this routine.
 * @param buf[4] array of 4 characters
 * @param in[16] array of unsigned integer bit 32 bit each
 */
void MD5Transform(uint32 buf[4], uint32 const in[16])
{
    register uint32 a, b, c, d;

    a = buf[0];
    b = buf[1];
    c = buf[2];
    d = buf[3];

    MD5STEP(F1, a, b, c, d, in[0] + 0xd76aa478, 7);
    MD5STEP(F1, d, a, b, c, in[1] + 0xe8c7b756, 12);
    MD5STEP(F1, c, d, a, b, in[2] + 0x242070db, 17);
    MD5STEP(F1, b, c, d, a, in[3] + 0xc1bdceee, 22);
    MD5STEP(F1, a, b, c, d, in[4] + 0xf57c0faf, 7);
    MD5STEP(F1, d, a, b, c, in[5] + 0x4787c62a, 12);
    MD5STEP(F1, c, d, a, b, in[6] + 0xa8304613, 17);
    MD5STEP(F1, b, c, d, a, in[7] + 0xfd469501, 22);
    MD5STEP(F1, a, b, c, d, in[8] + 0x698098d8, 7);
    MD5STEP(F1, d, a, b, c, in[9] + 0x8b44f7af, 12);
    MD5STEP(F1, c, d, a, b, in[10] + 0xffff5bb1, 17);
    MD5STEP(F1, b, c, d, a, in[11] + 0x895cd7be, 22);
    MD5STEP(F1, a, b, c, d, in[12] + 0x6b901122, 7);
    MD5STEP(F1, d, a, b, c, in[13] + 0xfd987193, 12);
    MD5STEP(F1, c, d, a, b, in[14] + 0xa679438e, 17);
    MD5STEP(F1, b, c, d, a, in[15] + 0x49b40821, 22);

    MD5STEP(F2, a, b, c, d, in[1] + 0xf61e2562, 5);
    MD5STEP(F2, d, a, b, c, in[6] + 0xc040b340, 9);
    MD5STEP(F2, c, d, a, b, in[11] + 0x265e5a51, 14);
    MD5STEP(F2, b, c, d, a, in[0] + 0xe9b6c7aa, 20);
    MD5STEP(F2, a, b, c, d, in[5] + 0xd62f105d, 5);
    MD5STEP(F2, d, a, b, c, in[10] + 0x02441453, 9);
    MD5STEP(F2, c, d, a, b, in[15] + 0xd8a1e681, 14);
    MD5STEP(F2, b, c, d, a, in[4] + 0xe7d3fbc8, 20);
    MD5STEP(F2, a, b, c, d, in[9] + 0x21e1cde6, 5);
    MD5STEP(F2, d, a, b, c, in[14] + 0xc33707d6, 9);
    MD5STEP(F2, c, d, a, b, in[3] + 0xf4d50d87, 14);
    MD5STEP(F2, b, c, d, a, in[8] + 0x455a14ed, 20);
    MD5STEP(F2, a, b, c, d, in[13] + 0xa9e3e905, 5);
    MD5STEP(F2, d, a, b, c, in[2] + 0xfcefa3f8, 9);
    MD5STEP(F2, c, d, a, b, in[7] + 0x676f02d9, 14);
    MD5STEP(F2, b, c, d, a, in[12] + 0x8d2a4c8a, 20);

    MD5STEP(F3, a, b, c, d, in[5] + 0xfffa3942, 4);
    MD5STEP(F3, d, a, b, c, in[8] + 0x8771f681, 11);
    MD5STEP(F3, c, d, a, b, in[11] + 0x6d9d6122, 16);
    MD5STEP(F3, b, c, d, a, in[14] + 0xfde5380c, 23);
    MD5STEP(F3, a, b, c, d, in[1] + 0xa4beea44, 4);
    MD5STEP(F3, d, a, b, c, in[4] + 0x4bdecfa9, 11);
    MD5STEP(F3, c, d, a, b, in[7] + 0xf6bb4b60, 16);
    MD5STEP(F3, b, c, d, a, in[10] + 0xbebfbc70, 23);
    MD5STEP(F3, a, b, c, d, in[13] + 0x289b7ec6, 4);
    MD5STEP(F3, d, a, b, c, in[0] + 0xeaa127fa, 11);
    MD5STEP(F3, c, d, a, b, in[3] + 0xd4ef3085, 16);
    MD5STEP(F3, b, c, d, a, in[6] + 0x04881d05, 23);
    MD5STEP(F3, a, b, c, d, in[9] + 0xd9d4d039, 4);
    MD5STEP(F3, d, a, b, c, in[12] + 0xe6db99e5, 11);
    MD5STEP(F3, c, d, a, b, in[15] + 0x1fa27cf8, 16);
    MD5STEP(F3, b, c, d, a, in[2] + 0xc4ac5665, 23);

    MD5STEP(F4, a, b, c, d, in[0] + 0xf4292244, 6);
    MD5STEP(F4, d, a, b, c, in[7] + 0x432aff97, 10);
    MD5STEP(F4, c, d, a, b, in[14] + 0xab9423a7, 15);
    MD5STEP(F4, b, c, d, a, in[5] + 0xfc93a039, 21);
    MD5STEP(F4, a, b, c, d, in[12] + 0x655b59c3, 6);
    MD5STEP(F4, d, a, b, c, in[3] + 0x8f0ccc92, 10);
    MD5STEP(F4, c, d, a, b, in[10] + 0xffeff47d, 15);
    MD5STEP(F4, b, c, d, a, in[1] + 0x85845dd1, 21);
    MD5STEP(F4, a, b, c, d, in[8] + 0x6fa87e4f, 6);
    MD5STEP(F4, d, a, b, c, in[15] + 0xfe2ce6e0, 10);
    MD5STEP(F4, c, d, a, b, in[6] + 0xa3014314, 15);
    MD5STEP(F4, b, c, d, a, in[13] + 0x4e0811a1, 21);
    MD5STEP(F4, a, b, c, d, in[4] + 0xf7537e82, 6);
    MD5STEP(F4, d, a, b, c, in[11] + 0xbd3af235, 10);
    MD5STEP(F4, c, d, a, b, in[2] + 0x2ad7d2bb, 15);
    MD5STEP(F4, b, c, d, a, in[9] + 0xeb86d391, 21);

    buf[0] += a;
    buf[1] += b;
    buf[2] += c;
    buf[3] += d;
}

#endif


/******************************************************************************/
/* randpool.h from pgp source                                                 */
/******************************************************************************/

//#include "usuals.h"

/* Set this to whatever you need (must be > 512) */
#define RANDPOOLBITS 3072

void randPoolStir(void);
void randPoolAddBytes(byte const *buf, unsigned len);
void randPoolGetBytes(byte *buf, unsigned len);
byte randPoolGetByte(void);


/******************************************************************************/
/* randpool.c from pgp source                                                 */
/******************************************************************************/

/*
 * True random number computation and storage
 *
 * (c) Copyright 1990-1994 by Philip Zimmermann.  All rights reserved.
 * The author assumes no liability for damages resulting from the use
 * of this software, even if the damage results from defects in this
 * software.  No warranty is expressed or implied.
 *
 * Note that while most PGP source modules bear Philip Zimmermann's
 * copyright notice, many of them have been revised or entirely written
 * by contributors who frequently failed to put their names in their
 * code.  Code that has been incorporated into PGP from other authors
 * was either originally published in the public domain or is used with
 * permission from the various authors.
 *
 * PGP is available for free to the public under certain restrictions.
 * See the PGP User's Guide (included in the release package) for
 * important information about licensing, patent restrictions on
 * certain algorithms, trademarks, copyrights, and export controls.
 *
 * Written by Colin Plumb.
 */
#include <stdlib.h>
#include <string.h>

//#include "randpool.h"
//#include "usuals.h"
//#include "md5.h"

/* The pool must be a multiple of the 16-byte (128-bit) MD5 block size */
#define RANDPOOLWORDS ((RANDPOOLBITS+127 & ~127) >> 5)
#if RANDPOOLWORDS <= 16
/* #error is not portable, this has the same effect */
#include "Random pool too small - please increase RANDPOOLBITS in randpool.h"
#endif

/* Must be word-aligned, so make it words.  Cast to bytes as needed. */
static word32 randPool[RANDPOOLWORDS];	/* Random pool */
static unsigned randPoolGetPos = sizeof(randPool); /* Position to get from */
static unsigned randPoolAddPos = 0;	/* Position to add to */
/*****************************************************************************/
/** implement logical operator XOR
 * @param dest pointer to destination byte, result after invocation
 * @param src pointer to source byte
 * @param len length of string to be XORed
 */
 
static void xorbytes(byte *dest, byte const *src, unsigned len)
{
	while (len--)
		*dest++ ^= *src++;
}

/*****************************************************************************/
/**
 * Destroys already-used random numbers.  Ensures no sensitive data
 * remains in memory that can be recovered later.  This is also
 * called to "stir in" newly acquired environmental noise bits before
 * removing any random bytes.
 *
 * The transformation is carried out by "encrypting" the data in CFB
 * mode with MD5 as the block cipher.  Then, to make certain the stirring
 * operation is strictly one-way, we destroy the key, getting 64 bytes
 * from the beginning of the pool and using them to reinitialize the
 * key.  These bytes are not returned by randPoolGetBytes().
 *
 * The stirring operation is done twice, to ensure that each bit in the
 * pool depends on each bit of entropy XORed in after each call to
 * randPoolStir().
 *
 * To make this useful for pseudo-random (that is, repeatable) operations,
 * the MD5 transformation is always done with a consistent byte order.
 * MD5Transform itself works with 32-bit words, not bytes, so the pool,
 * usually an array of bytes, is transformed into an array of 32-bit words,
 * taking each group of 4 bytes in big-endian order.  At the end of the
 * stirring, the transformation is reversed.
 */

void randPoolStir(void)
{
	int i;
	byte *p;
	word32 t;
	word32 iv[4];
	static word32 randPoolKey[16] = {0};

	/* Convert to word32s for stirring operation */
	p = (byte *)randPool;
	for (i = 0; i < RANDPOOLWORDS; i++) {
		t = (word32)((unsigned)p[3]<<8 | p[2]) << 16 |
		             (unsigned)p[1]<<8 | p[0];
		randPool[i] = t;
		p += 4;
	}

	/* Start IV from last block of randPool */
	memcpy(iv, randPool+RANDPOOLWORDS-4, sizeof(iv));

	/* First CFB pass */
	for (i = 0; i < RANDPOOLWORDS; i += 4) {
		MD5Transform(iv, randPoolKey);
		iv[0] = randPool[i  ] ^= iv[0];
		iv[1] = randPool[i+1] ^= iv[1];
		iv[2] = randPool[i+2] ^= iv[2];
		iv[3] = randPool[i+3] ^= iv[3];
	}

	/* Get new key */
	memcpy(randPoolKey, randPool, sizeof(randPoolKey));

	/* Second CFB pass */
	for (i = 0; i < RANDPOOLWORDS; i += 4) {
		MD5Transform(iv, randPoolKey);
		iv[0] = randPool[i  ] ^= iv[0];
		iv[1] = randPool[i+1] ^= iv[1];
		iv[2] = randPool[i+2] ^= iv[2];
		iv[3] = randPool[i+3] ^= iv[3];
	}

	/* Get new key */
	memcpy(randPoolKey, randPool, sizeof(randPoolKey));

	/* Wipe iv from memory */
	memset(iv, 0, sizeof(iv));

	/* Convert randPool back to bytes for further use */
	p = (byte *)randPool;
	for (i = 0; i < RANDPOOLWORDS; i++) {
		t = randPool[i];
		p[0] = t>>24;
		p[1] = t>>16;
		p[2] = t>>8;
		p[3] = t;
		p += 4;
	}

	/* Set up pointers for future addition or removal of random bytes */
	randPoolAddPos = 0;
	randPoolGetPos = sizeof(randPoolKey);
}

/*****************************************************************************/
/** Make a deposit of information (entropy) into the pool.  The bits
 *  deposited need not have any particular distribution; the stirring
 *  operation transformes them to uniformly-distributed bits.
 * @param pointer to characterbuffer
 * @param len length
 */
 
void randPoolAddBytes(byte const *buf, unsigned len)
{
	unsigned t;

	while (len > (t = sizeof(randPool) - randPoolAddPos)) {
		xorbytes((byte *)randPool+randPoolAddPos, buf, t);
		buf += t;
		len -= t;
		randPoolStir();
	}

	if (len) {
		xorbytes((byte *)randPool+randPoolAddPos, buf, len);
		randPoolAddPos += len;
		randPoolGetPos = sizeof(randPool); /* Force stir on get */
	}
}

/*****************************************************************************/
/** Withdraw some bits from the pool.  Regardless of the distribution of the
 *  input bits, the bits returned are uniformly distributed, although they
 *  cannot, of course, contain more Shannon entropy than the input bits.
 * @param buf pointer to byte buffer
 * @param len length
 */
 
void randPoolGetBytes(byte *buf, unsigned len)
{
	unsigned t;

	while (len > (t = sizeof(randPool) - randPoolGetPos)) {
		memcpy(buf, (byte *)randPool+randPoolGetPos, t);
		buf += t;
		len -= t;
		randPoolStir();
	}

	if (len) {
		memcpy(buf, (byte *)randPool+randPoolGetPos, len);
		randPoolGetPos += len;
	}
}

byte
randPoolGetByte(void)
{
	if (randPoolGetPos == sizeof(randPool))
		randPoolStir();

	return (((byte *)randPool)[randPoolGetPos++]);
}


/******************************************************************************/
/* idea.c from pgp source                                                     */
/******************************************************************************/

#ifndef IDEA_H
#define IDEA_H

/*
 *	idea.h - header file for idea.c
 */

//#include "usuals.h"  /* typedefs for byte, word16, boolean, etc. */

#define IDEAKEYSIZE 16
#define IDEABLOCKSIZE 8

#define IDEAROUNDS 8
#define IDEAKEYLEN (6*IDEAROUNDS+4)

/*****************************************************************************/
/** iv[] is used as a circular buffer.  bufleft is the number of
 *  bytes at the end which have to be filled in before we crank
 *  the block cipher again.  We do the block cipher operation
 *  lazily: bufleft may be 0.  When we need one more byte, we
 *  crank the block cipher and set bufleft to 7.
 *
 *  oldcipher[] holds the previous 8 bytes of ciphertext, for use
 *  by ideaCfbSync() and Phil's, ahem, unique (not insecure, just
 *  unusual) way of doing CFB encryption.
 */
 
struct IdeaCfbContext {
	byte oldcipher[8];
	byte iv[8];
	word16 key[IDEAKEYLEN];
	int bufleft;
};

struct IdeaRandContext {
	byte outbuf[8];
	word16 key[IDEAKEYLEN];
	int bufleft;
	byte internalbuf[8];
};

void ideaCfbReinit(struct IdeaCfbContext *context, byte const *iv);
void ideaCfbInit(struct IdeaCfbContext *context, byte const (key[16]));
void ideaCfbSync(struct IdeaCfbContext *context);
void ideaCfbDestroy(struct IdeaCfbContext *context);
void ideaCfbEncrypt(struct IdeaCfbContext *context,
		    byte const *src, byte *dest, int count);
void ideaCfbDecrypt(struct IdeaCfbContext *context,
		    byte const *src, byte *dest, int count);
void ideaRandInit(struct IdeaRandContext *context, byte const (key[16]),
		  byte const (seed[8]));
byte ideaRandByte(struct IdeaRandContext *c);
void ideaRandWash(struct IdeaRandContext *c, struct IdeaCfbContext *cfb);
void ideaRandState(struct IdeaRandContext *c, byte key[16], byte seed[8]);

#endif /* !IDEA_H */

/******************************************************************************/
/* idea.c from pgp source                                                     */
/******************************************************************************/

/*
 *    idea.c - C source code for IDEA block cipher.
 *      IDEA (International Data Encryption Algorithm), formerly known as
 *      IPES (Improved Proposed Encryption Standard).
 *      Algorithm developed by Xuejia Lai and James L. Massey, of ETH Zurich.
 *      This implementation modified and derived from original C code
 *      developed by Xuejia Lai.
 *      Zero-based indexing added, names changed from IPES to IDEA.
 *      CFB functions added.  Random number routines added.
 *
 *      Extensively optimized and restructured by Colin Plumb.
 *
 *      There are two adjustments that can be made to this code to
 *      speed it up.  Defaults may be used for PCs.  Only the -DIDEA32
 *      pays off significantly if selectively set or not set.
 *      Experiment to see what works best for your machine.
 *
 *      Multiplication: default is inline, -DAVOID_JUMPS uses a
 *              different version that does not do any conditional
 *              jumps (a few percent worse on a SPARC), while
 *              -DSMALL_CACHE takes it out of line to stay
 *              within a small on-chip code cache.
 *      Variables: normally, 16-bit variables are used, but some
 *              machines (notably RISCs) do not have 16-bit registers,
 *              so they do a great deal of masking.  -DIDEA32 uses "int"
 *              register variables and masks explicitly only where
 *              necessary.  On a SPARC, for example, this boosts
 *              performace by 30%.
 *
 *      The IDEA(tm) block cipher is covered by patents held by ETH and a
 *      Swiss company called Ascom-Tech AG.  The Swiss patent number is
 *      PCT/CH91/00117, the European patent number is EP 0 482 154 B1, and
 *      the U.S. patent number is US005214703.  IDEA(tm) is a trademark of
 *      Ascom-Tech AG.  There is no license fee required for noncommercial
 *      use.  Commercial users may obtain licensing details from Dieter
 *      Profos, Ascom Tech AG, Solothurn Lab, Postfach 151, 4502 Solothurn,
 *      Switzerland, Tel +41 65 242885, Fax +41 65 235761.
 *
 *      The IDEA block cipher uses a 64-bit block size, and a 128-bit key
 *      size.  It breaks the 64-bit cipher block into four 16-bit words
 *      because all of the primitive inner operations are done with 16-bit
 *      arithmetic.  It likewise breaks the 128-bit cipher key into eight
 *      16-bit words.
 *
 *      For further information on the IDEA cipher, see the book:
 *        Xuejia Lai, "On the Design and Security of Block Ciphers",
 *        ETH Series on Information Processing (ed. J.L. Massey) Vol 1,
 *        Hartung-Gorre Verlag, Konstanz, Switzerland, 1992.  ISBN
 *        3-89191-573-X.
 *
 *      This code runs on arrays of bytes by taking pairs in big-endian
 *      order to make the 16-bit words that IDEA uses internally.  This
 *      produces the same result regardless of the byte order of the
 *      native CPU.
 */

//#include "idea.h"
//#include "psx-cti-idea.h"
//#include "randpool.h"

#ifdef IDEA32			/* Use >16-bit temporaries */
#define low16(x) ((x) & 0xFFFF)
typedef unsigned int uint16;	/* at LEAST 16 bits, maybe more */
#else
#define low16(x) (x)		/* this is only ever applied to uint16's */
typedef word16 uint16;
#endif

#ifdef _GNUC_
/* __const__ simply means there are no side effects for this function,
 * which is useful info for the gcc optimizer
 */
#define CONST __const__
#else
#define CONST
#endif


#ifdef SMALL_CACHE
/*****************************************************************************/
/**Multiplication, modulo (2**16)+1
 * Note that this code is structured on the assumption that
 * untaken branches are cheaper than taken branches, and the
 * compiler doesn't schedule branches.
 * @param a unsigned integer with 16 bit (should be stored in register)
 * @param b unsigned integer with 16 bit (should be stored in register)
 */

CONST static uint16
 mul(register uint16 a, register uint16 b)
{
    register word32 p;

    p = (word32) a *b;
    if (p) {
	b = low16(p);
	a = p >> 16;
	return (b - a) + (b < a);
    } else if (a) {
	return 1 - a;
    } else {
	return 1 - b;
    }
}				/* mul */
#endif				/* SMALL_CACHE */

/*****************************************************************************/
/** Compute the multiplicative inverse of x, modulo 65537, using Euclid's
 *  algorithm. It is unrolled twice to avoid swapping the registers each
 *  iteration, and some subtracts of t have been changed to adds.
 * @param x unsigned integer with 16 bit
 */
 
CONST static uint16 mulInv(uint16 x)
{
    uint16 t0, t1;
    uint16 q, y;

    if (x <= 1)
	return x;		/* 0 and 1 are self-inverse */
    t1 = 0x10001L / x;		/* Since x >= 2, this fits into 16 bits */
    y = 0x10001L % x;
    if (y == 1)
	return low16(1 - t1);
    t0 = 1;
    do {
	q = x / y;
	x = x % y;
	t0 += q * t1;
	if (x == 1)
	    return t0;
	q = y / x;
	y = y % x;
	t1 += q * t0;
    } while (y != 1);
    return low16(1 - t1);
}				/* mukInv */

/*****************************************************************************/
/** Expand a 128-bit user key to a working encryption key EK
 * @param userkey byte pointer to userkey
 * @param EK pointer to expandkey 
 */
 
static void ideaExpandKey(byte const *userkey, word16 * EK)
{
    int i, j;

    for (j = 0; j < 8; j++) {
	EK[j] = (userkey[0] << 8) + userkey[1];
	userkey += 2;
    }
    for (i = 0; j < IDEAKEYLEN; j++) {
	i++;
	EK[i + 7] = EK[i & 7] << 9 | EK[i + 1 & 7] >> 7;
	EK += i & 8;
	i &= 7;
    }
}				/* ideaExpandKey */


/*****************************************************************************/
/** Compute IDEA decryption key DK from an expanded IDEA encryption key EK
 *  Note that the input and output may be the same.  Thus, the key is
 *  inverted into an internal buffer, and then copied to the output.
 * @param EK byte pointer to expand key
 * @param DK byte pointer to IDEA Key
 */
 
static void ideaInvertKey(word16 const *EK, word16 DK[IDEAKEYLEN])
{
    int i;
    uint16 t1, t2, t3;
    word16 temp[IDEAKEYLEN];
    word16 *p = temp + IDEAKEYLEN;

    t1 = mulInv(*EK++);
    t2 = -*EK++;
    t3 = -*EK++;
    *--p = mulInv(*EK++);
    *--p = t3;
    *--p = t2;
    *--p = t1;

    for (i = 0; i < IDEAROUNDS - 1; i++) {
	t1 = *EK++;
	*--p = *EK++;
	*--p = t1;

	t1 = mulInv(*EK++);
	t2 = -*EK++;
	t3 = -*EK++;
	*--p = mulInv(*EK++);
	*--p = t2;
	*--p = t3;
	*--p = t1;
    }
    t1 = *EK++;
    *--p = *EK++;
    *--p = t1;

    t1 = mulInv(*EK++);
    t2 = -*EK++;
    t3 = -*EK++;
    *--p = mulInv(*EK++);
    *--p = t3;
    *--p = t2;
    *--p = t1;
/* Copy and destroy temp copy */
    memcpy(DK, temp, sizeof(temp));
    burn(temp);
}				/* ideaInvertKey */


/*****************************************************************************/
/** MUL(x,y) computes x = x*y, modulo 0x10001.  Requires two temps,
 *  t16 and t32.  x is modified, and must be a side-effect-free lvalue.
 *  y may be anything, but unlike x, must be strictly less than 65536
 *  even if low16() is #defined.
 *  All of these are equivalent - see which is faster on your machine
 */
 
#ifdef SMALL_CACHE
#define MUL(x,y) (x = mul(low16(x),y))
#else				/* !SMALL_CACHE */
#ifdef AVOID_JUMPS
#define MUL(x,y) (x = low16(x-1), t16 = low16((y)-1), \
		t32 = (word32)x*t16 + x + t16, x = low16(t32), \
		t16 = t32>>16, x = (x-t16) + (x<t16) + 1)
#else				/* !AVOID_JUMPS (default) */
#define MUL(x,y) \
	((t16 = (y)) ? \
		(x=low16(x)) ? \
			t32 = (word32)x*t16, \
			x = low16(t32), \
			t16 = t32>>16, \
			x = (x-t16)+(x<t16) \
		: \
			(x = 1-t16) \
	: \
		(x = 1-x))
#endif
#endif

/*      IDEA encryption/decryption algorithm */
/* Note that in and out can be the same buffer */
static void ideaCipher(byte const inbuf[8], byte outbuf[8],
		       word16 const *key)
{
    register uint16 x1, x2, x3, x4, s2, s3;
    word16 *in, *out;
#ifndef SMALL_CACHE
    register uint16 t16;	/* Temporaries needed by MUL macro */
    register word32 t32;
#endif
    int r = IDEAROUNDS;

    in = (word16 *) inbuf;
    x1 = *in++;
    x2 = *in++;
    x3 = *in++;
    x4 = *in;
#ifndef HIGHFIRST
    x1 = (x1 >> 8) | (x1 << 8);
    x2 = (x2 >> 8) | (x2 << 8);
    x3 = (x3 >> 8) | (x3 << 8);
    x4 = (x4 >> 8) | (x4 << 8);
#endif
    do {
	MUL(x1, *key++);
	x2 += *key++;
	x3 += *key++;
	MUL(x4, *key++);

	s3 = x3;
	x3 ^= x1;
	MUL(x3, *key++);
	s2 = x2;
	x2 ^= x4;
	x2 += x3;
	MUL(x2, *key++);
	x3 += x2;

	x1 ^= x2;
	x4 ^= x3;

	x2 ^= s3;
	x3 ^= s2;
    } while (--r);
    MUL(x1, *key++);
    x3 += *key++;
    x2 += *key++;
    MUL(x4, *key);

    out = (word16 *) outbuf;
#ifdef HIGHFIRST
    *out++ = x1;
    *out++ = x3;
    *out++ = x2;
    *out = x4;
#else				/* !HIGHFIRST */
    x1 = low16(x1);
    x2 = low16(x2);
    x3 = low16(x3);
    x4 = low16(x4);
    *out++ = (x1 >> 8) | (x1 << 8);
    *out++ = (x3 >> 8) | (x3 << 8);
    *out++ = (x2 >> 8) | (x2 << 8);
    *out = (x4 >> 8) | (x4 << 8);
#endif
}				/* ideaCipher */

/*-------------------------------------------------------------*/

#ifdef TEST

#include <stdio.h>
#include <time.h>
/*
 * This is the number of Kbytes of test data to encrypt.
 * It defaults to 1 MByte.
 */
#ifndef BLOCKS
#ifndef KBYTES
#define KBYTES 1024
#endif
#define BLOCKS (64*KBYTES)
#endif

int idea_main(void)
{				/* Test driver for IDEA cipher */
    int i, j, k;
    byte userkey[16];
    word16 EK[IDEAKEYLEN], DK[IDEAKEYLEN];
    byte XX[8], YY[8], ZZ[8];
    clock_t start, end;
    long l;

    /* Make a sample user key for testing... */
    for (i = 0; i < 16; i++)
	userkey[i] = i + 1;

    /* Compute encryption subkeys from user key... */
    ideaExpandKey(userkey, EK);
    printf("\nEncryption key subblocks: ");
    for (j = 0; j < IDEAROUNDS + 1; j++) {
	printf("\nround %d:   ", j + 1);
	if (j < IDEAROUNDS)
	    for (i = 0; i < 6; i++)
		printf(" %6u", EK[j * 6 + i]);
	else
	    for (i = 0; i < 4; i++)
		printf(" %6u", EK[j * 6 + i]);
    }

    /* Compute decryption subkeys from encryption subkeys... */
    ideaInvertKey(EK, DK);
    printf("\nDecryption key subblocks: ");
    for (j = 0; j < IDEAROUNDS + 1; j++) {
	printf("\nround %d:   ", j + 1);
	if (j < IDEAROUNDS)
	    for (i = 0; i < 6; i++)
		printf(" %6u", DK[j * 6 + i]);
	else
	    for (i = 0; i < 4; i++)
		printf(" %6u", DK[j * 6 + i]);
    }

    /* Make a sample plaintext pattern for testing... */
    for (k = 0; k < 8; k++)
	XX[k] = k;

    printf("\n Encrypting %d bytes (%ld blocks)...", BLOCKS * 16, BLOCKS);
    fflush(stdout);
    start = clock();
    memcpy(YY, XX, 8);
    for (l = 0; l < BLOCKS; l++)
	ideaCipher(YY, YY, EK);	/* repeated encryption */
    memcpy(ZZ, YY, 8);
    for (l = 0; l < BLOCKS; l++)
	ideaCipher(ZZ, ZZ, DK);	/* repeated decryption */
    end = clock() - start;
    l = end / (CLOCKS_PER_SEC / 1000) + 1;
    i = l / 1000;
    j = l % 1000;
    l = (16 * BLOCKS * (CLOCKS_PER_SEC / 1000)) / (end / 1000);
    printf("%d.%03d seconds = %ld bytes per second\n", i, j, l);

    printf("\nX %3u  %3u  %3u  %3u  %3u  %3u  %3u %3u\n",
	   XX[0], XX[1], XX[2], XX[3], XX[4], XX[5], XX[6], XX[7]);
    printf("\nY %3u  %3u  %3u  %3u  %3u  %3u  %3u %3u\n",
	   YY[0], YY[1], YY[2], YY[3], YY[4], YY[5], YY[6], YY[7]);
    printf("\nZ %3u  %3u  %3u  %3u  %3u  %3u  %3u %3u\n",
	   ZZ[0], ZZ[1], ZZ[2], ZZ[3], ZZ[4], ZZ[5], ZZ[6], ZZ[7]);

    /* Now decrypted ZZ should be same as original XX */
    for (k = 0; k < 8; k++)
	if (XX[k] != ZZ[k]) {
	    printf("\n\07Error!  Noninvertable encryption.\n");
	    exit(-1);		/* error exit */
	}
    printf("\nNormal exit.\n");
    return 0;			/* normal exit */
}				/* main */

#endif				/* TEST */


/*************************************************************************/

void ideaCfbReinit(struct IdeaCfbContext *context, byte const *iv)
{
    if (iv)
	memcpy(context->iv, iv, 8);
    else
	fill0(context->iv, 8);
    context->bufleft = 0;
}

void ideaCfbInit(struct IdeaCfbContext *context, byte const key[16])
{
    ideaExpandKey(key, context->key);
    ideaCfbReinit(context, 0);
}

void ideaCfbDestroy(struct IdeaCfbContext *context)
{
    burn(*context);
}

/*****************************************************************************/
/** Philip Zimmermann invented a unique way of doing CFB that's sensitive to semantic
 * boundaries within the data being encrypted.  One way to phrase
 * CFB en/decryption is to say that you XOR the current 8 bytes with
 * IDEA(previous 8 bytes of ciphertext).  Normally, you repeat this
 * at 8-byte intervals, but Phil decided to resync things on the
 * boundaries between elements in the stream being encrypted.
 *
 * That is, the last 4 bytes of a 12-byte field are en/decrypted using
 * the first 4 bytes of IDEA(previous 8 bytes of ciphertext), but then
 * the last 4 bytes of that IDEA computation are thrown away, and the
 * first 8 bytes of the next field are en/decrypted using
 * IDEA(last 8 bytes of ciphertext).  This is equivalent to using a
 * shorter feedback length (if you're familiar with the general CFB
 * technique) briefly, and doesn't weaken the cipher any (using shorter
 * CFB lengths makes it stronger, actually), it just makes it a bit unusual.
 *
 * Anyway, to accomodate this behaviour, every time we do an IDEA
 * encrpytion of 8 bytes of ciphertext to get 8 bytes of XOR mask,
 * we remember the ciphertext.  Then if we have to resync things
 * after having processed, say, 2 bytes, we refill the iv buffer
 * with the last 6 bytes of the old ciphertext followed by the
 * 2 bytes of new ciphertext stored in the front of the iv buffer.
 */
 
void ideaCfbSync(struct IdeaCfbContext *context)
{
    int bufleft = context->bufleft;

    if (bufleft) {
	memmove(context->iv + bufleft, context->iv, 8 - bufleft);
	memcpy(context->iv, context->oldcipher + 8 - bufleft, bufleft);
	context->bufleft = 0;
    }
}

/*****************************************************************************/
/** Encrypt a buffer of data, using IDEA in CFB mode.
 * There are more compact ways of writing this, but this is
 * written for speed.
 */
 
void ideaCfbEncrypt(struct IdeaCfbContext *context, byte const *src,
		    byte * dest, int count)
{
    int bufleft = context->bufleft;
    byte *bufptr = context->iv + 8 - bufleft;

    /* If there are no more bytes to encrypt that there are bytes
     * in the buffer, XOR them in and return.
     */
    if (count <= bufleft) {
	context->bufleft = bufleft - count;
	while (count--) {
	    *dest++ = *bufptr++ ^= *src++;
	}
	return;
    }
    count -= bufleft;
    /* Encrypt the first bufleft (0 to 7) bytes of the input by XOR
     * with the last bufleft bytes in the iv buffer.
     */
    while (bufleft--) {
	*dest++ = (*bufptr++ ^= *src++);
    }
    /* Encrypt middle blocks of the input by cranking the cipher,
     * XORing 8-byte blocks, and repeating until the count
     * is 8 or less.
     */
    while (count > 8) {
	bufptr = context->iv;
	memcpy(context->oldcipher, bufptr, 8);
	ideaCipher(bufptr, bufptr, context->key);
	bufleft = 8;
	count -= 8;
	do {
	    *dest++ = (*bufptr++ ^= *src++);
	} while (--bufleft);
    }
    /* Do the last 1 to 8 bytes */
    bufptr = context->iv;
    memcpy(context->oldcipher, bufptr, 8);
    ideaCipher(bufptr, bufptr, context->key);
    context->bufleft = 8 - count;
    do {
	*dest++ = (*bufptr++ ^= *src++);
    } while (--count);
}

/*****************************************************************************/
/** Decrypt a buffer of data, using IDEA in CFB mode.
 * There are more compact ways of writing this, but this is
 * written for speed.
 * @param context pointer to IDEA context
 * @param src byte pointer to source
 */
 
void ideaCfbDecrypt(struct IdeaCfbContext *context, byte const *src,
		    byte * dest, int count)
{
    int bufleft = context->bufleft;
    static byte *bufptr;
    byte t;

    bufptr = context->iv + (8 - bufleft);
    if (count <= bufleft) {
	context->bufleft = bufleft - count;
	while (count--) {
	    t = *bufptr;
	    *dest++ = t ^ (*bufptr++ = *src++);
	}
	return;
    }
    count -= bufleft;
    while (bufleft--) {
	t = *bufptr;
	*dest++ = t ^ (*bufptr++ = *src++);
    }
    while (count > 8) {
	bufptr = context->iv;
	memcpy(context->oldcipher, bufptr, 8);
	ideaCipher(bufptr, bufptr, context->key);
	bufleft = 8;
	count -= 8;
	do {
	    t = *bufptr;
	    *dest++ = t ^ (*bufptr++ = *src++);
	} while (--bufleft);
    }
    bufptr = context->iv;
    memcpy(context->oldcipher, bufptr, 8);
    ideaCipher(bufptr, bufptr, context->key);
    context->bufleft = 8 - count;
    do {
	t = *bufptr;
	*dest++ = t ^ (*bufptr++ = *src++);
    } while (--count);
}

/********************************************************************/
/** Cryptographically strong pseudo-random-number generator.
 * The design is from Appendix C of ANSI X9.17, "Financial
 * Institution Key Management (Wholesale)", with IDEA
 * substituted for the DES.
 *
 * Initialize a cryptographic random-number generator.
 * key and seed should be arbitrary.
 */
 
void ideaRandInit(struct IdeaRandContext *context, byte const key[16],
		  byte const seed[8])
{
    int i;

    ideaExpandKey(key, context->key);
    context->bufleft = 0;
    memcpy(context->internalbuf, seed, 8);
}


/*
 * Read out the RNG's state.
 */
void ideaRandState(struct IdeaRandContext *context, byte key[16], byte seed[8])
{
    int i;

    memcpy(seed, context->internalbuf, 8);
    for (i = 0; i < 8; i++) {
	key[2 * i] = context->key[i] >> 8;
	key[2 * i + 1] = context->key[i];
    }

}

/*****************************************************************************/
/** Encrypt the RNG's state with the given CFB encryptor.
 * @param context pointer to IDEA context
 * @param cfb pointer to Cfb context
 */
 
void ideaRandWash(struct IdeaRandContext *context, struct IdeaCfbContext *cfb)
{
    byte keyseed[16 + 8];
    int i;

    ideaRandState(context, keyseed, keyseed + 16);
    ideaCfbEncrypt(cfb, keyseed, keyseed, 16 + 8);
    ideaRandInit(context, keyseed, keyseed + 16);

    memset(keyseed, 0, 16 + 8);
}

/*****************************************************************************/
/** Cryptographic pseudo-random-number generator, used for generating
 * session keys.
 * @param c pointer to Context
 */
 
byte ideaRandByte(struct IdeaRandContext *c)
{
    int i;

    if (!c->bufleft) {
	byte timestamp[8];

	/* Get some true-random noise to help */
	randPoolGetBytes(timestamp, sizeof(timestamp));

	/* Compute next 8 bytes of output */
	for (i = 0; i < 8; i++)
	    c->outbuf[i] = c->internalbuf[i] ^ timestamp[i];
	ideaCipher(c->outbuf, c->outbuf, c->key);
	/* Compute new seed vector */
	for (i = 0; i < 8; i++)
	    c->internalbuf[i] = c->outbuf[i] ^ timestamp[i];
	ideaCipher(c->internalbuf, c->internalbuf, c->key);
	burn(timestamp);
	c->bufleft = 8;
    }
    return c->outbuf[--c->bufleft];
}

/* end of idea.c */






/******************************************************************************/
/* Private                                                         Adaptation */
/******************************************************************************/
/** initialize idea by logging the beginning
 */
 
BOOL idea_init (void)
{
 LOG (PSX_MOD_IDEA,"idea init");


 return (TRUE);
}

/******************************************************************************/
/** exit idea by logging the end
 */
 
BOOL idea_exit (void)
{
 LOG (PSX_MOD_IDEA,"idea exit");
 return (TRUE);
}

/******************************************************************************/
/** manage encoding
 * @param k string pointer to key string
 * @param p byte pointer
 * @param n number of characters
 */
 
BOOL idea_enc (char * k,BYTE * p,NUM n)
{
 BYTE t [1000],q [1000];
 struct IdeaCfbContext c;
 byte key [16];
 int l;

 l = strlen (k);

 if (l > 16)
  l = 16;

 memset (key,0,16);
 memcpy (key,k,l);

 memset (t,0,sizeof (t));
 memcpy (t,p,n);

 //

 ideaCfbInit (&c,key);
 ideaCfbEncrypt (&c,t,q,n);
 memcpy (p,q,n);
 ideaCfbDestroy (&c);

 //

 return (TRUE);
}

/******************************************************************************/
/** manage decoding
 * @param k string pointer to key string
 * @param p byte pointer
 * @param n unmber of characters
 */
 
BOOL idea_dec (char * k,BYTE * p,NUM n)
{
 BYTE t [1000],q [1000];
 struct IdeaCfbContext c;
 byte key [16];
 int l;

 l = strlen (k);

 if (l > 16)
  l = 16;

 memset (key,0,16);
 memcpy (key,k,l);

 memset (t,0,sizeof (t));
 memcpy (t,p,n);

 //

 ideaCfbInit (&c,key);
 ideaCfbDecrypt (&c,t,q,n);
 memcpy (p,q,n);
 ideaCfbDestroy (&c);

 //


 //return (TRUE);


 return (TRUE);
}

/******************************************************************************/
/* Public                                                          CTI Driver */
/******************************************************************************/
/** initialize driver (calls idea_init)
 */
 
static BOOL drv_init (void)
{
 if (!idea_init ())
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** exit idea (call idea_exit)
 */
 
static BOOL drv_exit (void)
{
 if (!idea_exit ())
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** manage encryption (calls idea_enc)
 * @param k key
 * @param p data pointer
 * @param number of characters
 */
 
static BOOL drv_enc (KEY k,DAT p,NUM n)
{
 if (k == NULL || p == NULL)
  return (FALSE);

 if (!idea_enc (k,p,n))
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** manage decryption
 * @param k key
 * @param p data pointer
 * @param n number of characters
 */
 
static BOOL drv_dec (KEY k,DAT p,NUM n)
{
 if (k == NULL || p == NULL)
  return (FALSE);

 if (!idea_dec (k,p,n))
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/

CTI_DRV cti_idea =
{
 drv_init,
 drv_exit,
 drv_enc,
 drv_dec
} ;

