/******************************************************************************/
/* psx-cti.c                      Cryptographic Transformation Interface      */
/******************************************************************************/
/** @file psx-cti.c Cryptographic Transformation Interface - Source Code File
 * Functions supporting the cryptographic transformation of data using either
 * AES or IDEA algorithm.
 * Functions in this file provide the service of selecting the correct driver
 * for the specified encryption/decryption algorithm.
 */
#include "psx.h"
#include "psx-cti-idea.h"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

static struct
{
 CTT       ctt;
 CTI_DRV * drv;
} cti_drv [] =
{
 { CTT_AES,&cti_aes },
 { CTT_IDEA,&cti_idea }
};

/******************************************************************************/
/** lookup cryptographic transformation interface driver (cti_drv)
 * @param ctt cryptographic transformation type (md5,AES or IDEA)
 * @param cryptographic transformation inteface driver structure 
 */

static BOOL cti_drv_lookup (CTT ctt,CTI_DRV ** drv)
{
 NUM n;
 IDX i;

 if (drv == NULL)
  return (FALSE);

 n = sizeof (cti_drv) / sizeof (cti_drv [0]);

 for (i = 0;i < n;i++)
  if (cti_drv [i].ctt == ctt)
  {
   *drv = cti_drv [i].drv;
   return (TRUE);
  }

 return (FALSE);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** initialize cti
 */

BOOL cti_init (void)
{
 NUM n,i;

 n = sizeof (cti_drv) / sizeof (cti_drv [0]); //

 // printf ("cti_drv n: %d\n",n);

 for (i = 0; i < n; i++)
 {
  CTI_DRV * d = cti_drv [i].drv;

  assert (d != NULL);

  if (!d -> init ())
  {
   IDX j;

   for (j = 0;j < i;j++)
    d -> exit ();

   return (FALSE);
  }
 }

 return (TRUE);
}

/******************************************************************************/
/** exit cti and reinitialize
 */

BOOL cti_exit (void)
{
 NUM n,i;
 BOOL r = TRUE;

 n = sizeof (cti_drv) / sizeof (cti_drv [0]);

 for (i = 0;i < n;i++)
 {
  CTI_DRV * d = cti_drv [i].drv;

  assert (d != NULL);

  if (!d -> init ())
   r = FALSE;
 }

 return (r);
}

/******************************************************************************/
/** encryption
 * @param ctt cryptographic transformation type
 * @param k encryption key
 * @param p data to be encrypted
 * @param n 
 */

BOOL cti_enc (CTT ctt,KEY k,DAT p,NUM n)
{
 CTI_DRV * d;

 if (k == NULL || p == NULL)
  return (FALSE);

 if (!cti_drv_lookup (ctt,&d))
  return (FALSE);

 if (!d -> enc (k,p,n))    // call encryption routine stored in drv structure
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/* decryption
 * @param ctt cryptographic transformation type 
 * @param k decryption key
 * @param p transformation data
 * @param n
 */

BOOL cti_dec (CTT ctt,KEY k,DAT p,NUM n)
{
 CTI_DRV * d;

 if (k == NULL || p == NULL)
  return (FALSE);

 if (!cti_drv_lookup (ctt,&d))         // find drv and store point d to it
  return (FALSE);

 if (!d -> dec (k,p,n))                // call dexcryption procedure stored in drv d
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** get encryption mode AES||IDEA
 * @param ctt cryptographic transformation type
 * @param sym result string (encryption mode)
 */

BOOL cti_ctt_enc (CTT ctt,PSX_STR sym)
{
 if (sym == NULL)
  return (FALSE);

 switch (ctt)
 {
  case CTT_AES:		strcpy (sym,"AES");	break;
  case CTT_IDEA:	strcpy (sym,"IDEA");	break;
  default:		strcpy (sym,"?");	return (FALSE);
 }
 return (TRUE);
}

/******************************************************************************/
/** get decryption mode
 * @param ctt cryptographic transformation type
 * @param sym result string (decryption mode)
 */

BOOL cti_ctt_dec (CTT * ctt,PSX_STR sym)
{
 if (ctt == NULL || sym == NULL)
  return (FALSE);

 if (strcmp (sym,"IDEA") == 0)
  *ctt = CTT_IDEA;
 else
  if (strcmp (sym,"AES") == 0)
   *ctt = CTT_AES;
  else
  {
   *ctt = CTT_ZERO;
   return (FALSE);
  }

 return (TRUE);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/



