/******************************************************************************/
/* psx-dbi-odbc.c                                           DBI Driver - ODBC */
/******************************************************************************/
/** @file psx-dbi-odbc.c Database Interface: ODBC Driver - Source Code File
 * Definitions and functions implementing an ODBC Interface for the connection
 * between PSX and a database.
 * This module provides functions for
 * - splitting the datasource string given by the config file
 * - opening the connection to the database
 * - deleting all tables from the database
 * - executing a sql statement
 * - fetching data delivered by the execution of a query
 * - closing the connection to the database
 * Contrasting the Postgres Database Interface the ODBC Driver does not contain
 * procedures for the generation of a database or for a status query.
 */
 
#include "psx.h"

#ifdef HAVE_ODBC /* is set by configure */

#include <sqlext.h>

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

/* For Logging: */
#define MOD "dbi/odbc"

/* maximum number of tables, appearing in this DB 
 * (used by odbc_delete() ):                      */
#define MAX_NUM_TABLES 100

/** maximum size of field fetched by odbc_fetch() */
#define MAX_COL_SIZE 10000

/** This structure is filled by odbc_parse()      */
typedef struct _DBI_DSP
{
 PSX_STR dsn; /**< data source name               */
 PSX_STR usr; /**< user name                      */
 PSX_STR pwd; /**< password                       */
} DBI_DSP;

/** for memory management of strings stored by odbc_fetch() :    */
typedef struct MemPoolNodeStruct
{
 char * ptr; /**< pointer to allocated memory block (string)     */
 struct MemPoolNodeStruct * next; /**< pointer to next node      */
} MemPoolNode;

/** DBI-data for ODBC driver (pointer to this structure exists   
 * in DBI.tag):
 */
typedef struct
{
 SQLHENV henv;       /**< ODBC Environment Handle                             */
 SQLHDBC hdbc;       /**< ODBC Connection Handle                              */
 MemPoolNode * pool; /**< allocated strings in concatenated list              */
 PSX_STK * stk;      /**< Stack with pushed (dbi_push()) statement handles    */
 SQLHSTMT hstmt;     /**< current ODBC statement handle                       */
} ODBC_CTX;

/******************************************************************************/

/* Prototypes:                                                                */
static BOOL odbc_execute (DBI * dbi, const char * sql);


/******************************************************************************/
/* poolinsert(ctx,ptr):
/** insert new memory block (string) at address ptr in concatenated list of
 *  all allocated memory blocks
 * @param ctx pointer to database context
 * @param ptr string pointer
 */
 
static BOOL poolinsert (ODBC_CTX * ctx,char * ptr)
{
 /* allocate node structure and initialize:          */
 MemPoolNode * n = malloc (sizeof (MemPoolNode));
 if(n == NULL) return (FALSE);
 n -> ptr = ptr;

 /* insert node at head of list:                     */
 n -> next = ctx -> pool;
 ctx -> pool = n;
 
 return (TRUE);                           /* success */
}

/*****************************************************************************/
/** free all allocated string memory blocks
 * @param ctx pointer to database context 
 */
 
static void freepool (ODBC_CTX * ctx)
{
 MemPoolNode * pool = ctx -> pool;                /* head of list  */

 while (pool != NULL)              /* while there are list entries */
 {
  /* store next entry :                                            */
  MemPoolNode * next;
  next = pool -> next;

  /* free memory block and node:                                   */
  free (pool -> ptr);
  free (pool);

  /* go on with next node:                                         */
  pool = next;
 }
 
 ctx -> pool = NULL;                         /* mark list as empty */
}

/******************************************************************************/
/** split dbi string in its parts and fill dbi_dsp structure with them
 * @param spc specification string to be splitted
 * @param dsp database specification structure
 */
 
static BOOL odbc_parse (PSX_STR spc,DBI_DSP * dsp)
{
 int i = 0;

 if (spc == NULL || dsp == NULL)
  return (FALSE);

 /* split string in DataSource-Name, User-Name & Password:        */
 if (!str_tok (spc,&i,":",dsp -> dsn))        /* data source name */
  return (FALSE);
 if (!str_tok (spc,&i,":",dsp -> usr))        /* user name        */
  return (FALSE);
 if (!str_tok (spc,&i,":",dsp -> pwd))        /* password         */
  strcpy (dsp -> pwd,"");

 return (TRUE); /* success */
}

/******************************************************************************/
/** put out detailed ODBC error message into log file
 * @param type type of handle hnd
 * @param hnd handle of type type
 * @param func string pointer to name of SQL..-function, in which the error happened
 */
 
static BOOL odbc_diag (SQLSMALLINT type,SQLHANDLE hnd,char * func)
{
 SQLCHAR state[6];
 SQLCHAR msg[256];
 SQLRETURN ret;
 PSX_STR s;

 sprintf (s,"ODBC error: %s failed!",func);
 LOG (MOD,s);

 ret = SQLGetDiagRec (type,hnd,1,state,NULL,msg,sizeof(msg),NULL);
 if (ret == SQL_SUCCESS || ret == SQL_SUCCESS_WITH_INFO)
 {
	 sprintf (s,"%s: %s: %s",func,state,msg);
  LOG (MOD,s);
  return (TRUE);
 }
 LOG (MOD,"Unknown ODBC error: SQLGetDiagRec() failed!\n");

 return (FALSE);
}

/*****************************************************************************/
/** dummy function: at this moment ODBC databases can not be created,
 *  because for that a database of an special DBMS has to exist and a new
 *  ODBC DataSource musst be applied. The user shall do this manually.
 * @param dbs string pointer to name of database
 */
 
static BOOL odbc_create (char * dbs)
{
 /* No-op. */
 return (TRUE);
}

/******************************************************************************/
/** delete all tables in the DB without deleting the database itself,
 *  also see comment to odbc_create().
 * @param dbs string pointer to anme of database
 */
 
static BOOL odbc_delete (char * dbs)
{
 DBI_DSP dsp;
 PSX_STR newdbs;
 DBI * dbi;
 ODBC_CTX * ctx;
 SQLRETURN ret;
 SQLHSTMT hstmt;
 SQLINTEGER ind;
 SQLCHAR tablename[1024];
 char * tables[MAX_NUM_TABLES];
 int i;
 int num_tables ;
 PSX_STR cmd;

 if (dbs == NULL)
  return (FALSE);

 /* split dbs-String in DSN, User-Name & Password:                           */
 if (!odbc_parse (dbs,&dsp))
  return (FALSE);
 
 /* manufacture new string for dbi_open() (prefix "odbc:" ):                 */
 sprintf(newdbs,"odbc:%s",dbs);

 /* open database:                                                           */
 if (!dbi_open(&dbi,newdbs))
  return (FALSE);

 /* fetch DBI_CTX pointer:                                                   */
 ctx = (ODBC_CTX *) dbi -> tag;

 /* allocate statement handle:                                               */
 ret = SQLAllocHandle(SQL_HANDLE_STMT,ctx -> hdbc,&hstmt);
 if (ret != SQL_SUCCESS && ret != SQL_SUCCESS_WITH_INFO)
 {
  dbi_close(&dbi);
  return (FALSE);
 }

 /* request list of tables:                                                  */
 ret = SQLTables (hstmt,NULL,0,NULL,0,NULL,0,"TABLE",SQL_NTS);
 if (ret != SQL_SUCCESS && ret != SQL_SUCCESS_WITH_INFO)
 {
  odbc_diag (SQL_HANDLE_STMT,hstmt,"SQLTables()");
  SQLFreeStmt(hstmt,SQL_CLOSE);
  dbi_close(&dbi);
  return (FALSE);
 }

 /* write tablenames (3. column) automaticly in tablename:                   */
 ret = SQLBindCol (hstmt,3,SQL_C_CHAR,tablename,sizeof(tablename),&ind);
 if (ret != SQL_SUCCESS && ret != SQL_SUCCESS_WITH_INFO)
 {
  odbc_diag (SQL_HANDLE_STMT,hstmt,"SQLBindCol()");
  SQLFreeStmt(hstmt,SQL_CLOSE);
  dbi_close(&dbi);
  return (FALSE);
 }
                   
 /* delete all tables individually:                                          */
 for (i = 0; i < MAX_NUM_TABLES; i++)  /* loop over all tables               */
 {
  ret = SQLFetch (hstmt); /* fetch new tablename (-> tablename, s.a.)        */
  if (ret == SQL_SUCCESS || ret == SQL_SUCCESS_WITH_INFO)
  {
   if(ind == SQL_NULL_DATA) /* empty entry...  */
    continue;               /* ...jump over    */

   /* else copy tablenames and store in array tables[i]:                     */
   tables[i] = malloc (ind + 1);
   strcpy (tables[i],tablename);
  }
  else if (ret == SQL_NO_DATA) /* Ende der Tabellenliste */
   break; /* break loop    */
  else    /* else: error   */
  {
   odbc_diag (SQL_HANDLE_STMT,hstmt,"SQLFetch()");
   SQLFreeStmt (hstmt,SQL_CLOSE);
   dbi_close (&dbi);
   return (FALSE);
  }
 }
 num_tables = i;

 /* statment handle is not necessary any more:                            */
 SQLFreeStmt (hstmt,SQL_CLOSE);

 /* try twice to delete all tables. Twice, because of dependencies between
    the tables which may cause failure while deleting for the first time. */
  for (i = 0; i < num_tables; i++)
 {
  sprintf(cmd,"DROP TABLE %s",tables[i]);  /* build SQL-Kommando */
  if(dbi_execute(dbi,cmd))                 /* execute            */
  {
   free (tables[i]);
   tables[i] = NULL;
  }
 }
 /* again the same loop (see comment above)                      */
 for (i = 0; i < num_tables; i++)
 {
  if (tables[i] == NULL)
   continue;

  sprintf(cmd,"DROP TABLE %s",tables[i]);
  free (tables[i]);
  dbi_execute(dbi,cmd);
 }
 
 /* close DB  */
 dbi_close(&dbi);

 return (TRUE); /* success */
}

/******************************************************************************/
/** open database connection
 * @param dbi pointer to database interface structure
 * @param dbs string pointer to name of database
 */
 
static BOOL odbc_open (DBI * dbi,char * dbs)
{
 SQLRETURN ret;
 SQLHENV henv;   // environment handle of database
 DBI_DSP dsp;    // database specification
 int i = 0;

 if (dbs == NULL)
  return (FALSE);

 /* split dbs string in DSN, User-Name & Password:                           */
 if (!odbc_parse(dbs,&dsp))
  return (FALSE);

 /* get environment handle:                                                  */
 ret = SQLAllocHandle (SQL_HANDLE_ENV,SQL_NULL_HANDLE,&henv);
 if (ret == SQL_SUCCESS || ret == SQL_SUCCESS_WITH_INFO)
 {
  /* announce the use of ODBC-API v3.0 :                                     */
  ret = SQLSetEnvAttr (henv,SQL_ATTR_ODBC_VERSION,(SQLPOINTER)SQL_OV_ODBC3,0);
  if (ret == SQL_SUCCESS || ret == SQL_SUCCESS_WITH_INFO)
  {
   SQLHDBC hdbc;

   /* get connection handle:                                                 */
   ret = SQLAllocHandle (SQL_HANDLE_DBC,henv,&hdbc);
   if (ret == SQL_SUCCESS || ret == SQL_SUCCESS_WITH_INFO)
   {
    /* connect to DB :                                                       */
    ret = SQLConnect (hdbc,dsp.dsn,SQL_NTS,dsp.usr,SQL_NTS,dsp.pwd,SQL_NTS);
    if (ret == SQL_SUCCESS || ret == SQL_SUCCESS_WITH_INFO)
    {
     /* allocate DBI_CTX structure:                                          */
     ODBC_CTX * ctx;
 
     ctx = malloc (sizeof (ODBC_CTX));
     if(ctx != NULL)
     {
      /* create stack:                                                       */
      if (stk_create (&ctx -> stk))
      {
       /* set size of stack element (= one statement handle):                */
       stk_set_esz (ctx -> stk,sizeof(SQLHSTMT));

       /* fill in values into DBI_CTX structure:                             */
       ctx -> henv = henv;
       ctx -> hdbc = hdbc;
       ctx -> hstmt = SQL_NULL_HANDLE;
       ctx -> pool = NULL;
       dbi -> tag = ctx;
       return (TRUE); /* success */
      }
     }
    }
    else odbc_diag (SQL_HANDLE_ENV,henv,"SQLConnect()");

    SQLFreeHandle (SQL_HANDLE_DBC,hdbc);
   }
   else odbc_diag (SQL_HANDLE_ENV,henv,"SQLAllocHandle()");
  }
  else odbc_diag (SQL_HANDLE_ENV,henv,"SQLSetEnvAttr()");

  SQLFreeHandle (SQL_HANDLE_ENV,henv);
 }

 /* error: */
 LOG (MOD,"open failed!");
 return (FALSE);
}

/******************************************************************************/
/** close database
 * @param dbi pointer to database interface structure
 */
 
static BOOL odbc_close (DBI * dbi)
{
 /* fetch DBI_CTX structure:                                                  */
 ODBC_CTX * ctx = (ODBC_CTX *) dbi -> tag;

 if (ctx == NULL)
  return (FALSE);

 /* free allocated memory:                                                    */
 freepool (ctx);

 /* free stack with statement handles:                                        */
 stk_delete (&ctx -> stk);

 /* disconnect connection to database:                                        */
 SQLDisconnect (ctx -> hdbc);
 
 /* free handles:                                                             */
 SQLFreeHandle (SQL_HANDLE_DBC,ctx -> hdbc);
 SQLFreeHandle (SQL_HANDLE_ENV,ctx -> henv);

 /* free ctx structure :                                                      */
 free (ctx);

 dbi -> tag = NULL; /* to make sure */

 return (TRUE); /* success */
}

/******************************************************************************/
/** push current statement handle to stack
 * @param dbi pointer to database interface structure
 */
 
static BOOL odbc_push (DBI * dbi)
{
 ODBC_CTX * ctx = (ODBC_CTX *) dbi -> tag;

 if (ctx == NULL)
  return (FALSE);

 /* push handle to stack:                                                   */
 if (!stk_push (ctx -> stk,&ctx -> hstmt))
  return (FALSE);

 /* mark current handle as non-valid:                                       */
 ctx -> hstmt = SQL_NULL_HANDLE;
 
 return (TRUE); /* success */
}

/******************************************************************************/
/** pop handle statement from stack and set as current handle
 * @param dbi pointer to database interface structure
 */
 
static BOOL odbc_pop (DBI * dbi)
{
 ODBC_CTX * ctx = (ODBC_CTX *) dbi -> tag;

 if (ctx == NULL)
  return (FALSE);

 /* pop handle from stack and apply as current handle:                        */
 if (!stk_pop (ctx -> stk,&ctx -> hstmt))
  return (FALSE);
 
 return (TRUE); /* success */
}

/******************************************************************************/
/** execute sql statement
 * @param dbi pointer to database interface structure
 * @param sql string pointer to SQL statement
 */
 
static BOOL odbc_execute (DBI * dbi,const char * sql)
{
 ODBC_CTX * ctx = (ODBC_CTX *) dbi -> tag;
 SQLHSTMT hstmt;
 SQLRETURN ret;
 PSX_STR semicolon;
 char * temp;

 if (ctx == NULL)
  return (FALSE);

 if (!(temp = (char *) malloc(strlen(sql)+2))) // allocate memory to hold sql and semicolon
	 return (FALSE);

 /* free current handle in case                                               */ 
 if (ctx -> hstmt != SQL_NULL_HANDLE)
 {
  SQLFreeStmt (ctx -> hstmt,SQL_CLOSE);
  ctx -> hstmt = 0;
 }

 /* get new statement handle                                                  */
 ret = SQLAllocHandle (SQL_HANDLE_STMT,ctx -> hdbc,&hstmt);
 if (ret == SQL_SUCCESS || ret == SQL_SUCCESS_WITH_INFO)
 {
  /* execute SQL statement                                                    */

  /* append semicolon to sql statement, if necessary :                        */
	 if(psx_cfg_get(PSX_CFG_SQL_SEM, semicolon)){      
	  if(strcmp(semicolon, "1") == 0){

		//strcat(temp, ";"); 
		sprintf(temp,"%s;",sql);
		ret = SQLExecDirect (hstmt,temp,SQL_NTS);
	  
	  } else 
		  ret = SQLExecDirect (hstmt,sql,SQL_NTS);

	 } else 
		 ret = SQLExecDirect (hstmt,sql,SQL_NTS);


  //ret = SQLExecDirect (hstmt,sql,SQL_NTS);

  if(ret == SQL_SUCCESS || ret == SQL_SUCCESS_WITH_INFO)
  {
   /* set current statement handle                                            */
   ctx -> hstmt = hstmt;
   free (temp);
   return (TRUE);
  }
  else
   odbc_diag (SQL_HANDLE_STMT,hstmt,"SQLExecDirect()");
 
  /* (only in case of failure!) set handle free:                               */
  SQLFreeStmt (hstmt,SQL_CLOSE);
 }
 else
  odbc_diag (SQL_HANDLE_ENV,ctx -> henv,"SQLAllocHandle()");
 
 /* log error message                                                         */
 ctx -> hstmt = SQL_NULL_HANDLE;
 LOG (MOD,"execute failed!");
 free (temp);
 return (FALSE);
}

/******************************************************************************/
/** fetch next data record if v == NULL or get field with index f(+1)
 * @param dbi pointerto databasse interface structure
 * @param f field index
 * @param v pointer to value string pointer
 */
static BOOL odbc_fetch (DBI * dbi,IDX f,char ** v)
{
 ODBC_CTX * ctx = (ODBC_CTX *) dbi -> tag;
 SQLRETURN ret;

 if (ctx == NULL)
  return (FALSE);

 /* does next handle exist ?                                            */
 if (ctx -> hstmt == SQL_NULL_HANDLE)
  return (FALSE);

 if(v == NULL) /* fetch next record  */
 {
  /* fetch: */
  ret = SQLFetch (ctx -> hstmt);
  if (ret == SQL_SUCCESS || ret == SQL_SUCCESS_WITH_INFO)
   return (TRUE);
  else if (ret == SQL_NO_DATA) /* no data available */
   return (FALSE);
  else /* else: error */
   odbc_diag (SQL_HANDLE_STMT,ctx -> hstmt,"SQLFetch()");
 }
 else                          /* fetch field with index f + 1         */
 {
  SQLINTEGER len = 0;
  char dummy[10];
  char * ptr;
                
  /* try to retrieve the size of one column : */
  ret = SQLColAttribute (ctx -> hstmt,(SQLUSMALLINT)(f + 1),SQL_DESC_DISPLAY_SIZE,NULL,0,NULL,&len);
  if (ret == SQL_SUCCESS || ret == SQL_SUCCESS_WITH_INFO)
  {
   /* ensure, values are not senseless: */
  if (len <= 0 || len > MAX_COL_SIZE)
        len = 10000;
 
   /*  allocate buffer: */
   if (ptr = malloc (len + 1))
   {
    SQLUINTEGER newlen;
                        
    /* fetch data and put on address ptr : */
    ret = SQLGetData (ctx -> hstmt,(SQLSMALLINT)(f + 1),SQL_C_CHAR,ptr,len + 1,&newlen);
    if (ret == SQL_SUCCESS || ret == SQL_SUCCESS_WITH_INFO)
    {
     if(newlen > 0) /* real size found => allocate new (smaller!) string */
     {
      if(*v = malloc (newlen + 1))
      {
       strncpy (*v,ptr,newlen + 1); /* copy string */
      free (ptr); /* old block is no longer used   */
      }
     }
     else
      *v = ptr; /* still use old memory block      */
 
     /* insert string into memory management pool: */    
     if (poolinsert (ctx,*v))
      return (TRUE);
    }
    else
     odbc_diag (SQL_HANDLE_STMT,ctx -> hstmt,"SQLGetData()");
                
    free (ptr);
   }
  }
  else
   odbc_diag (SQL_HANDLE_STMT,ctx -> hstmt,"SQLColAttribute()");
 }

 /* log failure: */
 LOG (MOD,"fetch failed!");
 return (FALSE);
}

/******************************************************************************/
/** dummy function, do we need it !?
 * @param dbi pointer to database interface structure
 * @param msg pointer to message string pointer
 */
 
static BOOL odbc_status (DBI * dbi,char ** msg)
{
 /* No-op. */
 return (FALSE); /* just tell about failure */
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/

/* Callbacks in DBI_DRV-Struktur eintragen: */

DBI_DRV dbi_drv_odbc =
{
 odbc_create,
 odbc_delete,
 NULL,
 odbc_open,
 odbc_close,
 odbc_push,
 odbc_pop,
 odbc_execute,
 odbc_fetch,
 odbc_status
};

#endif // HAVE_ODBC

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


