/******************************************************************************/
/* psx-dbi-pgs.c                                      DBI Driver - PostgreSQL */
/******************************************************************************/
/** @file psx-dbi-pgs.c Database Interface: PostgreSQL Driver - Source Code File
 * Functions and definitions implementing an interface for a PostgreSQL database.
 * This module provides functions for:
 * - opening the connection to a database
 * - parsing and splitting the connection string specified in the config file
 * - creation of a new database (unlike the ODBC Driver)
 * - deleting an old database
 * - executing a SQL statement
 * - fetching data from the database
 * - querying the status of the database (unlike ODBC Driver)
 * - closing the connection to database
 */
 
#include "psx.h"

#ifdef HAVE_PGSQL   /* is set by configure */

/******************************************************************************/
/*                                                                            */
/******************************************************************************/

//#include <postgresql/libpq-fe.h>

#include <libpq-fe.h>

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

/** query context
 */
 
typedef struct _DRV_QCX
{
 NUM t;             /**< number of tuples */
 NUM f;             /**< number of fields */
 IDX i;             /**< index            */
} DRV_QCX;

/** status
 */
typedef struct _DRV_STS
{
 PGresult * rsl;    /**< result        */
 DRV_QCX    qcx;    /**< query context */
} DRV_STS;

/** context
 */
typedef struct _DRV_CTX
{
 PGconn   * con;    /**< connection    */
 PGresult * rsl;    /**< result        */
 DRV_QCX    qcx;    /**< query context */
 PSX_STK  * stk;    /**< stack         */
} DRV_CTX;

// temporary in dbi.h

/** driver specification
 */
typedef struct _DBI_DSP
{
 PSX_STR hst;        /**< host          */
 PSX_STR prt;        /**< port          */
 PSX_STR dbs;        /**< database      */
 PSX_STR usr;        /**< user name     */
 PSX_STR pwd;        /**< password      */
} DBI_DSP;


#define	MOD	PSX_MOD_DBI_PGS

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/* notice processor                                                           */

static void drv_npr (void * arg,const char * msg)
{
 // fprintf (stderr,"> %s",msg);
}

/******************************************************************************/
/** parse and split connect specification string and fill dsp with correct values
 * @param spc string pointer to connect specification
 * @param dsp pointer to (empty) database specification
 */

static BOOL drv_parse (PSX_STR spc,DBI_DSP * dsp)
{
 int i,r;

 if (spc == NULL || dsp == NULL)
  return (FALSE);

 //r = sscanf (spc,"%s:%s:%s:%s:%s",dsp -> hst,dsp -> prt,dsp -> dbs,dsp -> usr,dsp -> pwd);

 i = 0;

 if (!str_tok (spc,&i,":",dsp -> hst))	return (FALSE); // host
 if (!str_tok (spc,&i,":",dsp -> prt))	return (FALSE); // port
 if (!str_tok (spc,&i,":",dsp -> dbs))	return (FALSE); // database
 if (!str_tok (spc,&i,":",dsp -> usr))	return (FALSE); // user
 if (!str_tok (spc,&i,":",dsp -> pwd))                  // password
  strcpy (dsp -> pwd,"");

 return (TRUE);
}

/******************************************************************************/
/** clear driver
 * @param dbi database interface structure
 */

static BOOL drv_clear (DBI * dbi)
{
 DRV_CTX * ctx;

 if (dbi == NULL)
  return (FALSE);

 ctx = (DRV_CTX *) dbi -> tag;

 if (ctx == NULL)
  return (FALSE);

 if (ctx -> rsl == NULL)
  return (FALSE);

 PQclear (ctx -> rsl);
 ctx -> rsl = NULL;

 return (TRUE);
}

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** create database
 * @param dbs string pointer to name of database
 */

static BOOL drv_create (char * dbs)
{
 DBI * dbi;
 BOOL r = TRUE;
 IDX i;
 SYS_STR sql,t;
 DBI_DSP dsp;

 if (dbs == NULL)
  return (FALSE);

 if (!drv_parse (dbs,&dsp))
  return (FALSE);

 sprintf (sql,"CREATE DATABASE %s",dsp.dbs);

 sprintf (t,"pgs:%s:%s:%s:%s:%s",dsp.hst,dsp.prt,"template1",dsp.usr,dsp.pwd);

 if (!dbi_open (&dbi,t))
 {
  printf ("dbi: cannot open template1\n");
  return (FALSE);
 }

 if (!dbi_execute (dbi,sql))
  r = FALSE;

 dbi_close (&dbi);

 return (r);
}

/******************************************************************************/
/** drop database
 * @param dbs string pointer to name of database
 */

static BOOL drv_delete (char * dbs)
{
 DBI * dbi;
 BOOL r = TRUE;
 IDX i;
 SYS_STR sql,t;
 DBI_DSP dsp;

 if (dbs == NULL)
  return (FALSE);

 if (!drv_parse (dbs,&dsp))
  return (FALSE);

 sprintf (sql,"DROP DATABASE %s",dsp.dbs);

 sprintf (t,"pgs:%s:%s:%s:%s:%s",dsp.hst,dsp.prt,"template1",dsp.usr,dsp.pwd);

 if (!dbi_open (&dbi,t))
  return (FALSE);

 if (!dbi_execute (dbi,sql))
  r = FALSE;

 dbi_close (&dbi);

#ifndef WIN32
 sleep (1);
#endif
 return (r);


}

/******************************************************************************/
/** open Postgres database
 * @param dbi pointer to (empty) database interface
 * @param dbs string pointer to database specification
 */

static BOOL drv_open (DBI * dbi,char * dbs)
{
 ConnStatusType c;
 char           t [1000];
 DBI_DSP dsp;
 DRV_CTX * ctx;

 if (dbi == NULL || dbs == NULL)
  return (FALSE);

 assert (dbi -> tag == NULL);

 if ((ctx = (DRV_CTX * ) malloc (sizeof (DRV_CTX))) == NULL)

  return (FALSE);

 memset (ctx,0,sizeof (*ctx));

 if (!drv_parse (dbs,&dsp))   // split database specification string
 {
  free (ctx);
  return (FALSE);
 }

 // build connect string:
 sprintf (t,"host='%s' dbname='%s' user='%s' password='%s' ",dsp.hst,dsp.dbs,dsp.usr,dsp.pwd); //JM: added pwd 22.07.05

 ctx -> con = PQconnectdb (t);

 if (ctx -> con == NULL)
 {
  free (ctx);
  return (FALSE);
 }

 // PQuntrace ((*ctx) -> ptr -> con);

 PQsetNoticeProcessor (ctx -> con,drv_npr,NULL);

 c = PQstatus (ctx -> con);

 if (c != CONNECTION_OK)
 {
  PQfinish (ctx -> con);
  ctx -> con = NULL;
  free (ctx);
  return (FALSE);
 }

 if (!stk_create (&ctx -> stk))
 {
  PQfinish (ctx -> con);
  free (ctx);
  return (FALSE);
 }
 
 stk_set_esz (ctx -> stk, sizeof(DRV_STS));

 dbi -> tag = ctx;

 return (TRUE);
}

/******************************************************************************/
/** close database interface
 * @param dbi pointer to database interface structure
 */

static BOOL drv_close (DBI * dbi)
{

 DRV_CTX * ctx;

 if (dbi == NULL)
  return (FALSE);

 ctx = (DRV_CTX *) dbi -> tag;

 if (ctx -> con == NULL)
  return (FALSE);

 if (ctx -> rsl)
 {
  // PQclear (ctx -> rsl);
  drv_clear (dbi);
  ctx -> rsl = NULL;
 }

 PQfinish (ctx -> con);
 ctx -> con = NULL;

 if (ctx -> stk != NULL)
  stk_delete (&ctx -> stk);

 return (TRUE);
}

/******************************************************************************/
/** push dbi tag on stack
 * @param dbi pointer to database interface structure
 */

static BOOL drv_push (DBI * dbi)
{
 DRV_CTX * ctx;
 DRV_STS stn;
 BOOL r;

 if (dbi == NULL)
  return (FALSE);

 ctx = dbi -> tag;
 if (ctx == NULL)
  return (FALSE);

 stn.rsl = ctx -> rsl;      /* result         */
 stn.qcx = ctx -> qcx;      /* query context  */

 r = stk_push (ctx -> stk,&stn);

 ctx -> rsl = NULL;
 memset (&ctx -> qcx,0,sizeof (DRV_QCX));

 return (r);
}

/******************************************************************************/


/** pop stn from ctx stack
 * @param dbi pointer to database interface structure
 */



 
static BOOL drv_pop (DBI * dbi)
{
 DRV_CTX * ctx;
 DRV_STS stn;

 if (dbi == NULL)
  return (FALSE);

 ctx = dbi -> tag;
 if (ctx == NULL)
  return (FALSE);

 if (!stk_pop (ctx -> stk,&stn))       
  return (FALSE);

 ctx -> rsl = stn.rsl;
 ctx -> qcx = stn.qcx;

 return (TRUE);
}

/******************************************************************************/
/** get number of tuples and store in t
 * @param dbi pointer to database interface structure
 * @param t number of retrieved tuples
 */

static BOOL drv_get_tuples (DBI * dbi,NUM * t)
{
 DRV_CTX * ctx;

 if (dbi == NULL)
  return (FALSE);

 ctx = dbi -> tag;
 if (ctx == NULL)
  return (FALSE);

 *t = ctx -> qcx.t;
 return (TRUE);
}

/******************************************************************************/
/** get number of fields and store in f
 * @param dbi pointer to database interface structure
 */

static BOOL drv_get_fields (DBI * dbi,NUM * f)
{
 DRV_CTX * ctx;

 if (dbi == NULL)
  return (FALSE);

 ctx = dbi -> tag;
 if (ctx == NULL)
  return (FALSE);

 *f = ctx -> qcx.f;
 return (TRUE);
}

/******************************************************************************/
/** execute sql statement sql
 * @param dbi pointer to database interface
 * @param sql string pointer to sql statement
 */

static BOOL drv_execute (DBI * dbi,const char * sql)
{
 DRV_CTX * ctx;
 ExecStatusType s;
 char * temp;
 
 if (dbi == NULL || sql == NULL)
  return (FALSE);

 ctx = (DRV_CTX *) dbi -> tag;

 if (ctx == NULL)
  return (FALSE);

 if (ctx -> con == NULL)
  return (FALSE);

 drv_clear (dbi);

 dbi -> oid = 0xFFFFFFFF;

 if (!(temp = (char *) malloc(strlen(sql)+2))) // allocate memory to hold sql and semicolon
	return (FALSE);


 if(!(strncmp(sql, "BEGIN",5) == 0)){ 
    sprintf(temp, "%s;",sql);  // append semicolon to sql statement string
	//strcat(sql,";");   // may cause error: writing to string constant
    ctx -> rsl = PQexec (ctx -> con,temp);  // execute sql statement
 }
 else
    ctx -> rsl = PQexec (ctx -> con,sql);  // execute sql statement

 if (ctx -> rsl == NULL)
 {
  LOG (MOD,"no results found");
  free (temp);
  return (FALSE);
 }

 s = PQresultStatus (ctx -> rsl); // get result status

 switch (s)
 {
  case PGRES_TUPLES_OK:
   LOG (MOD,"result: tuples ok");
   break;
                               
  case PGRES_COMMAND_OK:
   LOG (MOD,"result: command ok");
   break; 

  default:
  {
   char * t = PQerrorMessage (ctx -> con);
   str_rem_chr_end (t,'\n');
   LOG (MOD,"result: error:\n%s",t);
   dbi_sts_print (dbi);
   free (temp);
   return (FALSE);
  }
 }

 dbi -> oid = PQoidValue (ctx -> rsl);   /* object identifier   */
 ctx -> qcx.t = PQntuples (ctx -> rsl);  /* number of tuples    */
 ctx -> qcx.f = PQnfields (ctx -> rsl);  /* number of fields    */
 ctx -> qcx.i = IDX_NIL;                 /* index               */
free (temp);

 return (TRUE);
}

/******************************************************************************/
/** fetch result value of field f from database and store in v; increment current row
 * @param dbi pointer to database interface
 * @param f index of field
 * @param v string pointer to (empty) value, contains value after invocation
 */

static BOOL drv_fetch (DBI * dbi,IDX f,char ** v)
{
 DRV_CTX * ctx;

 if (dbi == NULL)
  return (FALSE);

 if ((ctx = (DRV_CTX *) dbi -> tag) == NULL)
  return (FALSE);

 if (ctx -> rsl == NULL)
  return (FALSE);

 if (v == NULL)
 {                                      
  if (ctx -> qcx.i == IDX_NIL)
   ctx -> qcx.i = 0;                    /* first row */
  else
   if (ctx -> qcx.i < ctx -> qcx.t)     /* next row */
    ctx -> qcx.i++;

  if (ctx -> qcx.i >= ctx -> qcx.t)
   return (FALSE);

  return (TRUE);   /* why return?  v not set yet! */
 }

 *v = PQgetvalue (ctx -> rsl,ctx -> qcx.i,f);   /* get value of result rsl in row i and field f */

 if (*v == NULL)
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** get status
 * @param dbi pointer to database interface structure
 * @param msg pointer to pointer to message string
 */

static BOOL drv_status (DBI * dbi,char ** msg)
{
 static char stg [1000];
 ExecStatusType s;
 char * t;
 DRV_CTX * ctx;

 if (dbi == NULL || msg == NULL)
  return (FALSE);

 *msg = NULL;

 if ((ctx = (DRV_CTX *) dbi -> tag) == NULL)
  return (FALSE);

 if (ctx -> con == NULL)
  return (FALSE);

 if (ctx -> rsl == NULL || TRUE)   // to be revised
 {
  char * t;

  t = PQerrorMessage (ctx -> con);
  strcpy (stg,t);
  str_rem_chr_end (stg,'\n');
  *msg = stg;
  return (TRUE);

 }

 s = PQresultStatus (ctx -> rsl);
 t = PQresStatus (s);

 return (TRUE);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/

/** database driver for Postgres database
 */
 
DBI_DRV dbi_drv_pgs =
{


 drv_create,      /** < pointer to function drv_create (char * dbs) */
 drv_delete,      /** < pointer to function drv_delete (char * dbs) */
 NULL,
 drv_open,
 drv_close,
 drv_push,
 drv_pop,
 drv_execute,
 drv_fetch,
 drv_status
};


/******************************************************************************/
/******************************************************************************/
/******************************************************************************/

#endif // HAVE_PGSQL


