/******************************************************************************/
/* psx-dbi.c                                             Database Interface   */
/******************************************************************************/
/** @file psx-dbs.c Database Interface - Source Code File
 * Functions providing the functionalities of a database interface hiding the
 * identity of the database driver (PostgreSQL or ODBC) to the main programme.
 * This module features the following funcionalities:
 * - creation of a new database (only works with PostgreSQL Driver)
 * - opening the connection to a database
 * - closing the connection to the database
 * - execution of a SQL statement (like SELECT, INSERT etc.)
 * - fetching the data retrieved by the execution of query statement
 * - escaping special characters like ' and \
 * - deleting an old database
 * - querying the status of the database (only works with PostgreSQL Driver)
 */

#include "psx.h"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

#define	MOD	PSX_MOD_DBI

#ifdef HAVE_PGSQL     
#  include "psx-dbi-pgs.h"
#endif

#ifdef HAVE_ODBC
#  include "psx-dbi-odbc.h"
#endif

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

#define DRV_ZERO  0x00
#define DRV_PGS   0x01
#define DRV_ODBC  0x02

static const struct
{
 BYTE      dri;
 const char    * sym;
 DBI_DRV * drv;
} dbi_dlt [] =
{
#ifdef HAVE_PGSQL  // BG bei Borland Test auskommentiert!
 { DRV_PGS, "pgs", &dbi_drv_pgs },
#endif
#ifdef HAVE_ODBC       // defined in win.h
 { DRV_ODBC,"odbc",&dbi_drv_odbc }
#endif
};

static BYTE dbi_dri = DRV_ZERO;

/******************************************************************************/


//static DBI_DRV * drv = &dbi_drv_pgs;

// temp:
/** Database specification.
 * The database specification string from the configuration file is read 
 * into this structure.
 */ 
typedef struct _DBI_DSP
{
 PSX_STR hst;				/**< Host string */
 PSX_STR prt;				/**< Port number */
 PSX_STR dbs;				/**< Database name */
 PSX_STR usr;				/**< User name* /
 PSX_STR pwd;				/**< Password */
} DBI_DSP;

/******************************************************************************/

/******************************************************************************/
/** select driver
 * @param dbs database name
 * @param dri
 * @param drs
 * @param drv database driver
 * @return TRUE if driver was retrieved
 * @return FALSE if driver could not be retrieved 
 */

static BOOL dbi_drv_sel (const PSX_STR dbs,BYTE * dri,PSX_STR drs,DBI_DRV ** drv)
{
 PSX_STR t;
 char *  p;
 int     i;

 if (dbs == NULL)
  return (FALSE);

 i = 0;

 if (!str_tok (dbs,&i,":",t))   // split first part of string to t
  return (FALSE);

 p = dbs + i;                   // second part of string dbs

 for (i = 0;i < sizeof (dbi_dlt) / sizeof (dbi_dlt [0]);i++)
  if (strcmp (t,dbi_dlt [i].sym) == 0)
  {
   if (dri)
    *dri = dbi_dlt [i].dri;     // get dri (PSGRS||ODBC)

   if (drv)
    *drv = dbi_dlt [i].drv;     // get driver

   strcpy (drs,p);              // get second part
   dbi_dri = dbi_dlt [i].dri;   // dbi_dri = kind of dbs

   return (TRUE);
  }

 return (FALSE);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** create database interface
 * @param dbs name of database
 */

BOOL dbi_create (const char * dbs)
{
 PSX_STR drs;
 BYTE dri;
 BOOL r;
 DBI_DRV * drv;

 if (!dbi_drv_sel (dbs,&dri,drs,&drv))   // select database driver
 {
  LOG (MOD,"invalid database driver");
  psx_put ("invalid database driver");
  return (FALSE);
 }

 r = drv -> create (drs);

 return (r);
}

/******************************************************************************/

/** delete database interface
 * @param name of database
 */

BOOL dbi_delete (const char * dbs)
{
 PSX_STR drs;
 BYTE dri;
 BOOL r;
 DBI_DRV * drv;

 if (!dbi_drv_sel (dbs,&dri,drs,&drv))  // select database driver
 {
  LOG (MOD,"invalid database driver");
  psx_put ("invalid database driver");
  return (FALSE);

 }

 r = drv -> delete (drs);

 return (r);
}

/******************************************************************************/
/** test function
 */

BOOL dbi_test (const char * dbs)
{
 DBI * dbi;

 if (!dbi_open (&dbi,dbs))
  return (FALSE);

 dbi_close (&dbi);

 return (TRUE);
}

/******************************************************************************/
/** open database
 * @param dbi database interface structure
 * @param dbs database
 */ 

BOOL dbi_open (DBI ** dbi,const char * dbs)
{
 PSX_STR drs;
 BOOL r;

 if (dbi == NULL || dbs == NULL)
  return (FALSE);

 if ((*dbi = (DBI *) malloc (sizeof (**dbi))) == NULL)
  return (FALSE);

 memset (*dbi,0,sizeof (**dbi));

 if (!dbi_drv_sel (dbs,NULL,drs,&(*dbi) -> drv))  /* get correct driver*/
 {
  LOG (MOD,"invalid database driver");
  psx_put ("invalid database driver");
  free (*dbi);
  return (FALSE);
 }

 r = (*dbi) -> drv -> open (*dbi,drs);    /* call respective open function of selected driver */

 if (!r)
 {
  free (*dbi);
  *dbi = NULL;
  return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/** close database
 * @param dbi pointer to database interface structure
 */

BOOL dbi_close (DBI ** dbi)
{
 BOOL r;

 if (dbi == NULL)
  return (FALSE);

 if (*dbi == NULL)
  return (FALSE);

 r = (*dbi) -> drv -> close (*dbi);

 free (*dbi);
 *dbi = NULL;


 return (r);
}

/******************************************************************************/
/** push dbi on stack
 * @param dbi database interface structure
 */

BOOL dbi_push (DBI * dbi)
{
 if (dbi == NULL)
  return (FALSE);

 psx_log_m (MOD,"push");

 return dbi -> drv -> push (dbi);
}

/******************************************************************************/
/** pop dbi from stack
 * @param dbi database interface structure
 */

BOOL dbi_pop (DBI * dbi)
{
 if (dbi == NULL)
  return (FALSE);

 psx_log_m (MOD,"pop");


 return dbi -> drv -> pop (dbi);
}


/******************************************************************************/
/* get the number of fields                                                   */

BOOL dbi_get_fields (DBI * dbi,NUM * f)
{
 char * v;
 if (dbi == NULL)
  return (FALSE);
 if (!dbi_select(dbi,"information_schema.columns","count(*)","table_name='rec'"))
  return (FALSE);
 if (!dbi_fetch(dbi,0,0))
  return (FALSE); 
 if (!dbi_fetch(dbi,0,&v))
  return (FALSE); 
 *f = atoi (v);
 return (TRUE);
}

/******************************************************************************/
/** calls execute function of corresponding driver
 * @param dbi database interface structure
 * @param sql SQL statement
 */

BOOL dbi_execute (DBI * dbi,const char * sql)
{
 BOOL r;

 if (dbi == NULL || sql == NULL)
  return (FALSE);

 psx_put_v ("dbi: %s",sql);
 psx_log_m (MOD,"execute: %s",sql);

 
 r = dbi -> drv -> execute (dbi,sql); 

 return (r);
}

/******************************************************************************/
/** calls fetch function of corresponding database driver
 * and stores fetched value in v
 * @param dbi database interface structure
 * @param f fetch index
 * @param v pointer to value string
 */

BOOL dbi_fetch (DBI * dbi,IDX f,char ** v)
{
 BOOL r;

 if (dbi == NULL)
  return (FALSE);

 r = dbi -> drv -> fetch (dbi,f,v);

 return (r);
}

/******************************************************************************/
/** prints status
 * @param dbi database interface structure
 */

BOOL dbi_sts_print (DBI * dbi)
{
 BOOL r;
 char * t;

 if (dbi == NULL)
  return (FALSE);

 r = dbi -> drv -> status (dbi,&t);

 if (!r)
  strcpy(t,"unknown status");

 psx_put ("dbi: %s",t);

 return (TRUE);
}

/******************************************************************************/
/** select max atr from src, n is maximum size of column + 1
 * increment maximal value of attribut atr (= index) of table src and store in n
 * @param ctx pointer to database context
 * @param src string pointer to table name
 * @param atr string pointer to column names
 * @param n pointer to (empty) new maximal value of column atr
*/

BOOL dbi_idx_inc (DBI * ctx,const char * src,const char * atr,NUM * n)
{
 DBI_STR sql;
 char * t;

 if (ctx == NULL || src == NULL || atr == NULL || n == NULL)
  return (FALSE);

 *n = 0xFFFFFFFF;
 sprintf (sql,"SELECT MAX(%s) FROM %s",atr,src);

 if (!dbi_execute (ctx,sql))  // get max value of column atr in table src
  return (FALSE);

 if (!dbi_fetch (ctx,0,NULL)) // fetch next record
 {
  *n = 1;                     // no record found
  return (TRUE);
 }

 if (!dbi_fetch (ctx,0,&t))   // fetch first record
  return (FALSE);

 if (t == NULL)
  return (FALSE);

 *n = atoi (t) + 1;           // convert returned number and increment

 return (TRUE);
}

/******************************************************************************/
/** initiate generation and execution of sql SELECT statement and fetch the
 *  retrieved values
 * @param ctx dbi content structure
 * @param src name string of source table(s)
 * @param atr string pointer to list of column names
 * @param cnd string pointer to condition string
 * @param v string pointer, contains result string after invocation
 */

BOOL dbi_lookup (DBI * ctx,const char * src,const char * atr,const char * cnd,char ** v)
{
 if (ctx == NULL || src == NULL || atr == NULL || cnd == NULL || v == NULL)
  return (FALSE);

 if (!dbi_select (ctx,src,atr,cnd)) // SELECT atr FROM src WHERE cnd
  return (FALSE);

 if (!dbi_fetch (ctx,0,NULL))
  return (FALSE);

 if (!dbi_fetch (ctx,0,v))
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** generate SQL statement: select value atr from table src under condition cnd
 *  and execute statement
 * @param ctx dbi structure context

 * @param src source table name
 * @param atr string pointer to list of column names 
 * @param cnd condition string
 */

BOOL dbi_select (DBI * ctx,const char * src,const char * atr,const char * cnd)
{
 DBI_STR s,a,c,sql;


 if (ctx == NULL || src == NULL)
  return (FALSE);

 sprintf (s,"%s",src);                         // build statement
 sprintf (a,"%s",atr ? atr : "*");
 sprintf (sql,"SELECT %s FROM %s",a,s);

 if (cnd)
 {
  DBI_STR t;

  sprintf (t," WHERE (%s)",cnd);
  strcat (sql,t);
 }

 //strcat (sql,";");

 if (!dbi_execute (ctx,sql))                   // execute statement
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** execute sql INSERT statement
 * @param ctx pointer to database interface struct
 * @param src string pointer to table name
 * @param atr string pointer to list of column names
 * @param val string pointer to list of values to be inserted
 */

BOOL dbi_insert (DBI * ctx,const char * src,const char * atr,const char * val)
{
 DBI_STR sql;

 if (ctx == NULL || src == NULL || atr == NULL || val == NULL)
  return (FALSE);


 sprintf (sql,"INSERT INTO %s (%s) VALUES (%s)",src,atr,val);

 if (!dbi_execute (ctx,sql))
  return (FALSE);

 return (TRUE);
}


/******************************************************************************/
/** execute update statement
 * @param ctx pointer to database interface struct
 * @param src string pointer to table name
 * @param cnd string pointer to condition (WHERE clause)
 * @param atr string pointer to column names
 * @param val string pointer to values to be inserted 
 */

BOOL dbi_update (DBI * ctx,const char * src,const char * cnd,const char * atr,const char * val)
{
 DBI_STR sql;

 if (ctx == NULL || src == NULL || cnd == NULL || atr == NULL || val == NULL)
  return (FALSE);


 sprintf (sql,"UPDATE %s SET %s=%s WHERE (%s)",src,atr,val,cnd);

 if (!dbi_execute (ctx,sql))
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** convert string, escape ' depending on dbi driver
 * @param src string pointer to source string
 * @param string pointer to destination
 */

BOOL dbi_val_cvt (const char * src,char * dst)
{
 int n,l,i,j;
 char * tmp;

 if (src == NULL || dst == NULL)
  return (FALSE);

 l = strlen (src);
 n = 0;

 for (i = 0;i < l;i++)
  if (src [i] == '\'')                // get number of symbols for conversion 
   n++;

 if ((tmp = (char *) malloc ((l + n + 1) * sizeof (char))) == NULL)
  return (FALSE);

 j = 0;

 for (i = 0;i < l;i++)
  switch (src [i])
  {
   case '\'':                         //  '
    switch (dbi_dri)
    {
     case DRV_PGS:                    // convert '  into \'
      tmp [j++] = '\\';               // insert backslash
      tmp [j++] = '\'';               // insert apostrophe
      break;

     case DRV_ODBC:                   // convert ' into ''

      tmp [j++] = '\'';               // insert apostrophe
      tmp [j++] = '\'';               // insert apostrophe
      break;
    }
    break;

   default:
    tmp [j++] = src [i];             // just copy the rest
    break;
  }

 tmp [j] = 0;

 strcpy (dst,tmp);
 free (tmp);

 return (TRUE);
}

/******************************************************************************/
/** generate sql expression "src IS NULL"
 * @param src source string
 * @param exp expression string will be ISNULL or IS NULL, ??? if an error occured
 */

BOOL dbi_exp_isnull (const PSX_STR src,PSX_STR exp)
{
	PSX_STR sql;

 if (src == NULL || exp == NULL)
  return (FALSE);

 switch (dbi_dri)
 {
  case DRV_PGS:
   sprintf (exp,"%s ISNULL",src);
   break;

  case DRV_ODBC:

   if(!psx_cfg_get(PSX_CFG_SQL_NULL, sql)) // get configuration for IS NULL statement
	   strcpy(sql, "IS NULL");

   sprintf (exp, "%s %s",src,sql);
   //sprintf (exp,"%s IS NULL",src);
   break;
   
  default:
   sprintf (exp,"???");
   break;
 }

 return (TRUE);
}


/******************************************************************************/
/******************************************************************************/
/******************************************************************************/








