/******************************************************************************/
/* psx-dic.c                                 Dictionary Management Interface  */
/******************************************************************************/
/** @file psx-dic.c Dictionary Management Interface - Source Code File
 * Functions providing the functionality of a dictionary for storing and
 * retrieving pairs of attributes and values.
 */


#include "psx.h"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** append item src to dictionary string dic->str
 * @param dic dictionary
 * @param src source attribute name or value
 * @param idx index of newly added item in dictionary string
 */
 
static BOOL dic_str_add (PSX_DIC * dic,const char * src,IDX * idx)
{
 char * t;
 NUM k;

 if (dic == NULL || src == NULL || idx == NULL)
  return (FALSE);

 k = strlen (src) + 1;

 // t = dictionary string of size s+k
 if ((t = (char *) realloc (dic -> stg,(dic -> s + k) * sizeof (char))) == NULL)
  return (FALSE);

 memcpy (t + dic -> s,src,k);  // append src to t
 *idx = dic -> s;

 dic -> stg = t;  // new dictionary string = t
 dic -> s += k;   // increment dictionary size

 return (TRUE);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** allocate memory for dic Instance
 * @param dic pointer to dic structure
 */
BOOL dic_create (PSX_DIC ** dic)
{
 if (dic == NULL)
  return (FALSE);

 if ((*dic = (PSX_DIC *) malloc (sizeof (**dic))) == NULL)
  return (FALSE);

 memset (*dic,0,sizeof (**dic));

 return (TRUE);
}

/******************************************************************************/
/** free allocated memory
 * @param dic pointer to dictionary structure
 */
 
BOOL dic_delete (PSX_DIC ** dic)
{
 if (dic == NULL)
  return (FALSE);

 dic_clear (*dic);

 free (*dic);
 *dic = NULL;

 return (TRUE);
}

/******************************************************************************/
/** clear String and Item
 * @param dictionary structure
 */

BOOL dic_clear (PSX_DIC * dic)
{
 if (dic == NULL)
  return (FALSE);

 if (dic -> n > 0)
 {
  int i;

  assert (dic -> itm != NULL);
  free (dic -> itm);
  dic -> itm = NULL;
 }

 assert (dic -> itm == NULL);

 dic -> n = 0;

 if (dic -> stg)
 {
  free (dic -> stg);
  dic -> stg = NULL;
  dic -> s = 0;
 }

 return (TRUE);
}

/******************************************************************************/
/** set value and attribute
 * @param dic target dictionary structure
 * @param atr attribute source string
 * @param val value source string
 */
 
BOOL dic_put (PSX_DIC * dic,const char * atr,const char * val)
{
 PSX_DIC_ITM * t;
 int k;
 IDX a,v;

 if (dic == NULL || atr == NULL || val == NULL)
  return (FALSE);

 if (!dic_str_add (dic,atr,&a))   // add attribute name
  return (FALSE);

 if (!dic_str_add (dic,val,&v))   // add value
  return (FALSE);

 k = dic -> n;                    // get number of entries

 if ((t = (PSX_DIC_ITM *) realloc (dic -> itm,(k + 1) * sizeof (PSX_DIC_ITM))) == NULL)
  return (FALSE);

 dic -> itm = t;


 dic -> itm [k].atr = a;          // set attribute
 dic -> itm [k].val = v;          // set value

 dic -> n++;

//dic_print (dic);                  // print dictionary

 return (TRUE);
}

/******************************************************************************/
/** lookup value by attribute, can also be used to get an empty object
 * @param dic dictionary structure
 * @param atr attribute string
 * @param pointer to value string
 */
 
BOOL dic_get (PSX_DIC * dic,const char * atr,char ** val)
{
 IDX i;

 if (dic == NULL || atr == NULL)
  return (FALSE);

 for (i = 0;i < dic -> n;i++)
  if (strcmp (dic -> stg + dic -> itm [i].atr,atr) == 0)
  {

   if (val)
    *val = dic -> stg + dic -> itm [i].val;  // get value
   return (TRUE);
  }

 return (FALSE);
}

/******************************************************************************/
/** lookup attribute by item index
 * @param dic dictionary structure
 * @param idx item index
 * @param atr pointer to attribute string
 */
 
BOOL dic_get_atr (PSX_DIC * dic,IDX idx,char ** atr)
{
 if (dic == NULL)
  return (FALSE);

 if (idx >= dic -> n)
  return (FALSE);

 if (atr)
  *atr = dic -> stg + dic -> itm [idx].atr;

 return (TRUE);
}

/******************************************************************************/
/** lookup value by index
 * @param dic dictionary structure
 * @param idx index
 * @param val pointer to target value string
 */
 
BOOL dic_get_val (PSX_DIC * dic,IDX idx,char ** val)
{
 if (dic == NULL)
  return (FALSE);

 if (idx >= dic -> n)
  return (FALSE);

 if (val)
  *val = dic -> stg + dic -> itm [idx].val;

 return (TRUE);
}

/******************************************************************************/
/** append new dic item
 * @param dic target dictionary structure
 * @param atr source attribute string
 * @param val source value string
 */

BOOL dic_put_new (PSX_DIC * dic,const char * atr,const char * val)
{
 if (dic == NULL || atr == NULL || val == NULL)
  return (FALSE);

 if (dic_get (dic,atr,NULL))   // get empty dic 
  return (FALSE);

 if (!dic_put (dic,atr,val))   // set attribute and value
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** copy src to dic
 * @param dic target dictionary structure
 * @param src source dictionary structure
 */
 
BOOL dic_copy (PSX_DIC * dic,PSX_DIC * src)
{
 IDX i;

 if (dic == NULL || src == NULL)
  return (FALSE);

 dic_clear (dic);

 for (i = 0;i < dic -> n;i++)
  if (!dic_put (dic, dic -> stg + dic -> itm [i].atr, dic -> stg + dic -> itm [i].val))
   return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** print dic structure
 * @param dic dictionary structure
 */
 
BOOL dic_print (PSX_DIC * dic)
{
 IDX i;

 if (dic == NULL)
  return (FALSE);

 psx_print ("DIC [%p]\n",dic);

 for (i = 0;i < dic -> n;i++)
  psx_print ("[%2d]: '%s' -> '%s'\n",i,dic -> stg + dic -> itm [i].atr,dic -> stg + dic -> itm [i].val);

 psx_print ("\n");

 return (TRUE);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


