/******************************************************************************/
/* psx-doc.c                                           Document Interface     */
/******************************************************************************/
/** @file psx-doc.c Document Interface - Source Code File
 * Functions implementing the datastructure of a document based on a string.
 * This module is used for the creation of html documents by the cgi referring
 * modules.
 * This module provides the following functionalities:
 * - creatiion of a new document
 * - deleting an old document
 * - loading an existing document
 * - assignment of a string to the document
 * - generation of a html document using the memory block module (blk)
 */

#include "psx.h"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** retrieve the size of a file
 * @param f file pointer to the file with unknown length
 */
 
static unsigned long int getFileSize (FILE * f)
{
 fpos_t pos;
 long r;

 fgetpos (f,&pos);
 fseek (f,0,SEEK_END);
 r = ftell (f);
 fsetpos (f,&pos);

 return (r);
}

/******************************************************************************/
/** map document txt, lookup in dictionary
 * @param blk pointer to memory block structure
 * @param txt pointer to string, input text (html code)
 * @param dic pointer to dictionary structure
 */
 
static BOOL doc_map (PSX_BLK * blk,char * txt,PSX_DIC * dic)
{
 IDX i;

 if (blk == NULL || txt == NULL || dic == NULL)
  return (FALSE);

 i = 0;

 while (txt [i] != '\0')		// search document for variables                       
 {
  IDX j;
  int max_len = 0;
  int max_idx = IDX_NIL;

  if (txt [i] != '$')				// $ indicates a variable
  {
   if (!blk_put (blk,txt + i,1))       // put txt from i on into block blk
    return (FALSE);

   i++;
   continue;
  }

  i++;

	/* loops through dictionary an checks for each variable if it is found
	   at the current text position */
  for (j = 0;j < dic -> n;j++)         // go through dictionary
  {
   char * t;
   int k;

   if (!dic_get_atr (dic,j,&t))        // get attribute nr. j from dictionary
    continue;

   if (t == NULL)
    continue;

   if ((k = strlen (t)) == 0)
    continue;

   if (k <= max_len)
    continue;

   if (strncmp (txt + i,t,k) != 0)     // check correctness of attribute
    continue;

   max_len = k;												// stores length of matching variable name
   max_idx = j;												// stores index of matching variable
  }
	/* j now holds the index of the last matching variable or IDX_NIL if
	 * there was no match.
	 */	 
  if (max_idx != IDX_NIL)
  {
   char * t;

   i += max_len;

   if (!dic_get_val (dic,max_idx,&t))  // get value from dictionary
    break;

   if (t == NULL)
    break;

   if (!blk_put_s (blk,t))             // put t in block
    return (FALSE);
  }

 } // end of while loop

 return (TRUE);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** create new document structure
 * @param doc pointer to document structure pointer
 */

BOOL doc_create (PSX_DOC ** doc)
{
 if (doc == NULL)
  return (FALSE);

 if ((*doc = (PSX_DOC *) malloc (sizeof (PSX_DOC))) == NULL)
  return (FALSE);

 memset (*doc,0,sizeof (**doc));

 return (TRUE);
}

/******************************************************************************/
/** delete document structure and free memory
 * @param doc pointer to document structure pointer
 */
 
BOOL doc_delete (PSX_DOC ** doc)
{
 if (doc == NULL)
  return (FALSE);

 if (*doc == NULL)
  return (FALSE);

 if ((*doc) -> txt)
  free ((*doc) -> txt);

 free (*doc);
 *doc = NULL;

 return (TRUE);
}

/******************************************************************************/
/** assign string src to document
 * @param doc pointer to document structure
 * @param src pointer to string source
 */
 
BOOL doc_assign (PSX_DOC * doc,char * src)
{
 if (doc == NULL)
  return (FALSE);

 if (doc -> txt)
  free (doc -> txt);

 doc -> txt = src ? strdup (src) : NULL;

 return (TRUE);
}

/******************************************************************************/
/** load document from file with path pth
 * @param doc pointer to document structure
 * @param pth pointer to string, contains path to source file
 */
 
BOOL doc_load (PSX_DOC * doc,char * pth)
{
 FILE * f;
 long   n,s;
 char * t;

 if (doc == NULL || pth == NULL)
  return (FALSE);

 if ((f = fopen (pth,"rt")) == NULL)    // open file for read access
  return (FALSE);

 if ((n = getFileSize (f)) == 0)
 {
  fclose (f);
  return (FALSE);
 }

 if ((t = (char *) realloc (doc -> txt,n + 1)) == NULL)
 {
  fclose (f);
  return (FALSE);
 }

 s = fread (t,sizeof (char),n,f);
 fclose (f);
 t [s] = 0;

 doc -> txt = t;

 return (TRUE);
}

/******************************************************************************/
/** generate html code
 * @param doc pointer to document structure
 * @param dic pointer to dictionary structure
 */
 
BOOL doc_compile (PSX_DOC * doc,PSX_DIC * dic)
{
 PSX_BLK * blk;
 BOOL r;

 if (doc == NULL || dic == NULL)
  return (FALSE);

 if (!blk_create (&blk))                      // create new block
  return (FALSE);

 r = TRUE;

//printf ("%s",HCI_CNT);
//dic_print (dic);

 if (doc_map (blk,doc -> txt,dic))            // map document
 {
  char * t;

  if ((t = (char *) malloc ((blk -> len + 1) * sizeof (char))) != NULL)
  {
   memcpy (t,blk -> ptr,blk -> len);         // get text from block
   t [blk -> len] = 0;

   if (doc -> txt)                           
    free (doc -> txt);

   doc -> txt = t;                           // copy text into document
  }
  else
   r = FALSE;
 }
 else
  r = FALSE;

 if (!blk_delete (&blk))
  r = FALSE;

 if (!r)
 {
  char t[] = "<html>ERROR</html>";

  if (doc -> txt)
   free (doc -> txt);

  doc -> txt = strdup (t);
 }

 return (r);
}

/******************************************************************************/

/******************************************************************************/

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


