/******************************************************************************/
/* psx-drs.c                                    Directory Service Interface   */
/******************************************************************************/
/** @file psx-drs.c Directory Service Interface - Source Code file
 * Funcions and definitions providing an interface to a LDAP system.
 * The following functionalities are given:
 * - initialisation
 * - exit of the interface
 * - search for specified attributes
 * - binding the LDAP to a the programme
 * - look up of the client
 * - look up of the configuration
 * - checking the system
 * Note: several string might have to be changed in order to personalize the
 * functions to your system.
 */
 

#include "psx.h"

#ifdef HAVE_LDAP

//#include <lber.h>
#include <ldap.h>

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

typedef struct _DRS_STS
{
 LDAP * con;       // ldap structure


} DRS_STS;

static DRS_STS sts;

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** check for LDAP_SUCCESS
 * @param r LDAP_SUCCESS or an error code
 */

static BOOL drs_chk (int r)
{
 if (r == LDAP_SUCCESS)
  return (TRUE);


 fprintf (stderr,"> error: %s\n",ldap_err2string (r));

 return (FALSE);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** initialize LDAP structure
 * @param svr name of server
 */

BOOL drs_init (const char * svr)
{
 LDAP * ld;
 int r;

 memset (&sts,0,sizeof (sts));

 ld = ldap_init (svr,389);    // returns LAPD struct. Server srv, defport = 389

 if (ld == NULL)
  return (FALSE);

 sts.con = ld;                // store ld in sts.con  (status context)

 return (TRUE);
}

/******************************************************************************/
/** unbind ldap
 */

BOOL drs_exit (void)
{
 int r;

 assert (sts.con != NULL);        // ldap has to exist

 if (sts.con == NULL)
  return (FALSE);

 r = ldap_unbind_s (sts.con);     // release ldap

 if (r != LDAP_SUCCESS)           // check on success
 {
//  int c;

  //c = ldap_get_lderrno (sts.con,NULL,NULL);
  fprintf (stderr,"> drs_exit: error:\n%s\n",ldap_err2string (r));
 }

 sts.con = NULL;
 return (TRUE);
}

/******************************************************************************/
/** bind ldap to programme
 */

BOOL drs_bind (void)
{
 int r;

 r = ldap_simple_bind_s (sts.con,"cn=Manager, o=University of Mainz, c=DE","secret");


 if (!drs_chk (r))
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** search status context for attributes
 */

BOOL drs_search (void)
{
 int r;
 char * atr [3];
 LDAPMessage * m;

 atr [0] = "cn";
 atr [1] = "mail";
 atr [2] = NULL;

 r = ldap_search_ext_s (sts.con,"o=University of Mainz,c=DE",LDAP_SCOPE_SUBTREE,"(cn=Markus Wagner)",atr,0,NULL,NULL,LDAP_NO_LIMIT,0,&m);

 if (!drs_chk (r))
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** lookup several attributes
 * @param sbs
 * @param flt 
 * @param atr attribute string
 * @param val value string
 */

BOOL drs_lookup_and_traverse (char * sbs,char * flt,char * atr,char * val)
{
 int r,n,i;
 char * at [3],* a;
 LDAPMessage * m,* e;
 BerElement * ber;
 char ** v;

 at [0] = "cn"; // atr;
 at [1] = NULL;
 at [2] = NULL;

 r = ldap_search_ext_s (sts.con,sbs,LDAP_SCOPE_SUBTREE,flt,at,0,NULL,NULL,LDAP_NO_LIMIT,0,&m);

 if (!drs_chk (r))
  return (FALSE);

 n = ldap_count_entries (sts.con,m);	// traverse

 printf ("n: %d\n",n);

 if (n != 1)
 {
  printf ("> lookup returned not exactly 1!\n");
  return (FALSE);
 }

 if ((e = ldap_first_entry (sts.con,m)) == NULL)
 {
  printf ("> zero entry\n");
  return (FALSE);
 }


 assert (sts.con != NULL);
 assert (e != NULL);

 for (a = ldap_first_attribute (sts.con,e,&ber);a != NULL;a = ldap_next_attribute (sts.con,e,ber))
 {
  if ((v = ldap_get_values (sts.con,e,a)) == NULL)
  {
   printf ("> zero value\n");
   return (FALSE);
  }

  if (v [0] == NULL)
  {
   printf ("> null value\n");
   return (FALSE);
  }

  for (i = 0;v [i] != NULL;i++)
   printf ("'%s': '%s'\n",a,v [i]);
 }

 ldap_value_free (v);
 ldap_memfree (a);

 if (ber != NULL)
  ber_free (ber,0);

 ldap_msgfree (m);
 // ldap_msgfree (e);

 return (TRUE);
}

/******************************************************************************/
/** lookup ldap configuration
 * @param sbs
 * @param flt
 * @param atr attribute string
 * @param val value string
 */

BOOL drs_lookup (const char * sbs,const char * flt,const char * atr,char * val)
{
 int r,n,i;
 char * atl [2],* a;
 LDAPMessage * m,* e;
 BerElement * ber;
 char ** v;

 atl [0] = atr;
 atl [1] = NULL;

 r = ldap_search_ext_s (sts.con,sbs,LDAP_SCOPE_SUBTREE,flt,atl,0,NULL,NULL,LDAP_NO_LIMIT,0,&m);

 if (!drs_chk (r))
  return (FALSE);

 n = ldap_count_entries (sts.con,m);

 if (n != 1)
 {
  printf ("> lookup returned not exactly 1!\n");
  return (FALSE);
 }

 if ((e = ldap_first_entry (sts.con,m)) == NULL)
 {
  printf ("> zero entry\n");
  return (FALSE);
 }

 assert (sts.con != NULL);
 assert (e != NULL);

 if ((a = ldap_first_attribute (sts.con,e,&ber)) == NULL)
 {
  printf ("> zero attribute\n");
  return (FALSE);
 }

 if ((v = ldap_get_values (sts.con,e,a)) == NULL)
 {
  printf ("> zero value\n");
  return (FALSE);
 }

 if (v [0] == NULL)
 {
  printf ("> null value\n");
  return (FALSE);
 }

 strcpy (val,v [0]);

 ldap_value_free (v);
 ldap_memfree (a);

 if (ber != NULL)
  ber_free (ber,0);

 ldap_msgfree (m);
 // ldap_msgfree (e);

 return (TRUE);
}

/******************************************************************************/
/** test ldap initialization
 */

BOOL drs_test (void)
{
 PSX_STR t;

 printf ("LDAP TEST\n\n");

 if (!drs_init ("magnus"))
 {
  printf ("> error: init failed\n");
  return (FALSE);
 }

 printf ("> init ok\n");

 if (!drs_lookup ("o=University of Mainz,c=DE","(cn=Markus Wagner)","mail",t))
 {
  printf ("> error: search failed\n");
  return (FALSE);
 }

 printf ("> search ok\n");
 printf ("  result: %s\n",t);

 if (!drs_exit ())
 {
  printf ("> error: exit failed\n");
  return (FALSE);
 }

 printf ("> exit ok\n");

 return (TRUE);
}

/******************************************************************************/
/** initialize, lookup and exit
 * @param srv server name (string)
 * @param sbs
 * @param flt
 * @param atr attribute string
 * @param val value string
 */

BOOL drs_lookup_client (const char * svr,const char * sbs,const char * flt,const char * atr,char * val)
{
 if (!drs_init (svr))
  return (FALSE);

 if (!drs_lookup (sbs,flt,atr,val))
  return (FALSE);

 if (!drs_exit ())
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** look up configuration, return true if everything is right
 */ 

BOOL drs_check ()
{
 PSX_STR t,uid,ser;
 PSX_STR dir_svr;
 PSX_STR dir_sbs = "o=University of Mainz,c=DE";
 PSX_STR dir_flt;


 if (!psx_cfg_get ("dir.lookup",t))
  return (TRUE);

 if (strcmp (t,"1") != 0)
  return (TRUE);

 if (!sys_env_get ("HTTPS",t))
  return (TRUE);

 if (strcmp (t,"on") != 0)
  return (TRUE);

 if (!psx_cfg_get ("dir.svr",dir_svr))
  return (FALSE);

 if (!sys_env_get ("SSL_CLIENT_M_SERIAL",ser))
  return (FALSE);

 sprintf (dir_flt,"(uid=)",ser);

 if (!drs_lookup_client (dir_svr,dir_sbs,dir_flt,"pid",t))
  return (FALSE);

 if (strcmp (t,"1") != 0)
  return (FALSE);

 return (TRUE);
}

#endif // HAVE_LDAP

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


