/******************************************************************************/
/* psx-exp.c                                                Expression        */
/******************************************************************************/
/** @file psx-exp.c Expression Handling - Source Code File
 * Functions and Definitions implementing the handling regular expressions in
 * order to use them in SQL statements. The module provides the following
 * functionalities:
 * - creation of one or more new regular expressions
 * - deletion of one or more old expressions
 * - assignment of a varaiable list of string to the expression
 * - adding an existing expression
 * - copying expressions
 * - formatting the expression as an SQL statement
 */
 
#include "psx.h"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** Expression sub type. 
 * Maps expression type and operator to corresponding symbol.
 * Used for SQL generation. 
 * \see exp_sbt 
 * */
static const struct _EXP_SBT
{
 BYTE   etp;              /**< expression type */
 BYTE   opr;              /**< operator        */
 const char * sym;        /**< symbol          */
} exp_sbt [] =
{
 { ETP_ROP,ROP_EQ ,"="   },
 { ETP_ROP,ROP_NE ,"<>"  },
 { ETP_LOP,LOP_AND,"AND" },
 { ETP_LOP,LOP_OR ,"OR"  },
 { ETP_LOP,LOP_NOT,"NOT" }
};

/******************************************************************************/

typedef struct _CTX_NODE
{
 PSX_STL * stl;
} CTX_NODE;

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** delete sub 
 * @param exp pointer to expression
 */
 
static BOOL exp_del_sub (PSX_EXP * exp)
{
 IDX i;

 if (exp == NULL)
  return (FALSE);

 if (exp -> sub == NULL)
 {
  assert (exp -> n == 0);     
  return (FALSE);
 }

 assert (exp -> n > 0);

 for (i = 0;i < exp -> n;i++)     // iterate over expression
 {
  exp_delete (&exp -> sub [i]);   // delete sub
 }

 free (exp -> sub);
 exp -> sub = NULL;

 return (TRUE);
}

/******************************************************************************/
/** lookup symbol
 * @param etp expression type
 * @param opr operator
 */
 
static BOOL exp_sym_lookup (BYTE etp,BYTE opr,PSX_STR sym)
{
 NUM n;
 IDX i;

 n = sizeof (exp_sbt) / sizeof (exp_sbt [0]);    // get number of elements

 for (i = 0;i < n;i++)
  if (exp_sbt [i].etp == etp && exp_sbt [i].opr == opr)
  {
   strcpy (sym,exp_sbt [i].sym);                 // find corresponding operator
   return (TRUE);
  }

 strcpy (sym,"?");

 return (FALSE);
}

/******************************************************************************/
/** encode expression
 * @param exp pointer to expression
 * @param efs expression format specification
 */
 
static BOOL exp_symbolize (PSX_EXP * exp,BYTE efs,PSX_STR sym)
{
 BOOL r = TRUE;

 if (exp == NULL || sym == NULL)
  return (FALSE);

 if (TEST (efs,EFS_CMT))
 {
  if (!str_isnil (exp -> cmt))
  {
   sprintf (sym,"[%s]",exp -> cmt);
   return (TRUE);
  }
 }

 switch (exp -> etp)
 {
  case ETP_CON:                         // constant
   sprintf (sym,"CON [%s]",exp -> opn);
   break;

  case ETP_VAR:                         // variable
   sprintf (sym,"VAR [%s]",exp -> opn);
   break;

  case ETP_ROP:                         // relational operator
  {
   PSX_STR t;

   exp_sym_lookup (ETP_ROP,exp -> opr,t);
   sprintf (sym,"ROP [%s]",t);
   break;
  }

  case ETP_LOP:                         // logical operator
  {
   PSX_STR t;

   exp_sym_lookup (ETP_LOP,exp -> opr,t);
   sprintf (sym,"LOP [%s]",t);
   break;
  }

  default:
   sprintf (sym,"???");
   break;
 }

 return (TRUE);
}

/******************************************************************************/
/** cut off '((' and '))' 
 * @param s pointer to string
 */
 
static BOOL exp_str_reduce (char * s)
{
 int n;

 if (s == NULL)
  return (FALSE);

 if ((n = strlen (s)) < 4)
  return (FALSE);

 if (s [0] != '(')	return (FALSE);
 if (s [1] != '(')	return (FALSE);
 if (s [n - 2] != ')')	return (FALSE);
 if (s [n - 1] != ')')	return (FALSE);

 memmove (s,s + 1,n - 2);
 s [n - 2] = 0;

 return (TRUE);
}

/******************************************************************************/
/** embrace a string
 * @param stl pointer to string list
 * @param efs expression format specification
 * @param ind index in expression
 * @param i0 index
 */
 
static BOOL exp_stl_embrace (PSX_STL * stl,BYTE efs,int ind,IDX i0)
{
 char pad [4000],b0 [1000],b1 [1000];

 if (stl == NULL)
  return (FALSE);

 if (stl -> n < 2 || i0 >= stl -> n)
  return (FALSE);

 str_gen_c (pad,' ',ind);

 if (TEST (efs,EFS_IND))
 {
  sprintf (b0,"(\n%s ",pad);
  sprintf (b1,"\n%s)",pad);
 }
 else
 {
  strcpy (b0,"(");
  strcpy (b1,")");
 }

 stl_ins (stl,i0,b0);
 stl_add (stl,b1);

 return (TRUE);
}

/******************************************************************************/
/** reduce relational operator
 * @param exp pointer to expression
 */      

static BOOL exp_red_rop (PSX_EXP * exp)
{
 IDX i;
 PSX_STR opn;
 BOOL equ;

 if (exp == NULL)
  return (FALSE);

 if (exp -> sub == NULL)
  return (FALSE);

 assert (exp -> etp == ETP_ROP);

 switch (exp -> opr)
 {
  case ROP_EQ: equ = TRUE;  break;
  case ROP_NE: equ = FALSE; break;
  default:
   return (FALSE);
 }

 if (exp -> n == 0)
  return (FALSE);

 assert (exp -> sub [0] != NULL);

 strcpy (opn,exp -> sub [0] -> opn);

 for (i = 0;i < exp -> n;i++)		// ETP!!!
 {
  if (exp -> sub [i] == NULL)
   return (FALSE);

  if ((strcmp (exp -> sub [i] -> opn,opn) == 0) != equ)	// EQ -> ==, NE -> !=
   return (FALSE);
 }

 exp_del_sub (exp);
 exp_assign (exp,ETP_CON,equ ? "" : "");


 return (TRUE);
}

/******************************************************************************/
/** reduce expression type
 * @param exp pointer to expression
 */

static BOOL exp_red (PSX_EXP * exp)
{
 if (exp == NULL)
  return (FALSE);

 if (exp -> sub == NULL)
  return (FALSE);

 switch (exp -> etp)
 {
  case ETP_CON:     // constant
  {
   break;
  }

  case ETP_VAR:     // variable
  {
   break;
  }

  case ETP_ROP:     // relational operator
  {
   switch (exp -> opr)
   {
    case ROP_EQ:          // equality expression
    {
     exp_red_rop (exp);

     break;
    }

    default:
    {
     break;
    }
   }
   break;
  }

  case ETP_LOP:      // logical operator
  {
   break;
  }

  default:
  {
   break;
  }
 }

 return (TRUE);
}

/******************************************************************************/
/* Private                                                             Format */
/******************************************************************************/
/** recursively format the expression
 * @param exp pointer to expression
 * @param efs expression format specification
 * @param ind size of generated string (for padding)
 * @param stl pointer to string list
 */
 
static BOOL exp_prc_fmt_p (PSX_EXP * exp,BYTE efs,int ind,PSX_STL * stl)
{
 char s [4000],t [4000],pad [1000];
 int i;

 if (exp == NULL)
  return (FALSE);

 exp_symbolize (exp,efs,t);

 str_gen_c (pad,' ',ind);       // generate string of length ind and fill with ' '
 sprintf (s,"%s%s\n",pad,t);

 if (!stl_add (stl,s))          // add s to string list
  return (FALSE);

 if (TEST (efs,EFS_CMT) && !str_isnil (exp -> cmt))
  return (TRUE);

 for (i = 0;i < exp -> n;i++)
 {
  assert (exp -> sub != NULL);
  assert (exp -> sub [i] != NULL);
  exp_prc_fmt_p (exp -> sub [i],efs,ind + 1,stl);   // recursion
 }

 return (TRUE);
}

/******************************************************************************/
/** recursively format a sql statement
 * @param exp pointer to expression
 * @param efs expression format specification
 * @param ind size of generated string (for padding)
 * @param stl pointer to string list
 */
 
static BOOL exp_prc_fmt_sql (PSX_EXP * exp,BYTE efs,int ind,PSX_STL * stl)
{
 BOOL r = TRUE;

 if (exp == NULL || stl == NULL)
  return (FALSE);

 switch (exp -> etp)
 {
 /* recursion ends with constants and variables, which are just added to
  * the formatted expression string
  */
  case ETP_CON:          // expression type: constant
  {
   PSX_STR t;

   sprintf (t,"%s",exp -> opn);

   if (!stl_add (stl,t))
    r = FALSE;
   break;
  }

  case ETP_VAR:          // expression type: variable
  {
   PSX_STR t;

   sprintf (t,"%s",exp -> opn);

   if (!stl_add (stl,t))
    r = FALSE;
   break;
  }

  case ETP_ROP:           // expression type: relational operator
  case ETP_LOP:           // expression type: logical operator
  {
   PSX_STR s,t;
   IDX     i;
   BOOL mid = FALSE;

   exp_sym_lookup (exp -> etp,exp -> opr,s);

   // <hack>
   /* comparision with empty value must be encoded as 'IS NULL' or 'IS NOT NULL' */
   if (exp -> n >= 2)
    if (exp -> etp == ETP_ROP && exp -> sub [1] -> etp == ETP_CON && strcmp (exp -> sub [1] -> opn,"NULL") == 0)
    {
     switch (exp -> opr)
     {
      case ROP_EQ:	strcpy (s,"IS");	break;      // equal
      case ROP_NE:	strcpy (s,"IS NOT");	break;  // not equal
     }
    }

   // </hack>

   /* add indentation if requested */
   if (TEST (efs,EFS_IND) && exp -> etp != ETP_ROP)
   {
    char pad [4000];

    str_gen_c (pad,' ',ind);

    sprintf (t,"\n%s%s\n%s",pad,s,pad);
    //sprintf (t,"\n%s%s",pad,s);
   }
   else
    sprintf (t," %s ",s);

   for (i = 0;i < exp -> n;i++)
   {
    IDX p = stl -> n;

    // Recursion
    if (!exp_prc_fmt_sql (exp -> sub [i],efs,ind + 1,stl))
    {
     r = FALSE;
     break;
    }
    /* was anything added to the expression string ? */
    if (stl -> n > p)
    {
     /* add braces around subexpressions with more than one item */
     if (exp -> sub [i] -> etp == ETP_LOP && exp -> sub [i] -> n > 1)
     {
      exp_stl_embrace (stl,efs,ind,p);
     }
	 /* add operator where neccessary */
     if (mid) // i + 1 < exp -> n)
     {
      stl_ins (stl,p,t);
     }

     mid = TRUE;
    }
   }

   break;
  }

  default:
   r = FALSE;
   break;
 }

 return (r);
}

/******************************************************************************/
/** start making expressions
 * @param exp pointer to expression
 * @param efs expression format specification
 * @param ind size of generated string (for padding)
 * @param stl pointer to string list
 */
 
static BOOL exp_prc_fmt (PSX_EXP * exp,BYTE efs,IDX ind,PSX_STL * stl)
{
 if (exp == NULL || stl == NULL)
  return (FALSE);

 if (TEST (efs,EFS_SQL))
 {
  if (!exp_prc_fmt_sql (exp,efs,ind,stl)) 
   return (FALSE);
 }
 else
 {
  if (!exp_prc_fmt_p (exp,efs,ind,stl))   
   return (FALSE);
 }

 return (TRUE);
}


/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/

int exp_sem = 0;
/** create new expression
 * @param exp pointer to pointer to expression structure
 */
 
BOOL exp_create (PSX_EXP ** exp)
{
 if (exp == NULL)
  return (FALSE);

 if ((*exp = (PSX_EXP *) malloc (sizeof (**exp))) == NULL)
  return (FALSE);

 memset (*exp,0,sizeof (**exp));

 exp_sem ++;

 return (TRUE);
}

/******************************************************************************/
/** create n expressions from variable parameter list
 * @param n number of expressions
 * @param ... varaible parameter list
 */
 
BOOL exp_create_n (int n,...)
{
 int     i;
 va_list l;

 va_start (l,n);

 for (i = 0;i < n;i++)
 {
  PSX_EXP ** e;

  e = va_arg (l,PSX_EXP **);

  if (!exp_create (e))
  {
   int k;

   for (k = 0;k < i - 1;k++)
   {
    PSX_EXP ** t = va_arg (l,PSX_EXP **);
    exp_delete (t);
   }
   return (FALSE);
  }
 }

 va_end (l);

 return (TRUE);
}

/******************************************************************************/
/** delete expression structure
 * @param exp pointer to pointer to expression
 */
 
BOOL exp_delete (PSX_EXP ** exp)
{
 if (exp == NULL)
  return (FALSE);

 if (*exp == NULL)
  return (FALSE);

 exp_del_sub (*exp);

 free (*exp);
 *exp = NULL;

 exp_sem --;

 return (TRUE);
}

/******************************************************************************/
/** delete n expressions
 * @param n umber of expressions
 * @param ... variable parameter list
 */
 
BOOL exp_delete_n (int n,...)
{
 int     i;
 va_list l;
 BOOL r = TRUE;

 va_start (l,n);

 for (i = 0;i < n;i++)
 {
  PSX_EXP ** e;

  e = va_arg (l,PSX_EXP **);

  if (e)
   if (!exp_delete (e))
    r = FALSE;
 }

 va_end (l);

 return (r);
}

/******************************************************************************/
/** assign expression
 * @param exp pointer to expression
 * @param etp expression type
 * @param ... variable parameter list
 */
 
BOOL exp_assign (PSX_EXP * exp,BYTE etp,...)
{
 BOOL r = TRUE;
 va_list l;

 if (exp == NULL)
  return (FALSE);

 va_start (l,etp);

 switch (etp)
 {
  case ETP_CON:
  {
   char * t;

   if ((t = va_arg (l,char *)) == NULL)
   {
    r = FALSE;
    break;
   }
   strcpy (exp -> opn,t);
   exp -> etp = etp;
   break;
  }

  case ETP_VAR:
  {
   char * t;

   if ((t = va_arg (l,char *)) == NULL)
   {
    r = FALSE;
    break;
   }
   strcpy (exp -> opn,t);
   exp -> etp = etp;
   break;
  }

  case ETP_ROP:
  {
   BYTE t;

   t = va_arg (l,int); // BYTE);
   exp -> opr = t;
   exp -> etp = etp;
   break;
  }

  case ETP_LOP:
  {
   BYTE t;

   t = va_arg (l,int); // BYTE);
   exp -> opr = t;
   exp -> etp = etp;
   break;
  }

  default:
   r = FALSE;
   break;
 }

 va_end (l);

 return (r);
}

/******************************************************************************/
/** add expression
 * @param exp pointer to expression
 * @param sub pointer to sub expression
 */
 
BOOL exp_add (PSX_EXP * exp,PSX_EXP * sub)
{
 NUM k;
 PSX_EXP ** t;

 if (exp == NULL || sub == NULL)
  return (FALSE);

 k = exp -> n;

 if ((t = (PSX_EXP **) realloc (exp -> sub,(k + 1) * sizeof (PSX_EXP *))) == NULL)
  return (FALSE);

 exp -> sub = t;
 exp -> sub [k] = sub;
 exp -> n++;

 return (TRUE);
}

/******************************************************************************/
/** add n expressions
 * @param exp pointer to expression
 * @param n number of expressions to be added
 * @param ... variable parameter list
 */
 
BOOL exp_add_n (PSX_EXP * exp,NUM n,...)
{
 int     i;
 va_list l;
 BOOL r = TRUE;

 if (exp == NULL)
  return (FALSE);

 va_start (l,n);

 for (i = 0;i < n;i++)
 {
  PSX_EXP * e;

  e = va_arg (l,PSX_EXP *);

  if (e)
   if (!exp_add (exp,e))
    r = FALSE;
 }

 va_end (l);

 return (r);
}

/******************************************************************************/
/** copy expression <<< under construction >>>
 * @param exp pointer to destination expression
 * @param src pointer to source expression
 */
 
BOOL exp_cpy (PSX_EXP * exp,PSX_EXP * src)
{
 if (exp == NULL || src == NULL)
  return (FALSE);

 return (FALSE);
}

/******************************************************************************/
/** set comment
 * @param exp pointer to expression
 * @param cmt comment string
 */
 
BOOL exp_cmt_set (PSX_EXP * exp,PSX_STR cmt)
{
 if (exp == NULL || cmt == NULL)
  return (FALSE);

 strcpy (exp -> cmt,cmt);

 return (TRUE);
}

/******************************************************************************/
/** reduce expression <<< under construction >>>
 * @param exp pointer to expression
 */
 
BOOL exp_reduce (PSX_EXP * exp)
{
 if (exp == NULL)
  return (FALSE);



 return (TRUE);
}

/******************************************************************************/
/** print expression
 * @param exp pointer to expression
 * @param efs expression format specification
 */
 
BOOL exp_print (PSX_EXP * exp,BYTE efs)
{
 char * t;

 if (exp == NULL)
  return (FALSE);

 if (!exp_format (exp,efs,&t))
  return (FALSE);

 assert (t != NULL);

 printf ("EXP [%p]\n",exp);

 printf ("%s\n",t);

 free (t);

 return (TRUE);
}

/******************************************************************************/
/** create string list, format string and free memory afterwards
 * @param exp pointer to expression
 * @param efs expression format specification
 * @param str pointer to pointer to string
 */
 
BOOL exp_format (PSX_EXP * exp,BYTE efs,char ** str)
{
 PSX_STL * stl;

 if (exp == NULL || str == NULL)
  return (FALSE);

 if (!stl_create (&stl))
  return (FALSE);

 if (!exp_prc_fmt (exp,efs,0,stl))
 {
  stl_delete (&stl);
  return (FALSE);
 }

 if (!stl_cat (stl,NULL,str))
 {
  stl_delete (&stl);
  return (FALSE);
 }

 if (!stl_delete (&stl))
 {
  if (*str)
  {
   free (*str);
   *str = NULL;
  }

  return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/** pack expression
 * @param exp pointer to pointer to expression
 * @param etp expression type
 * @param opr operator
 * @param n number of parameters in variable parameter list
 * @param ... variable parameter list
 */
 
BOOL exp_pack (PSX_EXP ** exp,ETP etp,BYTE opr,NUM n,...)
{
 PSX_EXP ** lst;
 va_list l;
 IDX i;
 NUM k;

 if (exp == NULL)
  return (FALSE);

 if (n == 0)
  return (FALSE);

 if (etp != ETP_ROP && etp != ETP_LOP)	/* we only support rop/lop here */
  return (FALSE);

 if ((lst = (PSX_EXP **) malloc (n * sizeof (PSX_EXP *))) == NULL)
  return (FALSE);

 va_start (l,n);

 k = 0;

 for (i = 0;i < n;i++)
 {
  PSX_EXP * e;

  if (e = va_arg (l,PSX_EXP *))
   lst [k++] = e;
 }

 va_end (l);

 if (k == 0)
 {
  free (lst);
  return (FALSE);
 }

 if (k == 1)
 {
  *exp = lst [0];
  free (lst);
  return (TRUE);
 }

 if (!exp_create (exp))
 {
  free (lst);
  return (FALSE);
 }

 exp_assign (*exp,etp,opr);

 for (i = 0;i < k;i++)
  exp_add (*exp,lst [i]);

 free (lst);

 return (TRUE);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/



