/******************************************************************************/
/* psx-exp.h                                                       Expression */
/******************************************************************************/
/** @file psx-exp.h Expression Handling - Header file
 * Definitions supporting the Expression Handling like
 * - Definition of the expression type (constant, variable, relational operator
 *   or logical operator)
 * - Definition of operators
 * - expression format style
 * - PSX_EXP structure containing all necessary information about the expression
 */
 
#ifndef _psx_exp_
#define _psx_exp_

/******************************************************************************/
/*                                                                            */
/******************************************************************************/

#include "psx.h"

/******************************************************************************/
/* Definition                                                                 */
/******************************************************************************/

typedef BYTE ETP;        /**< expression type     */

#define	ETP_ZERO	0x00
#define	ETP_CON		0x01   /**< Expression type: constant. Used as value of PSX_EXP#etp. */
#define	ETP_VAR		0x02   /**< Expression type: variable. Used as value of PSX_EXP#etp. */
#define	ETP_LOP		0x03   /**< Expression type: logical operator. Used as value of PSX_EXP#etp. */
#define	ETP_ROP		0x04   /**< Expression type: relational operator. Used as value of PSX_EXP#etp.  */

typedef BYTE ROP;        /**< relational operator */

#define	ROP_ZERO	0x00
#define	ROP_EQ		0x01   /**< Relational Operator: equal. Used as value of PSX_EXP#opr. */
#define	ROP_NE		0x02   /**< Relational Operator: not equal. Used as value of PSX_EXP#opr. */

typedef BYTE LOP;

#define	LOP_ZERO	0x00   /**< Logical operator */
#define	LOP_AND		0x01   /**< Logical operator: and. Used as value of PSX_EXP#opr. */
#define	LOP_OR		0x02   /**< Logical operator: or. Used as value of PSX_EXP#opr. */
#define	LOP_NOT		0x03   /**< Logical operator: not. Used as value of PSX_EXP#opr. */

/**************************************/
/* Expression Format  Style           */
/**************************************/

typedef BYTE ETP_EFS;

#define	EFS_ZERO	0x00
#define	EFS_SQL		0x01     /**< SQL statement */
#define	EFS_IND		0x02     /**< */
#define	EFS_CMT		0x04     /**< comment       */

/******************************************************************************/
/*                                                                            */
/******************************************************************************/

typedef struct _PSX_EXP PSX_EXP;

/** \brief Data Structure for Expressions
 *
 * Recursive data structure for expressions as used in SQL WHERE clauses. This
 * stores the database request as defined for a step in the matching process.
 * It is converted to an SQL expression in exp_prc_fmt_sql().
 * 
 * An expression is either a constant value (e.g. data typed in by the user), a
 * variable (a column in the database table) or multiple expressions connected
 * by an operator. The expression type is determined by the field #etp, which
 * takes as value one of the constants #ETP_VAR, #ETP_CON, #ETP_LOP, #ETP_ROP.
 * If present, the operator is represented by the field #opr, which takes as
 * value one of the constants #ROP_ZERO, #ROP_EQ, #ROP_NE, #LOP_ZERO, #LOP_AND, 
 * #LOP_OR, #LOP_NOT.
 *   
 */ 
struct _PSX_EXP
{
 BYTE	    etp;      /**< expression type (#ETP_VAR, #ETP_CON, #ETP_LOP, #ETP_ROP) */
 BYTE	    opr;      /**< operator  (<, >, <=, >=, !=, =)                      */
 PSX_STR    opn;    /**< Operand                                              */
 PSX_EXP ** sub;    /**< sub expression                                       */
 NUM        n;      /**< number of sub                                        */
 PSX_STR    cmt;    /**< comment                                              */
};

/******************************************************************************/
/* Service                                                                    */
/******************************************************************************/

LINKAGE	BOOL	exp_create	(PSX_EXP ** exp);
LINKAGE	BOOL	exp_create_n	(int n,...);
LINKAGE	BOOL	exp_delete	(PSX_EXP ** exp);
LINKAGE	BOOL	exp_delete_n	(int n,...);
LINKAGE	BOOL	exp_assign	(PSX_EXP * exp,BYTE etp,...);
LINKAGE	BOOL	exp_add		(PSX_EXP * exp,PSX_EXP * sub);
LINKAGE	BOOL	exp_add_n	(PSX_EXP * exp,NUM n,...);
LINKAGE	BOOL	exp_cpy		(PSX_EXP * exp,PSX_EXP * src);
LINKAGE	BOOL	exp_cmt_set	(PSX_EXP * exp,PSX_STR cmt);
LINKAGE	BOOL	exp_reduce	(PSX_EXP * exp);
LINKAGE	BOOL	exp_print	(PSX_EXP * exp,BYTE efs);
LINKAGE	BOOL	exp_format	(PSX_EXP * exp,BYTE efs,char ** str);

LINKAGE	BOOL	exp_pack	(PSX_EXP ** exp,ETP etp,BYTE opr,NUM n,...);

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/

#endif // _psx_exp_

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


