/******************************************************************************/
/* psx-fsi.c                                           File System Interface  */
/******************************************************************************/
/** @file psx-fsi.c File System Interface - Source Code File
 * Functions implementing an interface to the file system like:
 * - creation of a new file
 * - deletion of an old file
 * - open and close a file
 * - reading data from a file
 * - setting and getting of an offset in a file
 * - writing data in a file
 */
 
#include "psx.h"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/


/*******************************/
/** split source string get source file name
 * @param src source string
 * @param mode file open mode
 * @param fn filename
 */

static BOOL fil_src (char * src,char * mode,char * fn)
{
 char * p,pfx [SYS_STR_MAX];

 if (src == NULL || mode == NULL || fn == NULL)
  return (FALSE);

 if (strlen (src) > SYS_STR_MAX)
  return (FALSE);

 if ((p = strchr (src,':')) != NULL)          // ':' is delimiter (w:filename)
 {
  int k = p - src;

  strncpy (pfx,src,k);                        // prefix 
  pfx [k] = 0;
  strcpy (fn,p + 1);                          // part after delimiter
 }
 else
 {
  strcpy (fn,src);
  strcpy (pfx,"");                            // no prefix
 }

 if (pfx [0] != 0)
  strcpy (mode,pfx);                          // prefix is mode
 else
  strcpy (mode,"rw");                         // default mode

 return (TRUE);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** create new file
 * @param f pointer to file structure
 */

BOOL fil_create (SSI_FILE ** f)
{
 SSI_FILE * fil;

 if (f == NULL)
  return (FALSE);

 if ((fil = SYS_ALLOCATE (SSI_FILE)) == NULL) // SYS_ALLOCATE(type) = ((type *) malloc (sizeof (type)))
  return (FALSE);

// s -> ptr = NULL;

 *f = fil;

 return (TRUE);
}

/******************************************************************************/
/** delete file
 * @param f file pointer
 */

BOOL fil_delete (SSI_FILE ** f)
{
 if (f == NULL)
  return (FALSE);

 if (*f == NULL)
  return (FALSE);

// if ((*f) -> ptr)
//  free ((*str) -> ptr);

 free (*f);
 *f = NULL;

 return (TRUE);
}

/********************************************************************************/
/** open file
 * @param f file pointer
 * @param src input string
 */

BOOL fil_open (SSI_FILE * f,char * src)
{
 char mode [SYS_STR_MAX],pth [SYS_STR_MAX];

 if (f == NULL)
  return (FALSE);

 if (!fil_src (src,mode,pth))               // get mode and path
  return (FALSE);

 f -> f = fopen (pth,mode);

 if (f -> f == NULL)                        // file could not be opened
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** close file
 * @param f file pointer
 */

BOOL fil_close (SSI_FILE * f)
{
 if (f == NULL)
  return (FALSE);

 if (f -> f)
  fclose (f -> f);                      // close file

 f -> f = NULL;
 return (TRUE);
}

/********************************************************************************/
/** read from file
 * @param f file pointer
 * @param p pointer return string
 * @param n size of read object
 */

BOOL fil_read (SSI_FILE * f,PTR p,NUM n)
{
 if (f == NULL || p == NULL)
  return (FALSE);

 if (n == 0)
  return (TRUE);

 if (fread (p,n,1,f -> f) != n)               // read from file
  return (FALSE);

 return (TRUE);
}

/********************************************************************************/
/** writes on stream
 * @param f file pointer
 * @param p pointer to source string
 * @param n size of object
 */

BOOL fil_write (SSI_FILE * f,PTR p,NUM n)
{
 if (f == NULL || p == NULL)
  return (FALSE);

 if (n == 0)
  return (TRUE);

 if (fwrite (p,n,1,f -> f) != n)               // write file
  return (FALSE);

 return (TRUE);
}

/********************************************************************************/
/** set Offset in file
 * @param f file pointer
 * @param pos position in file
 */

BOOL fil_setpos (SSI_FILE * f,IDX pos)
{
 if (f == NULL)
  return (FALSE);

 if (fseek (f -> f,pos,SEEK_SET) != 0)
  return (FALSE);

 return (TRUE);}

/********************************************************************************/
/** return current position in file
 * @param f file pointer
 * @param pos position in file
 */

BOOL fil_getpos (SSI_FILE * f,IDX * pos)
{
 if (f == NULL || pos == NULL)
  return (FALSE);

 *pos = ftell (f -> f);

 return (TRUE);
}


/******************************************************************************/


/******************************************************************************/
/******************************************************************************/
/******************************************************************************/




