/******************************************************************************/
/* psx-fsp-tfm.inl                                       Format Specification */
/******************************************************************************/
/** @file psx-fsp-tfm.inl Format Specification: Transformation - Inline Source Code File
 * Definitions and Functions supporting the specification of the transformation
 * format.
 */
 
/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

#define TFM_STS_DEC 0x01
#define TFM_STS_PHO 0x02
#define TFM_STS_ENC 0x04

/******************************************************************************/
/** Transformation Symbol Transformation Parameter                            */
typedef struct
{
 const char * sym;           /**< symbol                         */
 TFM    tfm;                 /**< transformation specification   */
 int    sts;                 /**< status                         */
} TFM_SBT_PAR;

/** Decoding */
static TFM_SBT_PAR tfm_sbt_d[] =
{
 { "N",TFM_DEC_N,0x01 },
 { "F",TFM_DEC_F,0x02 },
 { NULL,0,0x00 }
};

/** Phonetic Transformation */
static TFM_SBT_PAR tfm_sbt_p[] =
{
 { "C",TFM_PHO_C,0x01 },
 { "H",TFM_PHO_H,0x02 },
 { "*",TFM_PHO_C | TFM_PHO_H,0x03 },
 { NULL,0,0x00 }
};

/** Encryption */
static TFM_SBT_PAR tfm_sbt_e[] =
{
 { "AES" ,TFM_ENC_MOD_ENC (TFM_ENC_MOD_AES) ,0x01 },
 { "IDEA",TFM_ENC_MOD_ENC (TFM_ENC_MOD_IDEA),0x02 },
 { NULL,0,0x00 }
};

/** Symbol Transformation Array */
static const struct
{
 const char *        sym;
 TFM           tfm;
 int           sts;
 TFM_SBT_PAR * par;
} tfm_sbt [] =
{
 { "D",TFM_DEC,TFM_STS_DEC,tfm_sbt_d },
 { "P",TFM_PHO,TFM_STS_PHO,tfm_sbt_p },
 { "E",TFM_ENC,TFM_STS_ENC,tfm_sbt_e }
};

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** split option src
 * @param src pointer to source string
 * @param sym pointer to symbol string
 * @param par pointer to parameter string
 */

static BOOL tfm_opt_split (PSX_STR src,PSX_STR sym,PSX_STR par)
{
 PSX_STR t;
 char * p;
 int k,l;

 if (src == NULL || sym == NULL || par == NULL)
  return (FALSE);

 strcpy (t,src);
 str_trim (t);

 if ((p = strchr (t,'[')) == NULL)
 {
  strcpy (sym,t);
  strcpy (par,"");
  return (TRUE);
 }

 k = (int) (p - t);
 strncpy (sym,t,k);
 sym [k] = 0;

 if (!str_debrace (p,par,'[',']'))
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** check if option src is valid
 * @param sbt pointer to symbol transformation structure
 * @param src pointer to source string
 * @param tfm pointer to transformation structure
 */

static BOOL tfm_opt_get_par (TFM_SBT_PAR * sbt,char * src,TFM * tfm)
{
 int i,s;

 if (sbt == NULL || src == NULL || tfm == NULL)
  return (FALSE);

 if (str_isnil (src))
  return (FALSE);

 i = 0;
 s = 0;

 while (sbt [i].sym != NULL)
 {
  if (strcmp (src,sbt [i].sym) == 0)
  {
   if (TEST (s,sbt [i].sts))
    return (FALSE);

   *tfm |= sbt [i].tfm;
   s    |= sbt [i].sts;
   return (TRUE);
  }
  i++;
 }

 return (FALSE);
}

/******************************************************************************/
/** get option from src
 * @param tfm pointer to transformation structure
 * @param sts integer pointer to status
 * @param src pointer to source string
 */

static BOOL tfm_opt_get (TFM * tfm,int * sts,char * src)
{
 PSX_STR t,sym,par;
 int i,n;
 char * p;

 if (tfm == NULL || sts == NULL || src == NULL)
  return (FALSE);

 strcpy (t,src);
 str_toupper (t);

 if (!tfm_opt_split (t,sym,par))
  return (FALSE);

 n = sizeof (tfm_sbt) / sizeof (tfm_sbt [0]);

 for (i = 0;i < n;i++)
 {
  if (strcmp (sym,tfm_sbt [i].sym) == 0)
  {
   if (TEST (*sts,tfm_sbt [i].sts))
    return (FALSE);

   *tfm |= tfm_sbt [i].tfm;
   *sts |= tfm_sbt [i].sts;

   if (tfm_sbt [i].par)
   {
    if (!tfm_opt_get_par (tfm_sbt [i].par,par,tfm))
     return (FALSE);
   }
   return (TRUE);
  }
 }

 return (FALSE);
}

/******************************************************************************/
/** get transformationformat from src
 * @param tfm pointer to transformation information structure
 * @param src pointer to source string
 */

static BOOL tfm_parse (TFM * tfm,char * src)
{
 PSX_STR t;
 int     i,sts;
 char * p;

 if (tfm == NULL || src == NULL)
  return (FALSE);

 if (str_isnil (src))
  return (FALSE);

 *tfm = TFM_ZERO;

 i = 0;

 // if no transformation code defined -> do not consider item in matching routine
 if (!str_tok (src,&i,":",t))    
 {
  *tfm = TFM_ZERO;
  return (TRUE);
 }

 if (strlen (t) != 1)
  return (FALSE);

 switch (t [0])       // first part always defines use of control numbers
 {
  case '-':
   *tfm = TFM_COL;    // do not generate control numbers
   break;

  case '*':
   *tfm |= TFM_COL;   // generate control numbers
   *tfm |= TFM_CTN;
   break;

  default:
   return (FALSE);
 }

 p = src + i; // p points to string behind ':'

 //printf ("rest: '%s'\n",p);

 if (str_isnil (p))
  if (i == 1)
   return (TRUE);
  else
   return (FALSE);	// no ':' at end

 /* get options */

 // if (!tfm_par_get (tfm,p))
 //  return (FALSE);

 sts = 0;
 i   = 0;

 while (str_tok (p,&i,":",t))   // while ':' found in p, store substring in t and new address in i
 {
  if (!tfm_opt_get (tfm,&sts,t))
   return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/** check/parse transformation format
 * @param tfm transformation format
 * @param sym contains symbolic transformation code after invocation
 */

static BOOL tfm_get_sym (TFM tfm,PSX_STR sym)
{
 if (sym == NULL)
  return (FALSE);

 strcpy (sym,"");		// initialize symbol

 if (tfm == TFM_ZERO)
 {
  strcpy (sym,"Z");
  return (TRUE);
 }

 if (!TEST (tfm,TFM_COL))
 {
  strcpy (sym,"?");
  return (FALSE);
 }

 if (TEST (tfm,TFM_CTN))
  strcpy (sym,"*");
 else
  strcpy (sym,"-");

 if (TEST (tfm,TFM_DEC))
 {
  strcat (sym,":");

 if (TEST (tfm,TFM_DEC_N))
  strcat (sym,"D[N]");
 else
  strcat (sym,"D[F]");
 }

 if (TEST (tfm,TFM_PHO))
 {
  strcat (sym,":");

  if (!(TEST (tfm,TFM_PHO_C) && TEST (tfm,TFM_PHO_H)))
   if (TEST (tfm,TFM_PHO_C))
    strcat (sym,"P[C]");
   else
    strcat (sym,"P[H]");
  else
   strcat (sym,"P[*]");
 }

 if (TEST (tfm,TFM_ENC))
 {
  BYTE e = TFM_ENC_MOD_DEC (tfm);	//  decrypt transformationformat:(((tfm) & 0x1800) >> 11)

  strcat (sym,":E");

  switch (e)
  {
   case TFM_ENC_MOD_AES:			// check encryption mode
    strcat (sym,"[AES]");
    break;

   case TFM_ENC_MOD_IDEA:
    strcat (sym,"[IDEA]");
    break;

   default:
    strcat (sym,"[?]");
    break;
  }


 }

 return (TRUE);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/






