/******************************************************************************/
/* psx-fsp.c                                   Format Specification Interface */
/******************************************************************************/
/** @file psx-fsp.c Format Specification Interface - Source Code File
 * Functions implementing an interface to the specification of format like the
 * position and length of every record item in a batchmode input file.
 * The format specification is held in a file, its path is given in the
 * configuration file. The format specification contains information about
 * - Datatype of record item
 * - fieldname of the record item
 * - label of the record item
 * - Startposition in line
 * - Endposition in line
 * - maximum string length 
 * - Encryption information (*:D[N]:P[H|C|*]:E[AES|IDEA])
 * - relevance for matching
 */
 
#include "psx.h"
#include "psx-fsp.inl"

// #define FEHLWERTBREITE 23 

#include "psx-fsp-sym.inl"
#include "psx-fsp-tfm.inl"
#include "psx-fsp-psr.inl"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

/******************************************************************************/
/* Public                                                                 DTP */
/******************************************************************************/
/** get datatype corresponding to sym and store in dtp
 * @param dtp datatype pointer, contains datatype after invocation
 * @param sym name of datatype
 * @return TRUE if valid datatype was found
 * @return FALSE if datatype could not be retrieved
 */

BOOL fsp_dtp_set (DTP * dtp,PSX_STR sym)
{
 IDX i;

 if (dtp == NULL || sym == NULL)
  return (FALSE);

 for (i = 0;i < (sizeof (fsp_dtp) / sizeof (fsp_dtp [0]));i++)
  if (strcmp (fsp_dtp [i].sym,sym) == 0)
  {
   *dtp = fsp_dtp [i].dtp;
   return (TRUE);
  }

 *dtp = DTP_ZERO;

 return (FALSE);
}

/******************************************************************************/
/** get name of datatype encoded by dtp and store in sym
 * @param dtp datatype
 * @param sym string, contains name of datatype after invocation
 * @return TRUE if name of datatype was found
 * @return FALSE if name of datatype was not found
 */

BOOL fsp_dtp_get (DTP dtp,PSX_STR sym)
{
 IDX i;

 if (sym == NULL)
  return (FALSE);

 for (i = 0;i < (sizeof (fsp_dtp) / sizeof (fsp_dtp [0]));i++)
  if (fsp_dtp [i].dtp == dtp)
  {
   strcpy (sym,fsp_dtp [i].sym);
   return (TRUE);
  }

 strcpy (sym,"?");

 return (FALSE);
}

/******************************************************************************/
/* Public                                                                 TFM */
/******************************************************************************/
/** get symbolic transformation format code
 * @param tfm transformation format
 * @param sym contains symbolic transformation code after invocation
 */

BOOL fsp_tfm_enc (TFM tfm,PSX_STR sym)
{
 if (!tfm_get_sym (tfm,sym))			
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** set transformation for sym
 * @param tfm tranformation structure
 * @param sym
 */

BOOL fsp_tfm_dec (TFM * tfm,char * sym)
{
 if (tfm == NULL || sym == NULL)
  return (FALSE);

 if (!tfm_parse (tfm,sym))
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** Get encryption type.
 * Reads encryption type (AES, IDEA) from tsp and stores it in ctt.
 * @param tsp Transformation specification code.
 * @param ctt Cryptographic transformation type. Takes one of the values CTT_AES, CTT_IDEA, CTT_ZERO.
 */

BOOL fsp_tsp_ctt_get (TSP tsp,CTT * ctt)
{
 if (ctt == NULL)
  return (FALSE);

 if (!TEST (tsp,TFM_ENC))
 {
  *ctt = CTT_ZERO;
  return (TRUE);
 }

 switch (TFM_ENC_MOD_DEC (tsp))
 {
  case TFM_ENC_MOD_AES:    *ctt = CTT_AES;    break;
  case TFM_ENC_MOD_IDEA:   *ctt = CTT_IDEA;   break;
  default:                 *ctt = CTT_ZERO;
                            return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** encode equality into string
 *  EQU_ZERO -> '-'               EQU_KEY -> '#'
 *  EQU_REQ  -> '+'               EQU_OPT -> '*'
 * @param equ equality expression (EQU_ZERO, EQU_KEY, EQU_REQ, EQU_OPT)
 * @param sym encoded equality expression (return value)
 */
// - zero
// + required
// * optional

BOOL fsp_equ_enc (EQU equ,PSX_STR sym)
{
 PSX_STR cmp,xch;

 if (sym == NULL)
  return (FALSE);

 strcpy (sym,"");         // initialize sym with empty string

 switch (EQU_CMP (equ))
 {
  case EQU_ZERO: strcpy (cmp,"-"); break;
  case EQU_KEY:  strcpy (cmp,"#"); break;
  case EQU_REQ:  strcpy (cmp,"+"); break;
  case EQU_OPT:  strcpy (cmp,"*"); break;
  default:       strcpy (cmp,"-"); break;
 }

 if (EQU_XCH (equ) != 0)
  sprintf (xch,"%d",EQU_XCH (equ));
 else
  strcpy (xch,"");

 sprintf (sym,"%s%s%s",cmp,xch [0] != 0 ? ":" : "",xch);

 return (TRUE);
}

/******************************************************************************/
/** decode equality string to defined expression
 * @param equ equality expression (result of procedure)
 * @param src source string
 */

BOOL fsp_equ_dec (EQU * equ,char * src)
{
 PSX_STR t [2];
 int i,x;

 if (equ == NULL || src == NULL)
  return (FALSE);

 *equ = EQU_ZERO;

 i = 0;

 if (!str_tok (src,&i,":",t [0]))
  return (TRUE);

 if (!str_tok (src,&i,":",t [1]))
  strcpy (t [1],"0");

 // get type

 if (strlen (t [0]) != 1)
  return (FALSE);

 switch (t [0][0])
 {
  case '-':	*equ = EQU_ZERO;	break;
  case '#':	*equ = EQU_KEY;		break;
  case '+':	*equ = EQU_REQ;		break;
  case '*':	*equ = EQU_OPT;		break;
  default:	return (FALSE);
 }

 x = atoi (t [1]);

 *equ = MKWORD (*equ,x);

 return (TRUE);
}

/******************************************************************************/
/** encipher ctt
 * @param ctt cryptographic transformation type
 * @param sym encryption mode (result of procedure)
 */

BOOL fsp_ctt_enc (CTT ctt,PSX_STR sym)
{
 if (sym == NULL)
  return (FALSE);

 if (ctt == CTT_ZERO)
 {
  strcpy (sym,"-");
  return (TRUE);

 }

 if (!cti_ctt_enc (ctt,sym))      // get encryption mode (AES||IDEA)
 {
  strcpy (sym,"?");
  return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/** set ctt
 * @param ctt cryptographic transformation type (result of procedure)
 * @param sym string encoded transfomation mode
 */

BOOL fsp_ctt_dec (CTT * ctt,PSX_STR sym)
{
 if (ctt == NULL || sym == NULL)

  return (FALSE);

 if (strcmp (sym,"-") == 0)
 {
  *ctt = CTT_ZERO;
  return (TRUE);
 }

 if (!cti_ctt_dec (ctt,sym))    // set ctt
 {
  *ctt = CTT_ZERO;
  return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** create and initialize a format specification object
 * @param fsp pointer to format specification
 */

BOOL fsp_create (PSX_FMT ** fsp)
{
 if (fsp == NULL)
  return (FALSE);

 if ((*fsp = (PSX_FSP *) malloc (sizeof (**fsp))) == NULL)
  return (FALSE);

 memset (*fsp,0,sizeof (**fsp));

 (*fsp) -> itm = NULL;
 (*fsp) -> n   = 0;

 return (TRUE);
}

/******************************************************************************/
/** delete  fsp object
 * @param fsp pointer to format specification
 */

BOOL fsp_delete (PSX_FSP ** fsp)
{
 if (fsp == NULL)
  return (FALSE);

 if (*fsp == NULL)
  return (FALSE);

 fsp_clr (*fsp);

 free (*fsp);
 *fsp = NULL;

 return (TRUE);
}

/******************************************************************************/
/** clear  fsp object
 * @param fsp format specification
 */

BOOL fsp_clr (PSX_FSP * fsp)
{
 if (fsp == NULL)
  return (FALSE);

 if (fsp -> itm)
 {
  free (fsp -> itm);
  fsp -> itm = NULL;
 }

 fsp -> n = 0;

 return (TRUE);
}

/******************************************************************************/
/** add itm to format specification
 * @param fsp format specification
 * @param itm format item
 */

BOOL fsp_add (PSX_FSP * fsp,FMT_ITM * itm)
{
 FMT_ITM * t;
 int i;

 if (fsp == NULL || itm == NULL)
  return (FALSE);

 i = fsp -> n;

 if ((t = (FMT_ITM *) realloc (fsp -> itm,(i + 1) * sizeof (FMT_ITM))) == NULL)
  return (FALSE);

 fsp -> itm = t;
 fsp -> itm [i] = *itm;
 fsp -> n++;



 return (TRUE);
}

/******************************************************************************/
/** copy src to fsp
 * @param fsp target format specification 
 * @param src source format specification
 */

BOOL fsp_copy (PSX_FSP * fsp,PSX_FSP * src)
{
 IDX i;

 if (fsp == NULL || src == NULL)
  return (FALSE);

 if (!fsp_clr (fsp))
  return (FALSE);

 for (i = 0;i < src -> n;i++)
  if (!fsp_add (fsp,&src -> itm [i]))
   return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** move src to fsp and free src
 * @param fsp target format specification
 * @param src source format specification
 */

BOOL fsp_move (PSX_FSP * fsp,PSX_FSP * src)
{
 IDX i;

 if (fsp == NULL || src == NULL)
  return (FALSE);

 if (!fsp_clr (fsp))
  return (FALSE);

 fsp -> itm = src -> itm;
 fsp -> n   = src -> n;
 src -> itm = NULL;
 src -> n   = 0;

 return (TRUE);
}

/******************************************************************************/
/** get format from file
 * @param fmt target format specification
 * @param f pointer to source file
 */

BOOL fmt_read (PSX_FMT * fmt,FILE * f)
{
 PSS_STR t;

 if (fmt == NULL || f == NULL)
  return (FALSE);

 while (fio_readln (f,t))  
 {
  FMT_ITM itm;

  if (str_isspace (t))
   continue;

  if (!fmt_parse_itm_l (t,&itm))
   return (FALSE);

  if (!fsp_add (fmt,&itm))
   return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/** write fsp to file
 * @param fsp  source format specification
 * @param f pointer to target file
 */


BOOL fsp_write (PSX_FSP * fsp,FILE * f)
{
 int i;

 if (fsp == NULL || f == NULL)
  return (FALSE);

 fio_put_bnr (f,'/',"Format Specification","");

 fprintf (f,"\n");

 for (i = 0;i < fsp -> n;i++)
 {
  if (!fsp_write_fld (fsp,i,f))
   return (FALSE);

  fprintf (f,"\n");
 }

 fio_put_bnr (f,'/',NULL,NULL);

 return (TRUE);
}

/******************************************************************************/
/** open file with name fn and get fsp
 * @param fsp format specification
 * @param fn file name string
 */

BOOL fsp_load_old (PSX_FSP * fsp,char * fn)
{
 FILE * f;
 BOOL r = TRUE;

 if (fsp == NULL || fn == NULL)
  return (FALSE);

 if ((f = fopen (fn,"r")) == NULL)
 {
  psx_put ("> fsp: load error [%s]",fn);
  return (FALSE);
 }

 if (!fmt_read (fsp,f))
  r = FALSE;

 fclose (f);


 return (r);
}

/******************************************************************************/
/** parse and load fn, store format specifications in fsp                                                        
  * @param fsp PSX_FSP pointer to input format specification
  * @param fn string pointer to input format file name (path of format specificaton)
 */

 
BOOL fsp_load (PSX_FSP * fsp,char * fn)
{
 PSX_SPI spi;                        // specification parser interface

 if (fsp == NULL || fn == NULL)
  return (FALSE);

 if (!spi_parse (&spi,SCP_FSP,fn))   // parse input format file and store results in spi
  return (FALSE);

 fsp_move (fsp,spi.fsp);             // copy spi.fsp to fsp

 return (TRUE);
}

/******************************************************************************/
/** save fmt in file fn
 * @param fmt source format specification
 * @param fn file name string
 */

BOOL fmt_save (PSX_FMT * fmt,char * fn)
{
 FILE * f;
 BOOL r;


 if (fmt == NULL || fn == NULL)
  return (FALSE);

 if ((f = fopen (fn,"wt")) == NULL)
  return (FALSE);

 r = TRUE;

 if (!fsp_write (fmt,f))
  r = FALSE;

 fclose (f);

 return (r);
}

/******************************************************************************/
/* log fmt
 * @param fmt format specification
 * @param fn file name string
 */

BOOL fmt_log (PSX_FMT * fmt,char * fn)
{
 FILE * f;
 int i;

 if (fmt == NULL)
  return (FALSE);

 if ((f = fopen (fn,"w")) == NULL)
  return (FALSE);

 fprintf (f,"fmt [%p]:\n",fmt);

 for (i = 0;i < fmt -> n;i++)
 {
  SYS_STR s;

  fsp_tfm_enc (fmt -> itm [i].tfm,s);

  fprintf (f,"['%s',%d,%d,%c]\n",fmt -> itm [i].sym,fmt -> itm [i].pos,fmt -> itm [i].len,s);
 }

 fclose (f);

 return (TRUE);
}

/******************************************************************************/
/** get length of format structure with all items 
 * @param fmt format specification
 * @param len targetlength
 */

BOOL fmt_get_len (PSX_FMT * fmt,NUM * len)
{
 IDX i;
 int n;

 if (fmt == NULL || len == NULL)
  return (FALSE);

 n = 0;

 for (i = 0;i < fmt -> n;i++)        // look up all items
 {
  int k;

  k = fmt -> itm [i].pos + fmt -> itm [i].len - 1;  // pos starts at 1

  if (k > n)
   n = k;
 }

 *len = n;

 return (TRUE);
}

/******************************************************************************/
/** print format
 *@param fmt source format
 */

BOOL fmt_print (PSX_FMT * fmt)
{
 int i;

 if (fmt == NULL)
  return (FALSE);

 printf ("fmt [%p]:\n",fmt);

 for (i = 0;i < fmt -> n;i++)
 {
  char s [1000];

  if (!fmt_itm_get_line (fmt,i,s))
   break;

  printf (" [%s]\n",s);
 }

 return (TRUE);
}


/******************************************************************************/
/** reverse fsp
 * @param fsp source format specification
 */

BOOL fsp_reverse (PSX_FSP * fsp)
{
 IDX i;

 printf ("FSP REVERSE\n");

 if (fsp == NULL)
  return (FALSE);

 for (i = 0;i < fsp -> n;i++)                   // for each fsp
 {
  switch (fsp -> itm [i].ctt)                   // get ctt from fsp item
  {
   case CTT_IDEA:
   {					
    fsp -> itm [i].tfm = TFM_COL + TFM_ENC + TFM_ENC_MOD_ENC (TFM_ENC_MOD_IDEA) + TFM_REV;
    break;			// = 0x0001	 + 0x0010  +  (((0x02) << 11) & 0x1800)		    + 0x0020                
   }

   case CTT_AES:
   {
    fsp -> itm [i].tfm = TFM_COL + TFM_ENC + TFM_ENC_MOD_ENC (TFM_ENC_MOD_AES) + TFM_REV;
    break;
   }

   default:
    fsp -> itm [i].tfm = TFM_COL;
    break;
  }
 }

 return (TRUE);
}



/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


