/******************************************************************************/
/* psx-fsp.h                                             Format Specification */
/******************************************************************************/
/** @file psx-fsp.h Format Specifcation Interface - Header File
 * Definitions supporting the Format Specification Interface like
 * - Definition of the FSP datatype and its possible values
 * - Transformation Specification Code.
 */
#ifndef _psx_fmt_
#define _psx_fmt_

/******************************************************************************/
/* Interface                                                                  */
/******************************************************************************/

#include "psx.h"

/******************************************************************************/
/* Definition                                                                 */
/******************************************************************************/

/**************************************/
/* Data Type                          */
/**************************************/

typedef BYTE DTP;     //!< datatype 

#define	DTP_ZERO	0x00            /**< zero                               */
#define	DTP_TEXT	0x01            /**< text data                          */
#define	DTP_DATE	0x02            /**< date                               */
#define	DTP_NAME	0x03            /**< name                               */
//#define	DTP_FNAME	0x04
#define	DTP_DD		0x05            /**< date / day                         */
#define	DTP_DM		0x06            /**< date / month                       */
#define	DTP_DY		0x07            /**< date / year                        */
#define	DTP_SEX		0x08            /**< male/female                        */
#define	DTP_HIO		0x09            /**< health insurance code              */
#define	DTP_HIC		0x0A            /**< health insurance code              */

/**************************************/
/* Transformation Code                */
/**************************************/
/*                                    */
/* |7|6|5|4|3|2|1|0|                  */
/*  | | | | | | | |                   */
/*  | | | | | | | +--- collect        */
/*  | | | | | | +----- control        */
/*  | | | | | +------- decompose      */
/*  | | | | +--------- phonetics      */
/*  | | | +----------- encrypt        */
/*  | | +-------------                */
/*  | +---------------                */
/*  +----------------- reverse enc    */
/*                                    */
/**************************************/
/*                                    */
/* |F|E|D|C|B|A|9|8|                  */
/*  | | | | | | | |                   */
/*  | | | | | | | +--- dec: N/V       */
/*  | | | | | | +----- pho: C         */
/*  | | | | | +------- pho: H         */
/*  | | | | +--------- ctt: bit 0     */
/*  | | | +----------- ctt: bit 1     */
/*  | | +-------------                */
/*  | +---------------                */
/*  +-----------------                */
/*                                    */
/**************************************/

/** Transformation mode.
 * A bit string with the following bits:<pre>
 *   Bit       Code
 *    0:-----> collect
 *    1:-----> control
 *    2:-----> decompose
 *    3:-----> phonetics
 *    4:-----> encrypt
 *    5:-----> 
 *    6:-----> 
 *    7:-----> reverse enc
 *    8:-----> dec: N/V 
 *    9:-----> pho: C 
 *    A:-----> pho: H
 *    B:-----> ctt: bit 0
 *    C:-----> ctt: bit 1
 *    D:-----> 
 *    E:-----> 
 *    F:-----></pre>
 * - Transformation Mode Code
 * - Item Identification (specification information)
 * - Format Specification structure
 */
typedef WORD TSP;
typedef TSP  TFM;	/* compatibility */
/**
 * Constants for transformation mode.
 */
//@{
#define TFM_ZERO	0x0000  /**< no transformation */
#define TFM_COL		0x0001	/**< collect (no control numbers) */
#define TFM_CTN		0x0002	/**< generate control number */
#define TFM_DEC		0x0004	/**< decompose */
#define TFM_PHO		0x0008	/**< phonetic codes */
#define TFM_ENC		0x0010	/**< encrypt */
#define TFM_REV		0x0020	/**< reverse encryption */

#define TFM_DEC_N	0x0100  /**< decompose name*/
#define TFM_DEC_F	0x0000  /**< decompose firstname*/

#define TFM_PHO_C	0x0200  /**< phonetic code (Cologne) */
#define TFM_PHO_H	0x0400  /**< phonetic code (Hannover) */
//@}

/******************/
/* encryption     */

#define	TFM_ENC_MOD_DEC(tfm)	(((tfm) & 0x1800) >> 11)
#define	TFM_ENC_MOD_ENC(mod)	(((mod) << 11) & 0x1800)

#define TFM_ENC_MOD_AES		0x01
#define TFM_ENC_MOD_IDEA	0x02

/**************************************/
/* Equality Code                      */
/**************************************/
// new: equ != plausibility

typedef WORD EQU;  //< equality code

#define	EQU_ZERO	0x00
#define	EQU_KEY		0x01
#define	EQU_REQ		0x02
#define	EQU_OPT		0x03

#define EQU_CMP(msp)		LOBYTE(msp)
#define EQU_XCH(msp)		HIBYTE(msp)           // exchange (e.g. lastname, alternative name)
#define EQU_GEN(cmp,xch)	MKWORD(cmp,xch)

/**************************************/
/* Item Identification                */
/**************************************/

#define	FSP_ITM_IDX	0x00
#define	FSP_ITM_SYM	0x01
#define	FSP_ITM_DTP	0x02
#define	FSP_ITM_LBL	0x03
#define	FSP_ITM_POS	0x04
#define	FSP_ITM_LEN	0x05
#define	FSP_ITM_MIN	0x06
#define	FSP_ITM_MAX	0x07
#define	FSP_ITM_TFM	0x08
#define	FSP_ITM_EQU	0x09
#define	FSP_ITM_CPS	0x0A
#define	FSP_ITM_CTT	0x0B
#define	FSP_ITM_N	0x0C

/******************************************************************************/
/*                                                                            */
/******************************************************************************/
typedef	BYTE CPS;
// format string: id[1,8,-];Name[10,22,N]

/** Format Specification Item.
 * Stores the format specification of one field with the 
 * parameters explained in the user manual. Some fields take
 * as values constants which are defined in psx-fsp.h.
 */
typedef struct _FSP_ITM
{
 IDX		idx;      /**< index                              */
 PSX_STR	sym;    /**< symbol                             */
 DTP		dtp;      /**< data type                          */
 PSX_STR	lbl;		/**< label                              */
 int		pos;			/**< position                           */
 int		len;			/**< length                             */
 int		min;			/**< minimum length                     */
 int		max;			/**< maximum length                     */
 TFM		tfm;			/**< transformation code                */
 EQU		equ;			/**< equality code                      */
 CPS		cps;			/**< component set                      */
 CTT		ctt;			/**< encryption type, status            */
} FSP_ITM;

typedef FSP_ITM FMT_ITM;		/* compatibility */

/** Format Specification     */
typedef struct _PSX_FSP
{
 FMT_ITM * itm;     //!< item (attribute and value)
 NUM       n;       //!< number of items
} PSX_FSP;

typedef PSX_FSP PSX_FMT;		/* compatibility */

/******************************************************************************/
/* Service                                                                    */
/******************************************************************************/

/**************************************/
/* Coding                             */
/**************************************/

LINKAGE	BOOL	fsp_dtp_set	(DTP * dtp,char * sym);
LINKAGE	BOOL	fsp_dtp_get	(DTP dtp,PSX_STR sym);

LINKAGE	BOOL	fsp_tfm_enc	(TFM tfm,PSX_STR sym);
LINKAGE	BOOL	fsp_tfm_dec	(TFM * tfm,char * sym);
LINKAGE	BOOL	fsp_tsp_ctt_get	(TSP tsp,CTT * ctt);

LINKAGE	BOOL	fsp_equ_enc	(EQU equ,PSX_STR sym);
LINKAGE	BOOL	fsp_equ_dec	(EQU * equ,char * src);

LINKAGE	BOOL	fsp_ctt_enc	(CTT ctt,PSX_STR sym);
LINKAGE	BOOL	fsp_ctt_dec	(CTT * ctt,PSX_STR sym);

/**************************************/
/* Format                             */
/**************************************/

LINKAGE BOOL	fsp_create	(PSX_FMT ** fsp);
LINKAGE BOOL	fsp_delete	(PSX_FMT ** fsp);
LINKAGE BOOL	fsp_clr		(PSX_FMT * fsp);
LINKAGE BOOL	fsp_add		(PSX_FSP * fsp,FMT_ITM * itm);
//LINKAGE BOOL	fsp_assign	(PSX_FSP * fsp,char * f);
LINKAGE BOOL	fsp_copy	(PSX_FSP * fsp,PSX_FSP * src);
LINKAGE BOOL	fsp_move	(PSX_FSP * fsp,PSX_FSP * src);
LINKAGE BOOL	fmt_read	(PSX_FMT * fmt,FILE * f);
LINKAGE BOOL	fsp_write	(PSX_FSP * fsp,FILE * f);
LINKAGE BOOL	fsp_load	(PSX_FSP * fsp,char * fn);
LINKAGE BOOL	fmt_save	(PSX_FMT * fmt,char * fn);
LINKAGE BOOL	fmt_log		(PSX_FMT * fmt,char * fn);
LINKAGE	BOOL	fmt_get_len	(PSX_FMT * fmt,NUM * len);
LINKAGE BOOL	fmt_print	(PSX_FMT * fmt);

LINKAGE	BOOL	fsp_reverse	(PSX_FSP * fsp);

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/

#endif // _psx_fmt_

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


