/******************************************************************************/
/* psx-fsp.inl                                           Format Specification */
/******************************************************************************/
/** @file psx-fsp.inl Format Specification - Inline Source Code File
 * Functions and definitions supporting the Format Specification by implementing
 * specification file operations and attribute handling.
 *
 */
 
/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** Format Specification Symboltable                                          */

static const struct
{
 BYTE atr;           /**< attribute in byte description    */
 const char * sym;   /**< symbol, attribute name as string */
}
fsp_sbt [] =
{
 { FSP_ITM_IDX,"idx" },
 { FSP_ITM_SYM,"sym" },
 { FSP_ITM_DTP,"type" },
 { FSP_ITM_LBL,"label" },
 { FSP_ITM_POS,"start" },
 { FSP_ITM_LEN,"length" },
 { FSP_ITM_MIN,"mini" },
 { FSP_ITM_MAX,"maxi" },
 { FSP_ITM_TFM,"transformation" },
 { FSP_ITM_EQU,"equ" },
 { FSP_ITM_CPS,"cps" },
 { FSP_ITM_CTT,"enc" }
};

/******************************************************************************/
/* Private                                                  old output format */
/******************************************************************************/
/** get symbolstring by byte description from symboltable
 * @param atr attribute in byte description
 * @param sym attribute name after invocation
 */

static BOOL fsp_atr_get_sym (BYTE atr,PSX_STR sym)
{
 IDX i;

 for (i = 0;i < sizeof (fsp_sbt) / sizeof (fsp_sbt [0]);i++)
  if (fsp_sbt [i].atr == atr)
  {
   strcpy (sym,fsp_sbt [i].sym);
   return (TRUE);
  }

 strcpy (sym,"?");

 return (FALSE);
}

/******************************************************************************/
/** get value referenced by attribute atr
 * @param pointer to format specification item
 * @param atr attribute in byte description
 * @param val attribute value after invocation
 */

static BOOL fsp_itm_get_val (FSP_ITM * itm,BYTE atr,PSX_STR val)
{
 switch (atr)
 {
  case FSP_ITM_IDX: sprintf (val,"%d",itm -> idx);	break;
  case FSP_ITM_SYM: strcpy (val,itm -> sym);		break;
  case FSP_ITM_DTP: fsp_dtp_get (itm -> dtp,val);	break;
  case FSP_ITM_LBL: strcpy (val,itm -> lbl);		break;
  case FSP_ITM_POS: sprintf (val,"%d",itm -> pos);	break;
  case FSP_ITM_LEN: sprintf (val,"%d",itm -> len);	break;
  case FSP_ITM_MIN: sprintf (val,"%d",itm -> min);	break;
  case FSP_ITM_MAX: sprintf (val,"%d",itm -> max);	break;
  case FSP_ITM_TFM: fsp_tfm_enc (itm -> tfm,val);	break;
  case FSP_ITM_EQU: fsp_equ_enc (itm -> equ,val);	break;
  case FSP_ITM_CPS: strcpy (val,"???");			break; // itm -> cps); break;
  case FSP_ITM_CTT: fsp_ctt_enc (itm -> ctt,val);	break;
  default:
   strcpy (val,"?");
   return (FALSE);
 }
 return (TRUE);
}

/******************************************************************************/
/** get the maximum fsp item symbol
 * @param fmt pointer to format specification structure
 * @param n int pointer to size of maximum fsp item symbol after invocation
 */

static BOOL fsp_sym_get_max (PSX_FMT * fmt,int * n)
{
 int i,k;

 if (fmt == NULL || n == NULL)
  return (FALSE);

 k = 0;

 for (i = 0;i < fmt -> n;i++)
 {
  int l;

  if ((l = strlen (fmt -> itm [i].sym)) > k)
   k = l;
 }

 *n = k;

 return (TRUE);
}

/******************************************************************************/
/** concatenate format elements into one string str
 * @param fmt pointer to format specification structure
 * @param idx index of format specification item
 * @param str value collection string, after invocation it contains one more
 *            value
 * @param cmp attribute in byte description
 * @param pad number of spaces for padding
 */

static BOOL fmt_itm_cat (PSX_FMT * fmt,IDX idx,PSS_STR str,BYTE cmp,int pad)
{
 char t [1000];

 if (fmt == NULL || str == NULL)
  return (FALSE);

 //if ((fmt -> fmc & fmc) == 0)
 // return (FALSE);

 switch (cmp)
 {
  case FSP_ITM_SYM: sprintf (t,"%s",fmt -> itm [idx].sym); break;
  case FSP_ITM_POS: sprintf (t,"%04d",fmt -> itm [idx].pos); break;
  case FSP_ITM_LEN: sprintf (t,"%04d",fmt -> itm [idx].len); break;
  case FSP_ITM_TFM:
  {
   char f [1000];

   fsp_tfm_enc (fmt -> itm [idx].tfm,f);
   sprintf (t,"%s",f);
   break;
  }
  default:      strcpy (t,"?"); break;
 }

 str_pad (t,pad);			// padding, filling in pad blanks

 if (str [0] != 0)
  strcat (str," ");

 strcat (str,t);

 return (TRUE);
}

/******************************************************************************/
/** generates a line for output
 * @param fmt pointer to format specification
 * @param idx index of format specification item
 * @param str attribute value collection string
 */

static BOOL fmt_itm_get_line (PSX_FMT * fmt,IDX idx,PSS_STR str)
{
 int var_xs;

 if (fmt == NULL || str == NULL)
  return (FALSE);

 fsp_sym_get_max (fmt,&var_xs);

 strcpy (str,"");

 fmt_itm_cat (fmt,idx,str,FSP_ITM_SYM,var_xs);
 fmt_itm_cat (fmt,idx,str,FSP_ITM_POS,0);
 fmt_itm_cat (fmt,idx,str,FSP_ITM_LEN,0);
 fmt_itm_cat (fmt,idx,str,FSP_ITM_TFM,0);

 return (TRUE);
}

/******************************************************************************/
/* Private                                                  new output format */
/******************************************************************************/

/******************************************************************************/
/** write field line to file f
 * @param itm pointer to format specification item
 * @param atr attribute in byte description
 * @param pad number of spaces for padding
 * @param f file pointer
 */

static BOOL fsp_write_fld_line (FSP_ITM * itm,BYTE atr,int pad,FILE * f)
{
 PSX_STR sym,val,s;

 if (!fsp_atr_get_sym (atr,sym))		// get sym by atr
  return (FALSE);


 if (!fsp_itm_get_val (itm,atr,val))	// get val by atr
  return (FALSE);

 strcpy (s,sym);
 str_pad (s,pad);
 strcat (s," = ");
 strcat (s,val);

 fprintf (f," %s\n",s);

 return (TRUE);
}


/*****************************************************************************/
/** generate field
 * @param fsp pointer to format specification
 * @param idx index of format specification item
 * @param f file pointer
 */

static BOOL fsp_write_fld (PSX_FSP * fsp,IDX idx,FILE * f)
{
 FSP_ITM * itm;
 int pad = 18;
 PSX_STR sym,val;

 if (fsp == NULL || f == NULL)
  return (FALSE);

 itm = &fsp -> itm [idx];

 if (idx >= fsp -> n)
  return (FALSE);

 fprintf (f,"Field %s\n",itm -> sym);
 fprintf (f,"{\n");

 fsp_write_fld_line (itm,FSP_ITM_DTP,pad,f);
 fsp_write_fld_line (itm,FSP_ITM_POS,pad,f);
 fsp_write_fld_line (itm,FSP_ITM_LEN,pad,f);
 fsp_write_fld_line (itm,FSP_ITM_CTT,pad,f);


 fprintf (f,"}\n");

 return (TRUE);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/





