/******************************************************************************/
/* psx-gui.c                                         Graphical User Interface */
/******************************************************************************/
/** @file psx-gui.c Graphical User Interface - Source Code File
 * Definitions and functions providing a graphical user interface using html. 
 * The actual interface (the html coded document) is produced by the Hypertext
 * Construction Interface, managed by the Web User Interface.
 */

#include "psx.h"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

typedef BOOL (* GUI_PRC)(GUI_CTX * ctx);
#define GUI_PROCEDURE(prc) BOOL prc (GUI_CTX * ctx)

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

#include "psx-gui-ptl.inl"
#include "psx-gui-sys.inl"
#include "psx-gui-req.inl"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** gui operation structure                                                   */
static struct
{
 BYTE    ctx;                 /**< context                                    */
 GUI_PRC prc;                 /**< procedure address                          */
} gui_prc [] =                /**< array holding pairs of web user interface commands and associated procedures */
{
 { WUI_CTX_PTL,prc_ptl },     /**< portal  (WUI_CTX_PTL = 0x02)               */
 { WUI_CTX_SYS,prc_sys },     /**< system  (WUI_CTX_SYS = 0x01)               */
 { WUI_CTX_REQ,prc_req }      /**< request (WUI_CTX_REQ = 0x04)               */
};

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** get gui procedure and store in prc                                                       
 * @param ctx BYTE encoding gui procedure                                     
 * @param prc gui procedure
 */
 
static BOOL gui_prc_lookup (BYTE ctx,GUI_PRC * prc)
{
 IDX i;

 for (i = 0;i < sizeof (gui_prc) / sizeof (gui_prc [0]);i++)
  if (gui_prc [i].ctx == ctx)
  {
   if (prc)
    *prc = gui_prc [i].prc;
   return (TRUE);
  }

  if (prc)
   *prc = prc_ptl;               /* default: procedure portal */
    
 return (TRUE);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** create gui context
 * @param pointer to GUI_CTX pointer holding the context data after invokation
 */
 
BOOL gui_create (GUI_CTX ** ctx)
{
 if (ctx == NULL)
  return (FALSE);

 if ((*ctx = (GUI_CTX *) malloc (sizeof (**ctx))) == NULL)
  return (FALSE);

 memset (*ctx,0,sizeof (*ctx));

 return (TRUE);
}

/******************************************************************************/
/** delete gui context
 * @param pointer to GUI_CTX pointer being frred in this function
 */
 
BOOL gui_delete (GUI_CTX ** ctx)
{
 if (ctx == NULL)
  return (FALSE);

 if (*ctx == NULL)
  return (FALSE);

 free (*ctx);

 return (TRUE);
}

/******************************************************************************/
/** initialize gui context
 * @param ctx pointer to gui context
 * @param cfg pointer to gui configuration structure
 */
 
BOOL gui_init (GUI_CTX * ctx,GUI_CFG * cfg)
{
 if (ctx == NULL || cfg == NULL)
  return (FALSE);

 memset (ctx,0,sizeof (*ctx));

 if (!cgi_init ())                       /* initialize cgi           */
  return (FALSE);    
  
 if (!pdi_open ())                       /* open database                     */
  return (FALSE);

 if (!pdi_sch_get (&ctx -> sch))         /* get schema from database          */
  return (FALSE);

 if (!wui_init (&ctx -> wui))            /* initialize wui context            */
  return (FALSE);

 if (!rec_create (&ctx -> rec))          /* create record                     */
  return (FALSE);

 if (!rec_create (&ctx -> ctn))          /* create record with controlnumbers */
  return (FALSE);

 if (!blk_create (&ctx -> blk))          /* create new block for html code    */
  return (FALSE);

 ctx -> sys = cfg -> sys;                /* get system data from config       */
 ctx -> dbg = cfg -> dbg;

 ctx -> wui.dbg = ctx -> dbg;            

 return (TRUE);
}

/******************************************************************************/
/** exit gui
 * @param ctx pointer to context
 */

BOOL gui_exit (GUI_CTX * ctx)
{
 BOOL r = TRUE;

 if (ctx == NULL)
  return (FALSE);

 if (!wui_exit (&ctx -> wui))  /* exit wui         */
  r = FALSE;

 if (!pdi_close ())            /* cklose database  */
  r = FALSE;

 if (!cgi_exit ())             /* exit cgi         */
  r = FALSE;

 if (!rec_delete (&ctx -> rec))
  r = FALSE;

 if (!rec_delete (&ctx -> ctn))
  r = FALSE;

 if (!blk_delete (&ctx -> blk))
  r = FALSE;

 return (r);
}

/******************************************************************************/
/** run graphical user interface, send HTML document to stdout (which int this
 *  case is the browser)
 * @param ctx pointer to GUI context
 */

BOOL gui_run (GUI_CTX * ctx)
{
 BOOL r;
 GUI_PRC p;

 if (ctx == NULL)
  return (FALSE);

 r = TRUE;

 ctx -> wui.tpl = TPL_ZERO;    // default web user interface template

 /*************************************/

#ifdef HAVE_LDAP
 if (!drs_check ())
 {
  ctx -> wui.tpl = TPL_MSG;
  sprintf (ctx -> msg,"Access denied");
  dic_put (ctx -> wui.dic,"TXT",ctx -> msg);
  wui_put (&ctx -> wui);
  return (FALSE);
 }
#endif

 /*************************************/

 gui_prc_lookup (ctx -> wui.ctx,&p);  // p = prc_ptl or prc_sys or prc_..; procedure creating HTML document
 assert (p != NULL);
 r = p (ctx);     // r = HTML document

 /*************************************/

 if (!wui_put (&ctx -> wui))
  r = FALSE;

 /*************************************/

 return (r);
}

/******************************************************************************/

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


