/******************************************************************************/
/* psx-hci.c                                 Hypertext Construction Interface */
/******************************************************************************/
/** @file psx-hci.c Hypertext Construction Interface - Source Code File
 * Functions implementing the generation of a HTML document.
 * This file also includes the definition of the Hypertext Construction
 * Interface Status structure (HCI_STS).
 */
  
#include "psx.h"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** hypertext construction interface status structure
 */
 
static struct _HCI_STS
{
 HCI_CTX * ctx;       /**< pointer to context structure                       */
                      /** put structure with file pointer and block pointer   */
 struct
 {
  FILE    * f;
  PSX_BLK * b;
 } put;

} sts =
{
 NULL,
 {
  NULL,
  NULL
 }
};

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

typedef BOOL (* HCI_PUT_PRC) (const char * str);
#define HCI_PUT_PROCEDURE(prc) BOOL prc (const char * str)

//#define HCI_CNT "Content-type: text/html\n\n"

/******************************************************************************/
/** BOOL hci_prc_put_f(char * str): print str to file sts.put.f
 * @param str string pointer, new status
 */
 
static HCI_PUT_PROCEDURE (hci_prc_put_f)
{
 if (sts.put.f == NULL || str == NULL)
  return (FALSE);

 fprintf (sts.put.f,"%s",str);  

 return (TRUE);
}

/******************************************************************************/
/** BOOL hci_prc_put_b(char * str): print str to block
 * @param str string pointer, new status
 */
 
static HCI_PUT_PROCEDURE (hci_prc_put_b)
{
 if (sts.put.b == NULL || str == NULL)
  return (FALSE);

 if (!blk_put (sts.put.b,str,strlen (str))) 
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** generate hypertext node
 * @param htx hypertext structure
 */
 
static BOOL hci_htn_gen (HTX * htx)
{
 HCI_HTN * t;
 NUM       k;

 if (sts.ctx == NULL)
  return (FALSE);

 k = sts.ctx -> n;             // get number of status context

 if ((t = (HCI_HTN *) realloc (sts.ctx -> htn,(k + 1) * sizeof (HCI_HTN))) == NULL)
  return (FALSE);

 sts.ctx -> n++;              // one added
 sts.ctx -> htn = t;          // set number of new node

 memset (&(sts.ctx -> htn [k]),0,sizeof (sts.ctx -> htn [k]));

 if (htx)                     
  *htx = k;                   // get number of status context

 return (TRUE);
}

/******************************************************************************/
/** get hypertext node
 * @param ctx context structure
 * @param htx hypertext structure
 * @param hypertext node
 */
 
static BOOL hci_htn_get (HCI_CTX * ctx,HTX htx,HCI_HTN ** htn)
{
 if (ctx == NULL)
  return (FALSE);

 if (htx >= ctx -> n)
  return (FALSE);

 if (htn)
  *htn = &(ctx -> htn [htx]);     // get hypertext node from context

 return (TRUE);
}

/******************************************************************************/
/** make hypertext node
 * @param htx hypertext structure
 * @param htn hypertext node
 */
 
static BOOL hci_htn_make (HTX * htx,HCI_HTN ** htn)
{
 if (sts.ctx == NULL)
  return (FALSE);

 if (!hci_htn_gen (htx))
  return (FALSE);

 if (!hci_htn_get (sts.ctx,*htx,htn))
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** generate string from hypertext node information
 * @param htx hypertext
 * @param idn identification number
 * @param prc name of procedure for output (hci_prc_put_f || hci_prc_put_b)
 */
 
const cfg_format = FALSE;

static BOOL hci_htn_print (HTX htx,NUM idn,HCI_PUT_PRC prc)
{
 HCI_HTN * htn;
 IDX i;
 NUM n;
 BOOL b;

 if (sts.ctx == NULL || prc == NULL)
  return (FALSE);

 if (!hci_htn_get (sts.ctx,htx,&htn))           // get hypertext node
  return (FALSE);

 if (htn -> txt)                                // text
 {
  PSX_STR t,s;
  txt_gen_chr (t,' ',idn > 0 ? idn - 1 : 0);    // generate string and initialize with ' '

  if (htn -> txt [0] != '<' && htn -> txt [0] != '>')
   t [0] = 0;

  //sprintf (s,"%s%s\n",t,htn -> txt);
  //prc (s);

  if (cfg_format)                         
   prc (t);                               // call variable procedure (depends on parameter)

  prc (htn -> txt);

  if (cfg_format)
   prc ("\n");
 }
 else
 {
  PSX_STR t;
  txt_gen_chr (t,' ',idn);
  //printf ("%s%s\n",t,"*");
 }

 b = FALSE;

 for (i = 0;i < htn -> n;i++)
 {
  HCI_HTN * h;
  if (hci_htn_get (sts.ctx,htn -> itm [i],&h))  // get hypertext node
   if (h -> txt)
    b = TRUE;
 }

 for (i = 0;i < htn -> n;i++)
  hci_htn_print (htn -> itm [i],b ? idn + 1 : idn,prc);   // print hypertext node

 return (TRUE);
}

/******************************************************************************/
/** free memory of hypertext node
 * @param htn hypertext node
 */
 
static BOOL hci_htn_free (HCI_HTN * htn)
{
 if (htn == NULL)
  return (FALSE);

 if (htn -> itm)
  free (htn -> itm);

 if (htn -> txt)
  free (htn -> txt);

 return (TRUE);
}

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** generate hypertext node
 * @param txt string
 * @return htx hypertext
 */
 
static HTX hci_htn_gen_txt (char * txt)
{
 HTX   htx;
 HTN * htn;

 if (txt == NULL)
  return (HTX_NIL);

 if (!hci_htn_make (&htx,&htn))    // make hypertext node
  return (HTX_NIL);

 htn -> txt = strdup (txt);        // copy txt to hypertext node

 return (htx);
}

/******************************************************************************/
/** appends string formatted according to fmt to str                          
 * @param str string pointer to which values are appended to                                
 * @param fmt string pointer which to append to str including formatting elements                 
 * @param ... arguments associated with formatting elements, variable argument list
 */
 
static BOOL hci_apf (char * str,const char * fmt,...)
{
 char t [1000];
 va_list l;

 if (str == NULL || fmt == NULL)
  return (FALSE);

 va_start (l,fmt);              // start variable argument list
 vsprintf (t,fmt,l);
 va_end (l);                    // end variable argument list

 strcat (str,t);                // append text to string

 return (TRUE);
}

/******************************************************************************/
/** get hypertext context
 * @return htx hypertext
 */
 
static HTX hci_hcx_get (void)
{
 HTX htx;

 if (sts.ctx == NULL)
  return (HTX_NIL);

 if (sts.ctx -> n == 0)
  return (HTX_NIL);

 htx = sts.ctx -> n - 1;

 return (htx);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** create hypertext construction context
 * @param ctx pointer to hypertext construction interface context
 */
 
BOOL hci_create (HCI_CTX ** ctx)
{
 if (ctx == NULL)
  return (FALSE);

 if ((*ctx = (HCI_CTX *) malloc (sizeof (**ctx))) == NULL)
  return (FALSE);

 memset (*ctx,0,sizeof (**ctx));

 return (TRUE);
}

/******************************************************************************/
/** delete hypertext construction interface context
 * @param ctx pointer to hypertext construction interface context structure
 */
 
BOOL hci_delete (HCI_CTX ** ctx)
{
 if (ctx == NULL)
  return (FALSE);

 if (*ctx == NULL)
  return (FALSE);

 if ((*ctx) -> htn)
 {
  IDX i;

  for (i = 0;i < (*ctx) -> n;i++)
   hci_htn_free (&(*ctx) -> htn [i]);
 }

 free (*ctx);
 *ctx = NULL;

 return (TRUE);
}

/******************************************************************************/
/** start new hypertext construction context
 */
 
BOOL hci_begin (void)
{
 if (sts.ctx != NULL)
  return (FALSE);

 if (!hci_create (&sts.ctx))
  return (FALSE);

 assert (sts.ctx != NULL);

 return (TRUE);
}

/******************************************************************************/
/** end hypertext construction context
 */
 
BOOL hci_end (void)
{
 if (sts.ctx == NULL)
  return (FALSE);

 if (!hci_delete (&sts.ctx))
  return (FALSE);

 assert (sts.ctx == NULL);

 return (TRUE);
}

/******************************************************************************/
/** set output stream and print hypertext to FILE * f, this also may be stdout
 * @param f output stream
 */
 
BOOL hci_put_f (FILE * f)
{
 HTX x;

 if (f == NULL)
  return (FALSE);

 if ((x = hci_hcx_get ()) == HTX_NIL)      // get context
  return (FALSE);

 sts.put.f = f;                            // set output stream
 //fprintf (sts.put.f,"%s",HCI_CNT);

 if (!hci_htn_print (x,0,hci_prc_put_f))   // print hypertext node
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** output block
 * @param blk block structure
 */
 
BOOL hci_put_b (PSX_BLK * blk)
{
 HTX x;
 char * t;

 if (blk == NULL)
  return (FALSE);

 if ((x = hci_hcx_get ()) == HTX_NIL)       // get context
  return (FALSE);

 //t = HCI_CNT; not used if 423 is commented out
 sts.put.b = blk;                           // set output stream
 //blk_put (sts.put.b,t,strlen (t));

 if (!hci_htn_print (x,0,hci_prc_put_b))    // print block (recursively)
  return (FALSE);

 blk_put (sts.put.b,"",1);                  // print empty block

 return (TRUE);
}

/******************************************************************************/
/** output block, assign doc
 * @param doc pointer to PSX_DOC structure
 */
 
BOOL hci_put_d (PSX_DOC * doc)
{
 PSX_BLK * blk;
 BOOL r;

 if (doc == NULL)
  return (FALSE);

 if (!blk_create (&blk))       // create new block
  return (FALSE);

 r = TRUE;

 if (hci_put_b (blk))          // print block
 {
  blk_put (blk,"",1);          // initialize block with blank

  if (!doc_assign (doc,blk -> ptr)) // assign text from block to document
   r = FALSE;
 }
 else
  r = FALSE;

 if (!blk_delete (&blk))
  r = FALSE;

 return (r);
}

/******************************************************************************/
/** do output, stream is stdout
 */
 
BOOL hci_put (void)
{
 if (!hci_put_f (stdout))      // use stdout as output stream
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** dump information on stdout
 */
 
BOOL hci_dump (void)
{
 IDX i;

 printf ("HCI Dump\n");
 printf ("> sts.ctx: %P\n",sts.ctx);
 printf ("> htn:\n");

 for (i = 0;i < sts.ctx -> n;i++)
 {
  printf ("  > htn [%2d]: %p, txt: %s",i,&sts.ctx -> htn,sts.ctx -> htn [i].txt);

  if (sts.ctx -> htn [i].n > 0)
  {
   IDX j;

   printf (", [");

   for (j = 0;j < sts.ctx -> htn [i].n;j++)
    printf ("%s%d",j > 0 ? "," : "",sts.ctx -> htn [i].itm [j]);
   printf ("]");
  }
  printf ("\n");
 }

 return (TRUE);
}

/******************************************************************************/
/** add another hypertext node
 * @param htx hypertext
 * @param sub sub-hypertext to be inserted into hypertext
 */
 
BOOL hci_add (HTX htx,HTX sub)
{
 HCI_HTN * htn;
 NUM k;
 IDX * t;

 if (sts.ctx == NULL)
  return (FALSE);

 if (htx >= sts.ctx -> n || sub >= sts.ctx -> n)
  return (FALSE);

 if (!hci_htn_get (sts.ctx,htx,&htn))       // get hypertext node
  return (FALSE);

 k = htn -> n;

 if ((t = (HTX *) realloc (htn -> itm,(k + 1) * sizeof (HTX))) == NULL)
  return (FALSE);

 htn -> n++;
 htn -> itm = t;
 htn -> itm [k] = sub;

 return (TRUE);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** get color
 * @param clr string or character, color
 * @param str pointer to string
 */
 
BOOL hci_clr (CLR clr,char * str)
{
 if (str == NULL)
  return (FALSE);

 sprintf (str,"#%06X",clr);

 return (TRUE);
}

/******************************************************************************/
/** generate new HTML element or combines multiple elements to a new one.
 *  Count of X marks, how many arguments are passed, e.g. "xxx",x1,xm,x2                                                  
 * @param fmt format [t|x|NULL]* (text | HTML element | NULL)                 
 * @return HTML element
 */
 
HTX hci_gen (const char * fmt,...)
{
 char c;
 va_list l;
 HTX htx;
 BOOL b;

 if (fmt == NULL)
 {
  if (!hci_htn_gen (&htx))     // generate hypertext node
   return (HTX_NIL);

  return (htx);
 }

 va_start (l,fmt);

 htx = HTX_NIL;
 b = FALSE;

 while ((c = *fmt++) != '\0')
 {
  HTX itm;

  switch (c)
  {
   case 't':
   {
    char * t;

    t = va_arg (l,char *);
    itm = hci_htn_gen_txt (t);    // generate hypertext node with text t
    break;
   }

   case 'x':
   {
    itm = va_arg (l,HTX);
    break;
   }

   default:
    va_end (l);
    return (HTX_NIL);
  } // end switch

  if (itm != HTX_NIL)      /* item produced */
  {
   if (htx != HTX_NIL)
   {
    if (!b)
    {
     HTX x;

     if (!hci_htn_gen (&x))
      return (HTX_NIL);

     if (!hci_add (x,htx))
      return (HTX_NIL);

     htx = x;

     if (!hci_add (htx,itm))    // add hypertext node
      return (HTX_NIL);
     b = TRUE;
    }
    else
    {
     if (!hci_add (htx,itm))
     {
      return (HTX_NIL);
     }
    }
   }
   else
    htx = itm;
  }
 } // end while

 va_end (l);

 return (htx);
}

/******************************************************************************/
/** generate html text
 * @param txt stringpointer to text to be inserted in hypertext node
 * @return x generated hypertext
 */
 
HTX hci_txt (const char * txt)
{
 HTX x;

 if (txt == NULL)
  return (HTX_NIL);

 x = hci_gen ("t",txt);   // generate HTML text element 

 return (x);              // return generated hypertext node
}

/******************************************************************************/
/**
 * @param src source hypertext
 * @param sym pointer to character symbol (attribute)
 * @param fmt pointer to character format
 * @param ... more string elements depending on specification 
 */
 
HTX hci_tag (HTX src,const char * sym,const char * fmt,...)
{
 va_list l;
 char p [2000],t1 [2000],t2 [2000];
 HTX x,x1,x2;

 if (sym == NULL)
  return (HTX_NIL);

 if (fmt)
 {
  va_start (l,fmt);
  vsprintf (p,fmt,l);
  va_end (l);
 }
 else
  strcpy (p,"");

 sprintf (t1,"<%s%s%s>",sym,p [0] != 0 ? " " : "",p);
 sprintf (t2,"</%s>",sym);

 x1 = hci_gen ("t",t1);
 x2 = hci_gen ("t",t2);

 if (src != HTX_NIL)
  x  = hci_gen ("xxx",x1,src,x2);     
 else
  x  = hci_gen ("xx",x1,x2);

 return (x);
}

/******************************************************************************/
/** generate title html tag
 * @param txt pointer to string, text for title
 * @return HTML element
 */
 
HTX hci_title (const char * txt)
{
 HTX x,x1,xt,x2;

 x1 = hci_gen ("t","<title>");
 x2 = hci_gen ("t","</title>");
 xt = hci_gen ("t",txt ? txt : "");
 x  = hci_gen ("xxx",x1,xt,x2);

 return (x);
}

/******************************************************************************/
/** generate head html tag
 * @param ttl string pointer to title name
 * @return HTML element
 */
 
HTX hci_head (const char * ttl)
{
 HTX x,x1,xt,x2;

 x1 = hci_gen ("t","<head>");
 x2 = hci_gen ("t","</head>");
 xt = hci_title (ttl);
 x  = hci_gen ("xxx",x1,xt,x2);

 return (x);
}

/******************************************************************************/
/** generate html body
 * @param bgc pointer to string, backgroundcolor of html file
 * @param txc pointer to string, textcolor
 * @param src pointer to string, source hypertext
 * @return HTML element
 */
/* 
HTX hci_body (char * bgc,char * txc,HTX src)
{
 HTX x,x1,x2;
 PSX_STR t;

 strcpy (t,"<body");

 if (bgc)		hci_apf (t," bgcolor='%s'",bgc);
 if (txc)		hci_apf (t," text='%s'",txc);

 strcat (t,">");

 x1 = hci_gen ("t",t);
 x2 = hci_gen ("t","</body>");
 x  = hci_gen ("xxx",x1,src,x2);

 return (x);
}
*/
/******************************************************************************/
/** create HTML document                                                       
 * @param ttl title                                                           
 * @param src body content                                                    
 * @return HTML document
 */
HTX hci_doc (const char * ttl,HTX src)
{
 HTX x,x1,x2,xm,xh;

 x1 = hci_gen ("t","<html>");
 x2 = hci_gen ("t","</html>");
 xh = hci_head (ttl);
 xm = hci_gen ("xx",xh,src);
 x  = hci_gen ("xxx",x1,xm,x2);

 return (x);
}

/******************************************************************************/
/** generate a html link tag
 * @param url pointer to string, linked url
 * @param tgt pointer to string, targetframe
 * @param src pointer to hypertext, source
 * @return HTML link element
 */
 
HTX hci_a (const char * url,const char * tgt,HTX src)
{
 HTX x,x1,x2;
 PSX_STR t;

 strcpy (t,"<a");

 if (url)		hci_apf (t," href='%s'",url);
 if (tgt)		hci_apf (t," target='%s'",tgt);

 strcat (t,">");

 x1 = hci_gen ("t",t);
 x2 = hci_gen ("t","</a>");
 x  = hci_gen ("xxx",x1,src,x2);

 return (x);
}

/******************************************************************************/
/** generate center tag
 * @param src pointer to hypertext string
 * @return HTML center tag
 */
 
HTX hci_center (HTX src)
{
 HTX x,x1,x2;

 x1 = hci_gen ("t","<center>");
 x2 = hci_gen ("t","</center>");
 x  = hci_gen ("xxx",x1,src,x2);

 return (x);
}

/******************************************************************************/
/** generate font tag
 * @param f pointer to string, face
 * @param s text size
 * @param c color in RGB
 * @param src pointer to html text
 * @return HTML font tag
 */
 
HTX hci_font (const char * f,NUM s,CLR c,HTX src)
{
 HTX x,x1,x2;
 PSX_STR t;

 strcpy (t,"<font");

 if (f)			hci_apf (t," face='%s'",f);
 if (s != -1)		hci_apf (t," size='%d'",s);
 hci_apf (t," color='#%06X'",c);

 strcat (t,">");

 x1 = hci_gen ("t",t);
 x2 = hci_gen ("t","</font>");
 x  = hci_gen ("xxx",x1,src,x2);

 return (x);
}

/******************************************************************************/
/** create HTML formatted table                                                
 * @param brd strength of border                                             
 * @param xs width                                                           
 * @param ys height                                                           
 * @param bgc background color                                                
 * @param aln alignment                                                       
 * @param val vertical alignment                                              
 * @param src content (rows...)                                               
 * @return HTML formatted table
 */
 
HTX hci_table (NUM brd,const char * xs,const char * ys,CLR bgc,const char * aln,const char * val,HTX src)
{
 HTX x,x1,x2;
 PSX_STR t;

 strcpy (t,"<table");

 if (brd != -1)		hci_apf (t," border='%d'",brd);
 if (xs)		hci_apf (t," width='%s'",xs);
 if (ys)		hci_apf (t," height='%s'",ys);
 if (bgc != CLR_NIL)	hci_apf (t," bgcolor='#%06X'",bgc);
 if (aln)		hci_apf (t," align='%s'",aln);
 if (val)		hci_apf (t," valign='%s'",val);

 strcat (t,">");     // t contains opening table tag with attributes

 x1 = hci_gen ("t",t);
 x2 = hci_gen ("t","</table>");
 x  = hci_gen ("xxx",x1,src,x2);  // x contains HTML formatted table definition

 return (x);
}

/******************************************************************************/
/** generate tr tag
 * @param src pointer to html text
 * @return HTML formatted table row
 */
 
HTX hci_tr (HTX src)
{
 HTX x,x1,x2;

 x1 = hci_gen ("t","<tr>");
 x2 = hci_gen ("t","</tr>");
 x  = hci_gen ("xxx",x1,src,x2);

 return (x);
}

/******************************************************************************/
/** generate table data in html text
 * @param xs width
 * @param ys height
 * @param aln alignment
 * @param val vertical alignment
 * @param csp column span
 * @param bgc backgroundcolor
 * @param src pointer to html text
 * @return HTML formatted table data
 */
 
HTX hci_td (const char * xs,const char * ys,const char * aln,const char * val,const char * csp,CLR bgc,HTX src)
{
 HTX x,x1,x2;
 PSX_STR t;

 strcpy (t,"<td");

 if (xs)		hci_apf (t," width='%s'",xs);
 if (ys)		hci_apf (t," height='%s'",ys);
 if (aln)		hci_apf (t," align='%s'",aln);
 if (val)		hci_apf (t," valign='%s'",val);
 if (csp)		hci_apf (t," colspan='%s'",csp);
 if (bgc != CLR_NIL)	hci_apf (t," bgcolor='#%06X'",bgc);

 strcat (t,">");

 x1 = hci_gen ("t",t);
 x2 = hci_gen ("t","</td>");
 x  = hci_gen ("xxx",x1,src,x2);

 return (x);
}

/******************************************************************************/
/** generate html form
 * @param act pointer to string, action
 * @param mth pointer to string, method
 * @return HTML formatted form
 */
 
HTX hci_form (const char * act,const char * mth,HTX src)
{
 HTX x,x1,x2;
 PSX_STR t;

 strcpy (t,"<form");

 if (act)		hci_apf (t," action='%s'",act);
 if (act)		hci_apf (t," method='%s'",mth);

 strcat (t,">");

 x1 = hci_gen ("t",t);
 x2 = hci_gen ("t","</form>");
 x  = hci_gen ("xxx",x1,src,x2);

 return (x);
}

/******************************************************************************/
/** generate input tag
 * @param name string pointer to name of input
 * @param type string pointer to type of input
 * @param val string pointer to input value
 * @param size string pointerto size of input
 * @param maxsize string pointerto maximum size of input
 * @return HTML formatted input tag
 */
 
HTX hci_input (const char * name,const char * type,const char * val,const char * size,const char * maxsize)
{
 HTX x;
 PSX_STR t;

 strcpy (t,"<input");

 if (type)		hci_apf (t," type='%s'",type);
 if (name)		hci_apf (t," name='%s'",name);
 if (val)		hci_apf (t," value='%s'",val);
 if (size)		hci_apf (t," size='%s'",size);
 if (maxsize)  hci_apf(t, " maxlength='%s'", maxsize);
 
 strcat (t,">");

 x = hci_gen ("t",t);

 return (x);
}

/******************************************************************************/
/** generate select statement
 * @param name pointer to string, name
 * @param size pointer to string, size
 * @param opt pointer to string, option
 * @return HTML formatted table
 */
 
HTX hci_select (const char * name,const char * size,const char * opt)
{
 HTX x,x1,x2,o;
 PSX_STR t;
 int i;

 strcpy (t,"<select");

 if (name)		hci_apf (t," name='%s'",name);
 if (size)		hci_apf (t," size='%s'",size);

 strcat (t,">");

 x1 = hci_gen ("t",t);
 x2 = hci_gen ("t","</select>");

 o = hci_gen (NULL);
 i = 0;

 while (str_tok (opt,&i,";",t))
 {
  HTX h;
  PSX_STR t1,t2;
  int j;
  BOOL v;

  j = 0;

  if (str_tok (t,&j,":",t1))
   if (str_tok (t,&j,":",t2))
    v = TRUE;
   else
    v = FALSE;
  else
   v = FALSE;

  if (v)
   h = hci_tag (hci_txt (t2),"option","value='%s'",t1);
  else
   h = hci_tag (hci_txt (t1),"option",NULL);

  hci_add (o,h);
 }

 x  = hci_gen ("xxx",x1,o,x2);

 return (x);
}

/******************************************************************************/
/** create HTML element textarea                                                      
 * @param name name of textarea                                                     
 * @param xs number of columns
 * @param ys number of rows                                                      
 * @param wr wrap
 * @param ro readonly                                                     
 * @param src text to display
 * @return HTML formatted textarea
 */
HTX hci_textarea (const char * name,const char * xs,const char * ys,const char * wr,BOOL ro,HTX src)
{
 HTX x,x1,x2;
 PSX_STR t;

 strcpy (t,"<textarea");

 if (name)		hci_apf (t," name='%s'",name);
 if (xs)		hci_apf (t," cols='%s'",xs);
 if (ys)		hci_apf (t," rows='%s'",ys);
 if (wr)		hci_apf (t," wrap='%s'",wr);
 if (ro)		hci_apf (t," readonly");

 strcat (t,">");

 x1 = hci_gen ("t",t);
 x2 = hci_gen ("t","</textarea>");
 x  = hci_gen ("xxx",x1,src,x2);

 return (x);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


