/******************************************************************************/
/* psx-hfi.c                                            Health Fund Interface */
/******************************************************************************/
/** @file psx-hfi.c Health Fund Interface - Source Code File
 * Functions implementing the check on validity of Health Fund code numbers.
 * supported Health Insurance Organisations are:
 * - AOK
 * - BARMER
 * - DAK
 * - GAE
 * - GEK
 * - HMK
 * - HEK
 * - HKK
 * - NEP
 * - KEH
 * - TKK
 * - BKK
 * - IKKUC
 * - IS_KV
 * - KKH
 * - BRU
 * - BUC
 * - BKN
 * - LAK
 * - SEK
 * - IKK
 */
 
#include "psx.h"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

typedef BOOL (* HFI_CPR)(HIC hic);
#define HFI_CHECKPROCEDURE(prc) BOOL prc (HIC hic)

#include "psx-hfi.inl"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

static struct
{
 HIO    hio;
 const char * sym;
 const char * name;
 HFI_CPR prc;
} hfi_hio [] =
{
 { HIO_AOK   ,"AOK"   ,"AOK"   ,cpr_aok     },
 { HIO_BARMER,"BARMER","BARMER",cpr_barmer  },
 { HIO_DAK   ,"DAK"   ,"DAK"   ,cpr_dak     },
 { HIO_GAE   ,"GAE"   ,"GAE"   ,cpr_gae     },
 { HIO_GEK   ,"GEK"   ,"GEK"   ,cpr_gek     },
 { HIO_HMK   ,"HMK"   ,"HMK"   ,cpr_hmk     },
 { HIO_HEK   ,"HEK"   ,"HEK"   ,cpr_hek     },
 { HIO_HKK   ,"HKK"   ,"HKK"   ,cpr_hkk     },
 { HIO_NEP   ,"NEP"   ,"NEP"   ,cpr_nep_keh },
 { HIO_KEH   ,"KEH"   ,"KEH"   ,cpr_nep_keh },
 { HIO_TKK   ,"TKK"   ,"TKK"   ,cpr_tkk     },
 { HIO_BKK   ,"BKK"   ,"BKK"   ,cpr_bkk     },
 { HIO_IKKUC ,"IKKUC" ,"IKKUC" ,NULL        },
 { HIO_IS_KV ,"IS_KV" ,"IS_KV" ,NULL        },
 { HIO_KKH   ,"KKH"   ,"KKH"   ,NULL        },
 { HIO_BRU   ,"BRU"   ,"BRU"   ,NULL        },
 { HIO_BUC   ,"BUC"   ,"BUC"   ,NULL        },
 { HIO_BKN   ,"BKN"   ,"BKN"   ,NULL        },
 { HIO_LAK   ,"LAK"   ,"LAK"   ,NULL        },
 { HIO_SEK   ,"SEK"   ,"SEK"   ,NULL        },
 { HIO_IKK   ,"IKK"   ,"IKK"   ,cpr_bkk     }
};

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** check existence of control procedure for specified health insurance organization
 * @param hio health insurance organization structure
 * @param prc pointer to procedure, contains pointer to control procedure after invocation
 * @return TRUE if control functions exists
 * @return FALSE if control function was not found
 */

static BOOL hfi_cpr_lookup (HIO hio,HFI_CPR * prc)
{
 IDX i;

 for (i = 0;i < sizeof (hfi_hio) / sizeof (hfi_hio [0]);i ++)
  if (hfi_hio [i].hio == hio)
  {
   if (prc)
    *prc = hfi_hio [i].prc;
   return (TRUE);
  }

 return (FALSE);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** get name of health insurance organisation
 * @param hio health insurance organisation
 * @param sym contains health insurance organisation name after invocation
 * @return TRUE if name was found
 * @return FALSE if name was not found
 */
 
BOOL hfi_hio_enc (HIO hio,PSX_STR sym)
{
 IDX i;

 if (sym == NULL)
  return (FALSE);

 for (i = 0;i < sizeof (hfi_hio) / sizeof (hfi_hio [0]);i ++)
  if (hfi_hio [i].hio == hio)
  {
   strcpy (sym,hfi_hio [i].sym);
   return (TRUE);
  }

 return (FALSE);
}

/******************************************************************************/
/** get HIO code of health insurance organisation
  * @param hio health insurance organisation code, contains code of organisation after invocation
  * @param sym health insurance organisation name
  * @return TRUE if health insurance organisation code was found
  * @return FALSE if code was not found
 */
 
BOOL hfi_hio_dec (HIO * hio,PSX_STR sym)
{
 IDX i;
 PSX_STR t;

 if (hio == NULL || sym == NULL)
  return (FALSE);

 strcpy (t,sym);
 str_toupper (t);

 for (i = 0;i < sizeof (hfi_hio) / sizeof (hfi_hio [0]);i ++)
  if (strcmp (hfi_hio [i].sym,t) == 0)
  {
   *hio = hfi_hio [i].hio;
   return (TRUE);
  }

 return (FALSE);
}

/******************************************************************************/
/** list options, seperate by ':'
 * @param opt option list string
 */
 
BOOL hfi_hio_lst_hml (PSX_STR opt)
{
 IDX i;

 if (opt == NULL)
  return (FALSE);

 strcpy (opt,"");

 for (i = 0;i < sizeof (hfi_hio) / sizeof (hfi_hio [0]);i ++)
 {
  if (i > 0)
   strcat (opt,";");

  strcat (opt,hfi_hio [i].sym);
  strcat (opt,":");                 // separator
  strcat (opt,hfi_hio [i].name);
 }

 return (TRUE);
}

/******************************************************************************/
/** invoke HIC check procedure
 * @param hio HIO health insurance organisation
 * @param hic HIC health insurance code
 * @param hiv HIV health insurance validation
 */
 
BOOL hfi_check (HIO hio,HIC hic,HIV * hiv)
{
 HFI_CPR prc;

 if (hic == NULL)
 {
  if (hiv)
   *hiv = HIV_ERROR;
  return (FALSE);
 }

 if (!hfi_cpr_lookup (hio,&prc)) 
 {
  if (hiv)
   *hiv = HIV_UNKNOWN;
  return (TRUE);
 }

 if (!prc (hic))      // call check procedure <-> HFI_CHECKPROCEDURE (prc)
 {
  if (hiv)
   *hiv = HIV_ERROR;
  return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/* Main                                                                       */
/******************************************************************************/
/** main function of health insurance interface
 * @param argc number of arguments in argv[]
 * @param argv pointer to array of character, like in every main function
 */
 
int hfi_test (int argc, char* argv[])
{
 HIO hio;
 HIC hic;
 HIV hiv;

 if (argc != 3)
 {
  fprintf(stderr, "Usage: %s KRK KRANKENVERSICHERTENNUMMER\n", argv[0]);
  return 10;
 }

 hio = atoi (argv[1]);
 strcpy (hic,argv [2]);
 printf ("Checking KRK: %d, Code: %s.. ", hio, hic);

 if (hfi_check(hio, hic, &hiv))
 {
  switch (hiv)
  {
   case HIV_OK:
    printf("OK\n");
    break;
   case HIV_ERROR:
    printf("ERROR\n");
    break;
   case HIV_UNKNOWN:
    printf("UNKNOWN\n");
    break;
   default:
    printf("FATAL ERROR: Wrong value in hiv!\n");
  }
 }
 else
 {
  printf("krk_check returned FALSE!\n");
 }

 return (0);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


