/******************************************************************************/
/* psx-hfi.inl                         Health Fund Interface                  */
/******************************************************************************/
/** @file psx-hfi.inl Health Fund Interface - Inline Source Code File
 * Functions implementing the check on validity of Health Fund code numbers.
 * There are checkprocedures for the following Health Funds (in this file):
 * - AOK
 * - Barmer Ersatzkasse
 * - DAK
 * - GAE
 * - GEK
 * - HMK
 * - HEK
 * - HKK (new and old)
 * - KEH
 * - TKK
 * - BKK
 */
 

 
/******************************************************************************/
/* Private                                                            support */
/******************************************************************************/
/** check whether health insurance number consists of digits only
 * @param hic string pointer to health insurance number
 */
static BOOL is_number(char* hic)
{
 int i;
 int len = strlen(hic);

 for(i = 0; i < len; i++)
 {
  if (hic[i] < '0' || hic[i] > '9')
  {
   return FALSE;
  }
 }
 return TRUE;
}

/******************************************************************************/
/** get digit at position d in integer n
 * @param n integer
 * @param d position of digit (starting from right end at 0)
 * @return digit at position d of number n
 */
 
static int stelle(int n, int d)
{
 int i;
 int power = 1;

 for(i = 1; i <= d; i++)
 {
  power *= 10;
 }
 return (n/power) % 10;
}

/******************************************************************************/
/** get sum of digits of number n, maximum 11 digits
 * @param n integer of 11 digits (e.g. health insurance number)
 * @return sum of digits
 */
 
static int int_quersumme(int n)
{
 int i;
 int qs = 0;

 for(i = 0; i <= 10; i++)
 {
  qs += stelle(n, i);
 }

 return qs;
}

/******************************************************************************/
/** increment number by 10
 * @param c string pointer to health insurance number (consisting of 10 digits)
 */
 
static void add10(char* c)
{
 int i;

 for(i = 8; i >= 0; i--)
 {
  c[i]++;

  if(c[i] == '9' + 1)  // if digit was 9 set to 0 and increment next digit
  {
   c[i] = '0';
  }
  else
  {
   break;
  }
 }
}

/******************************************************************************/
/* Private                                                HIO specific checks */
/******************************************************************************/
/** check procedure for insurance codes of aok health insurance organisation
 *  BOOL cpr_aok (HIC hic)
 * @param hic Health Insurance Code
 */
 
static HFI_CHECKPROCEDURE (cpr_aok)
{
 int i;
 int qs = 0;

 if (!is_number (hic))
  return (FALSE);

 if (strlen (hic) != 13)
  return (FALSE);

 /* digits 1 to 12 are multiplied with 2, 1, 2, 1, ... 
    The sum of digits of each product are added        */

 for (i = 0;i <= 11;i++)
 {
  qs += int_quersumme ((hic [i] - '0') * (((i + 1) % 2) + 1));
  
 }

 /* divide sum with 10
    rest is checksum */

 if (qs % 10 != hic [12] - '0')
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** check procedure for insurance codes of barmer health insurance organisation
 *  BOOL cpr_barmer (HIC hic)
 * @param hic Health Insurance Code 
 */
static HFI_CHECKPROCEDURE (cpr_barmer) 
{
 int i;
 int sum = 0;
 int mult[9] = { 3, 2, 7, 6, 5, 4, 3, 2 };

 if (!is_number (hic))
  return (FALSE);

 if(strlen(hic) != 11 && strlen(hic) != 9)
  return (FALSE);

 /* multiply digits 1 to 8 with 3, 2, 7, 6, 5, 4, 3, 2 
    add the results                                    */

 for (i = 0;i <= 7;i++)
 {
  sum += (hic[i] - '0')*mult[i];
 }

 /* divide the sum by 11 
    subtract the rest from 11
    compare the result with the ninth digit */

 if (11 - (sum % 11) != hic [8] - '0')
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** check procedure for insurance codes of dak health insurance organisation
 *  BOOL cpr_dak (HIC hic)
 * @param hic Health Insurance Code
 */
static HFI_CHECKPROCEDURE (cpr_dak)
{
 int i;
 int qs = 0;

 if (!is_number (hic))
  return (FALSE);

 if(strlen(hic) != 12)
  return (FALSE);

 /* multiply the digits 1 to 8 with 1, 2, 1, 2, ... 
    if a single product is has more than one digit, get the sum of its digits
    sum up the results                                                        */

 for (i = 0; i <= 7; i++)
 {
  qs += int_quersumme((hic[i] - '0')*((i%2) + 1));
 }

 /* subtract the result from next higher number ending with 0
    the result is used as checksum                           */

 if ((10 - (qs % 10)) % 10 != hic[8] - '0')
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** check procedure for insurance codes of gae health insurance organisation
 *  BOOL cpr_gae (HIC hic)
 * @param hic Health Insurance Code
 */
 
static HFI_CHECKPROCEDURE (cpr_gae)
{
 int i;
 int sum = 0;

 if (!is_number (hic))
  return (FALSE);

 if (strlen(hic) != 11)
  return (FALSE);

 /* add the first 10 digits */

 for (i = 0; i <= 9; i++)
 {
  sum += hic[i] - '0';
 }

 /* subtract the result from the next higher number ending on 0
    The result of this operation is the checksum           */

 if ((10 - (sum % 10)) % 10 != hic[10] - '0')
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** heck procedure for insurance codes of gek health insurance organisation
 *  BOOL cpr_gek (HIC hic)
 * @param hic Health Insurance Code
 */
 
static HFI_CHECKPROCEDURE (cpr_gek)
{
 int i;
 int sum = 0;

 if (!is_number (hic))
  return (FALSE);

 if (strlen (hic) != 10)
  return (FALSE);

 /* If the first digit of the insurance code is a 0, use variant A,
    else use B                                                       */ 

 if (hic [0] == '0')
 {
  /* A: */

  /* multiply digits 1 to 7 with 2, 1, 2, 1, 2, ...
     add the resulting products                     */

  for(i = 0; i <= 6; i++)
  {
   sum += (hic[i] - '0')*(((i + 1)%2) + 1);
  }

  /* subtract sum from the next higher number ending on 0
     the result is used as checksum                  */

  if ((10 - (sum % 10)) % 10 != hic[7] - '0')
   return (FALSE);
 }
 else
 {
  /* B: */

  /* multiply digits 1 to 7 with 1, 2, 1, 2, ... 
     If a single product consists of more than one digit,
     get the sum of its digits.
     sum up the results.                                  */

  for(i = 0; i <= 6; i++)
  {
   sum += int_quersumme((hic[i] - '0')*((i%2) + 1));
  }

  /* divide sum by 10 and add 1 to rest
     this will be the checksum     */

  if ((sum % 10) + 1 != hic[7] - '0')
   return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/** check procedure for insurance codes of hmk health insurance organisation
 *  BOOL cpr_hmk (HIC hic)
 * @param hic Health Insurance Code
 */
 
static HFI_CHECKPROCEDURE (cpr_hmk)
{
 int i;
 int sum = 0;
 int checksum;

 if (!is_number (hic))
  return (FALSE);

 if (strlen(hic) != 9 && strlen(hic) != 10)
  return (FALSE);

 if (hic[0] == '0' && hic[1] == '0')
 {
  /* A: */

  /* multiply digits 3 to 8 with 7, 6, 5, 4, 3, 2
     and add the products                         */

  for(i = 2; i <= 7; i++)
  {
   sum += (hic[i] - '0')*(9 - i);
  }

  /* divide sum by 11 and subtract the rest of division from 11
     last digit of the result will be the checksum

     Caused by a carry over the counter, the checksum might be
     increased by 1.                                                */

  checksum = (11 - (sum % 11)) % 10;

  if (!((checksum == hic[8] - '0' || (checksum + 1) % 10 == hic[8] - '0')))
   return (FALSE);
 }
 else
 {
  /* B: */

  /* multiply digits 1 to 9 with 1, 2, 1, 2, ...
     add the sum of digits of each product       */

  for(i = 0; i <= 8; i++)
  {
   sum += int_quersumme((hic[i] - '0')*((i%2) + 1));
  }

  /* last digit of the sum is checknumber */

  if(sum % 10 != hic[9] - '0')
   return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/** check procedure for insurance codes of hek health insurance organisation
 *  BOOL cpr_hek (HIC hic)
 * @param hic Health Insurance Code
 */
 
static HFI_CHECKPROCEDURE (cpr_hek)
{
 int i;
 int sum = 0;
 char c[11];
 int checksum;

 if (!is_number (hic))
  return (FALSE);

 if (strlen(hic) != 10 || (hic[6] != '0' && hic[6] != '7'))
  return (FALSE);

 strcpy(c, hic);

 while (TRUE)
 {
  /* multiply digits 1 to 9 with 10, 9, 8, 7, 6, 5, 4, 3, 2
     add the results                                        */
  sum = 0;

  for(i = 0; i <= 8; i++)
  {
   sum += (hic[i] - '0')*(10 - i);
  }

  /* divide the sum of products by 11,
     if the remaining rest equal to 0, the checknumber is 0.
     If the remaining rest greater than 1, the checknumber is 11 - rest
     else if rest is 1, add 10 to health insurance code and start again */

  checksum = sum % 11;

  if(checksum != 1)
  {
   break;
  }

  add10 (c);
 }

 if(checksum != c[9] - '0')
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** check procedure for insurance codes of hkk health insurance organisation
 *  BOOL cpr_hkk_neu (HIC hic)
 * @param hic Health Insurance Code
 */
static HFI_CHECKPROCEDURE (cpr_hkk_neu)
{
 int i;
 int sum = 0;

 if (!is_number (hic))
  return (FALSE);

 /* new modus operandi version 1 (wrong mode): */

 /* multiply digits 1 to 9 with 1, 2, 1, 2, ...
    add every last digit of each product       */

 for(i = 0; i <= 8; i++)
 {
  sum += ((hic[i] - '0')*((i%2) + 1)) % 10;
 }

 /* last digit of sum is checknumber */

 if(sum % 10 != hic[9] - '0')
 {
  /* new modus operandi version 2: */

  /* multiply digits 1 to 9 with 1, 2, 1, ...
     add the sum of digits of each product    */

  sum = 0;

  for(i = 0; i <= 8; i++)
  {
   sum += int_quersumme((hic[i] - '0')*((i%2) + 1));
  }

  /* last digit of result is checknumber */

  if(sum % 10 != hic[9] - '0')
   return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/** check procedure for insurance codes of hkk health insurance organisation
 *  BOOL cpr_hkk (HIC hic)
 * @param hic Health Insurance Code
 */
 
static HFI_CHECKPROCEDURE (cpr_hkk)
{
 int i;
 int qs = 0;

 if (!is_number (hic))
  return (FALSE);

 if (strlen (hic) == 10)
  return cpr_hkk_neu (hic);

 if (strlen(hic) != 9 && strlen(hic) != 7)
  return (FALSE);

 /* the 7th digit is checknumber
    multiply digits 1 to 6 with 1, 2, 1, 2, ...
    If a product consists of more than one digit, get the sum of its digits.
    Add the retrieved results. */

 for(i = 0; i <= 5; i++)
 {
  qs += int_quersumme((hic[i] - '0')*((i%2) + 1));
 }

 /* subtract sum from the next greater number ending on 0,
    result is the checknumber. */

 if ((10 - (qs % 10)) % 10 != hic[6] - '0')
 {
  /* old procedure: */
  int sum = 0;
  int mult[6] = { 1, 7, 3, 1, 7, 3 };

  /* multiply digits 1 to 6 with 1, 7, 3, 1, 7, 3
     Add each product                             */

  for(i = 0; i <= 5; i++)
  {
   sum += (hic[i] - '0')*mult[i];
  }

  /* Divide the sum of products by 10 
     subtract the remaining rest from 10
     checknumber is the last digit of the result */

  if ((10 - (sum % 10)) % 10 != hic[6] - '0')
   return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/** check procedure for insurance codes of nep keh health insurance organisation
 *  BOOL cpr_nep_keh (HIC hic)
 * @param hic Health Insurance Code
 */
 
static HFI_CHECKPROCEDURE (cpr_nep_keh)
{
 int i;
 int sum = 0;

 if (!is_number (hic))
  return (FALSE);

 if (strlen (hic) == 9)
 {
  /* old procedure: */

  /* multiply digits 1 to 6 with 1, 7, 3, 1, 7, 3
     sum up each product                           */

  int mult [6] = { 1, 7, 3, 1, 7, 3 };

  for(i = 0; i <= 5; i++)
  {
   sum += (hic[i] - '0')*mult[i];
  }

  /* Divide the sum of products by 10, 
     subtract the remaining rest from 10
     the last digit of the result is the checknumber */

  if ((10 - (sum % 10)) % 10 != hic[6] - '0')
   return (FALSE);
 }
 else
 {
  if (strlen (hic) == 10)
  {
   /* new procedure: */

   /* multiply digits 1 to 9 with 1, 2, 1, ...
      If a result consists of more than one digit, get the sum of its digits
      add the retrieved results                                             */

   for (i = 0; i <= 8; i++)
   {
    sum += int_quersumme((hic[i] - '0')*((i%2) + 1));
   }

   /* the last digit of the result is checknumber */

   if(sum % 10 != hic[9] - '0')
    return (FALSE);
  }
  else
   return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/** check procedure for insurance codes of tkk health insurance organisation
 *  BOOL cpr_tkk (HIC hic)
 * @param hic Health Insurance Code
 */
 
static HFI_CHECKPROCEDURE (cpr_tkk)
{
 int i;
 int sum = 0;
 int checksum;

 if (!is_number (hic))
  return (FALSE);

 if (strlen (hic) != 10)
  return (FALSE);

 /* multiply digits 1 to 9 with 10, 9, 8, 7, 6, 5, 4, 3, 2
    add the single products                                 */

 for(i = 0; i <= 8; i++)
 {
  sum += (hic[i] - '0')*(10 - i);
 }

 /* divide the sum of products by 11, 
    subtract the remaining rest from 11 
    last digit of the result of the subtraction is checknumber */

 checksum = (11 - (sum % 11)) % 10;

 if (!((checksum == hic[9] - '0' || (checksum + 5) % 10 == hic[9] - '0')))
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** check procedure for insurance codes of bkk health insurance organisation
 *  BOOL cpr_bkk (HIC hic)
 * @param hic Health Insurance Code
 */
 
static HFI_CHECKPROCEDURE (cpr_bkk)
{
 int i;
 int sum = 0;

 if (!is_number (hic))
  return (FALSE);

 if (strlen(hic) != 10)
  return (FALSE);

 /* multply digits 1 to 9 with 1, 2, 1, ... 
    add the sums of digits of each product */

 for (i = 0; i <= 8; i++)
 {
  sum += int_quersumme((hic[i] - '0')*((i%2) + 1));
 }

 /* last digit of result is checknumber */

 if (sum % 10 != hic[9] - '0')
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


