/******************************************************************************/
/* psx-int.c                                           Command Line Interface */
/******************************************************************************/
/** @file psx-int.c Command Line Interface - Source Code File
 * Functions supporting the Command Line Interface functionalities such as
 * getting options from commandline (of psx programme call) for every
 * possible command.
 */
 
#include "psx.h"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** set encipher type (raw or hex)
 * @param sym string pointer to name of encipherment type
 * @param enc BYTE pointer to encipher type
 */

static BOOL int_enc_get (const char * sym,BYTE * enc)
{
 if (sym == NULL || enc == NULL)
  return (FALSE);

 if (strcmp (sym,"raw") == 0)
  *enc = CPE_RAW; //OPT_ENC_RAW;
 else
  if (strcmp (sym,"hex") == 0)
   *enc = CPE_HEX; // OPT_ENC_HEX;
  else
   return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** fetch encipher type (raw or text)
 * @param sym pointer to string, symbolname
 * @param enc pointer to string, encipherment type
 */

static BOOL int_enc_fetch (const char * sym,BYTE * enc)
{
 PSX_STR s,t;

 if (sym == NULL || enc == NULL)
  return (FALSE);

 sprintf (s,"%s:s",sym);

 if (!sci_opt_get (s,t,"hex"))      // get option for sym:s
 {
  *enc = CPE_HEX; // OPT_ENC_HEX;
  return (TRUE);
 }

 if (!int_enc_get (t,enc))          // get encipher type (raw or hex)
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** set input character set (ISO or DIN)
 * @param sym string pointer to name of character set
 * @param ics string pointer to character set code
 */

static BOOL int_ics_get (char * sym,ICS * ics)
{
 if (sym == NULL || ics == NULL)
  return (FALSE);

 if (strcmp (sym,"din") == 0)    /* DIN 66003 */
  *ics = ICS_DIN;
 else
  if (strcmp (sym,"iso") == 0)
   *ics = ICS_ISO;              /* ISO 8859-1 */
  else
   return (FALSE);

 return (TRUE);
}
/******************************************************************************/
/** fetch input character set type (DIN or ISO)
 * @param sym pointer to string, symbolname
 * @param ics string pointer, contains code for character set after invocation
 */

static BOOL int_ics_fetch (const char * sym,ICS * ics)
{
 PSX_STR s,t;

 if (sym == NULL || ics == NULL)
  return (FALSE);

 sprintf (s,"%s:s",sym);

 if (!sci_opt_get (s,t,"iso"))      /* get option for sym:s */
 {
  *ics = ICS_ISO;
  return (TRUE);
 }

 if (!int_ics_get (t,ics)) 
  return (FALSE);

 return (TRUE);
}


/******************************************************************************/
/** get options, specified or default
 * @param cfg pointer to string, configuration
 * @param spc pointer to string, specification
 * @param def pointer to string, default settings
 * @param tgt pointer to target string
 */
 
static BOOL int_opt_get (const char * cfg,const char * spc,const char * def,PTR tgt) //char * tgt)
{
 PSX_STR s,t;

 if (spc == NULL || tgt == NULL)
  return (FALSE);

 s [0] = 0;

 if (cfg)
 {
  if (!psx_cfg_get (cfg,s))                 /*  get configuration */
   if (def)
    strcpy (s,def);                         /* get default config */
   else
    strcpy (s,"");
  else
   psx_put_v ("loading default: %s",cfg);
 }
 else
 {
  if (def)
   strcpy (s,def);
  else
   strcpy (s,"");
 }

 if (!sci_opt_get (spc,tgt,s))     /* get option from spc or set default from s */
  return (FALSE);


 return (TRUE);
}

#define INT_OPT_GET(cfg,spc,tgt,msg,ret)	\
if (!int_opt_get ((cfg),(spc),NULL,(tgt)))	\
{						\
 psx_put ("%s",(msg));				\
 return (ret);					\
}

/******************************************************************************/
/** get first character of string sym and make sure chr contains character c
 * @param sym pointer to string, input symbol
 * @param chr pointer to string, character
 */
 
static BOOL int_opt_sel_chr (char * sym,char * chr,char * c)
{
 if (sym == NULL || chr == NULL || c == NULL)
  return (FALSE);

 if (strlen (sym) != 1)
  return (FALSE);

 *c = sym [0];

 if (strchr (chr,*c) == NULL)
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/* cmd_ctn_enc(void):                                                         */
/**  get options for encoding and execute command CMD_CTN_ENC
 */

static SCI_COMMAND (cmd_ctn_enc)
{
 PSX_CSP csp;
 char t [1000],fmt_def [1000];
 int r;

 cmd_csp_reset (&csp);               /* reset command specification */
 csp.cmd = CMD_CTN_ENC;              /* new command ist CMD_CTN_ENC */

 INT_OPT_GET (PSX_CFG_FSP,"f:s",csp.ctx.ctn.enc.ifs,"missing input format specification",-1);            

 if (!sci_opt_get ("i:s",csp.ctx.ctn.enc.idf,"") && !(csp.ctx.ctn.enc.bat))         /*:BG: d->i*/
 {
  psx_put_v ("no input data file, ok");
 }

 if (!sci_opt_get ("o:s",csp.ctx.ctn.enc.odf,"") && !(csp.ctx.ctn.enc.bat))
 {
  psx_put_v ("no output data file, ok");
 }

 /*:BG:begin*/
 if(!sci_opt_get("d:s",csp.ctx.ctn.enc.del,";"))
 {
     strcpy(csp.ctx.ctn.enc.del,"");
 }
 /*:BG:end*/
 
 sci_opt_get ("b:b",&csp.ctx.ctn.enc.bat,NULL);     /* batchmode */

 if (csp.ctx.ctn.enc.bat && (strlen(csp.ctx.ctn.enc.idf) < 1))
 {
   psx_put ("missing input data file");
   return (-1);
 }

 if (csp.ctx.ctn.enc.bat && (strlen(csp.ctx.ctn.enc.odf) < 1))
 {
   psx_put ("missing output data file");
   return (-1);
 }
                       
 sci_opt_get ("u:s",csp.ctx.ctn.enc.ofs,"");       /* output format specification */
 sci_opt_get ("p:i",&csp.ctx.ctn.enc.pos,"-1");    /* start position			  */	
 sci_opt_get ("h:b",&csp.ctx.ctn.enc.hdn,NULL);    /* headings                    */

 if (!int_enc_fetch ("e",&csp.ctx.ctn.enc.enc))    /* get encipher type           */
 {                                                 /* (raw or hex)                */
  psx_put ("invalid encoding specification");     
  return (-1);
 }

 if (!int_ics_fetch ("a", &csp.ctx.ctn.enc.ics) )   /* get input character set    */
 {                                                  /* DIN or ISO                 */
   psx_put ("invalid character set specification");
   return (-1);
 }   
  
 sci_opt_get ("k:s",csp.ctx.ctn.enc.key,"");        /* key k:s                    */

 r = cmd_execute (&csp);                            /* execute command       */

 return (r);
}

/******************************************************************************/
/* cmd_ctn_dec(void):                                                         */
/** get options for decoding and execute command CMD_CTN_DEC
 */

static SCI_COMMAND (cmd_ctn_dec)
{
 PSX_CSP csp;
 char t [1000],fmt_def [1000];
 int r;

 cmd_csp_reset (&csp);         /* reset command specification */
 csp.cmd = CMD_CTN_DEC;        /* new command is CMD_CTN_DEC  */

 INT_OPT_GET (PSX_CFG_FSP,"f:s",csp.ctx.ctn.dec.ifs,"missing input format specification",-1);
 
 if (!sci_opt_get ("i:s",csp.ctx.ctn.dec.idf,""))      /* input data file  */   /*:BG: d->i*/
 {
  psx_put_v ("no input data file, ok");
 }

 if (!sci_opt_get ("o:s",csp.ctx.ctn.dec.odf,""))      /* output data file */
 {
  psx_put_v ("no output data file, ok");
 }

 sci_opt_get ("u:s",csp.ctx.ctn.dec.ofs,"");           /* output format specification */

 sci_opt_get ("b:b",&csp.ctx.ctn.dec.bat,NULL);        /* batchmode                   */
 sci_opt_get ("h:b",&csp.ctx.ctn.dec.hdn,NULL);        /* headings                    */

 if (!int_enc_fetch ("e",&csp.ctx.ctn.dec.enc))        /* get encoding specification  */
 {
  psx_put ("invalid encoding specification");
  return (-1);
 }
 
 if (!int_ics_fetch ("a", &csp.ctx.ctn.dec.ics) )      /* get input character set     */
 {                                                     /* DIN or ISO                  */
   psx_put ("invalid character set specification");
   return (-1);
 }  
 
 sci_opt_get ("k:s",csp.ctx.ctn.dec.key,"");           /* key                         */

 r = cmd_execute (&csp);                               /* execute next command        */

 return (r);
}

/******************************************************************************/
/* cmd_ctn_cvt(void):                                                         */
/** get options for cryptographic conversion and execute command CMD_CTN_CVT
 */

static SCI_COMMAND (cmd_ctn_cvt)
{
 PSX_CSP csp;
 char t [1000],* p;
 int r;

 cmd_csp_reset (&csp);      /* reset command specificaton */
 csp.cmd = CMD_CTN_CVT;

 INT_OPT_GET (PSX_CFG_FSP,"f:s",csp.ctx.ctn.cvt.ifm,"missing format specification",-1);

 SCI_OPT_GET ("u:s",csp.ctx.ctn.cvt.ofm,"","missing output format file",-1);  /* output file format */
 SCI_OPT_GET ("i:s",csp.ctx.ctn.cvt.ifn,"","missing input file",-1);          /* input file         */
 SCI_OPT_GET ("o:s",csp.ctx.ctn.cvt.ofn,"","missing output file",-1);         /* output file        */

 /*:BG:begin*/
 if(!sci_opt_get("d:s",csp.ctx.ctn.cvt.del,";"))
 {
     strcpy(csp.ctx.ctn.cvt.del,0);
 }
 /*:BG:end*/
 
 if (!sci_opt_get ("e:s",t,""))                             /* encoding */
 {
  psx_put ("missing encoding translation specification");
  return (-1);
 }

 if ((p = strchr (t,':')) == NULL)                          /* get pointer to first ':' */                     
 {
  psx_put ("invalid encoding translation specification");
  return (-1);
 }

 *p = '\0';                                       /* change ':' to '\0' */
 p++;                                             /* go ahead           */

 if (!int_enc_get (t,&csp.ctx.ctn.cvt.iec))       /* get input encoding specification */
 {
  psx_put ("invalid input encoding specification");
  return (-1);
 }

 if (!int_enc_get (p,&csp.ctx.ctn.cvt.oec))       /* get output encoding specification */
 {
  psx_put ("invalid output encoding specification");
  return (-1);
 }
 
 if (!int_ics_fetch ("a", &csp.ctx.ctn.cvt.ics) )    /* get input character set */
 {                                                   /* DIN or ISO              */
   psx_put ("invalid character set specification");
   return (-1);
 }  

printf ("ifm:'%s'\n",csp.ctx.ctn.cvt.ifm);        
printf ("ofm:'%s'\n",csp.ctx.ctn.cvt.ofm);
printf ("ifn:'%s'\n",csp.ctx.ctn.cvt.ifn);
printf ("ofn:'%s'\n",csp.ctx.ctn.cvt.ofn);
printf ("iec:'%02X'\n",csp.ctx.ctn.cvt.iec);
printf ("oec:'%02X'\n",csp.ctx.ctn.cvt.oec);

 r = cmd_execute (&csp);                          /* execute command */

 return (r);
}

/******************************************************************************/
/* Private                                                                pid */
/******************************************************************************/
/** get options and execute PID request.                       
 * Command line options are read using sci_opt_get() and checked for consistency.
 * Actual execution of the request is delegated to cmd_execute()
 */
 
static SCI_COMMAND (cmd_pid_req)
{
 PSX_CSP csp;									  /* command specification */
 PSX_STR t;										  /* temporary string      */
 int r;

 cmd_csp_reset (&csp);                            /* reset command specification */
 csp.cmd = CMD_PID_REQ;                           /* command specified by byte   */
												  /* constant				     */

 /* get command line options */
 sci_opt_get ("b:b",&csp.ctx.pid.req.bat,NULL);   /* batch mode                  */

 sci_opt_get ("i:s",csp.ctx.pid.req.dat,"");      /* input data file			 */
 
 /*:BG:begin*/
 /* check if delimiter was set using the -d option and copy it to
  * csp structure 
  */
 if(!sci_opt_get("d:s",&csp.ctx.pid.req.del,";"))
 {
     strcpy(csp.ctx.pid.req.del,"");
 }
 /*:BG:end*/
 
 if (!sci_opt_get ("t:s",csp.ctx.pid.req.otf,"")) /* output trace file		*/
  psx_put_v ("missing output trace file");

 if (!sci_opt_get ("f:s",csp.ctx.pid.req.fsp,"")) /* format specification   */
  psx_put_v ("no format specification");

 sci_opt_get ("p:n",&csp.ctx.pid.req.pos,"0");    /* get start position     */

 sci_opt_get ("F:b",&csp.ctx.pid.req.frc,NULL);   /* force PID generation   */

 if (sci_opt_get ("s:s",t,""))					  /* sureness specification */
 {
  str_trim (t);

  if (strlen (t) != 1)
  {
   psx_put ("invalid sureness specification");
   return (-1);
  }

  switch (t [0])
  {                                                 /* get sureness */
   case '-':	csp.ctx.pid.req.sur = 0;	break;
   case '+':	csp.ctx.pid.req.sur = 1;	break;
   default:
    psx_put ("invalid sureness specification");
    return (-1);
  }
 }
 else
 {
  psx_put ("missing sureness specification");
  return (-1);
 }
 
 /* check consistency of options */
 if (csp.ctx.pid.req.bat)					/* check if batch mode is set	*/
 {
  if (str_isnil (csp.ctx.pid.req.dat))      /* check if input data file was given */
  {
   psx_put ("data file required for batch mode");
   return (-1);
  }
 
 }
 else										/* batch mode not set			*/
 {
  if (!str_isnil (csp.ctx.pid.req.fsp))     /* check if format file was given */
  {
   psx_put ("format specification only for batch mode");
   return (-1);
  }
 }

 if (!int_ics_fetch ("a", &csp.ctx.pid.req.ics) )    /* get input character set */
 {                                                   /* DIN or ISO              */
   psx_put ("invalid character set specification");
   return (-1);
 }  
 
 r = cmd_execute (&csp);							/* call command execution   */

 return (r);
}

/******************************************************************************/
/* Private                                                                gui */
/******************************************************************************/
/* cmd_gui_cgi(void):                                                         */
/** get options for graphical user interface and execute command CMD_GUI_CGI
 */

static SCI_COMMAND (cmd_gui_cgi)
{
 PSX_CSP csp;
 int r;

 cmd_csp_reset (&csp);                             /* reset command specification */
 csp.cmd = CMD_GUI_CGI;

 sci_opt_get ("s:b",&csp.ctx.gui.cgi.sys,NULL);    /* enable system panel         */
 sci_opt_get ("d:b",&csp.ctx.gui.cgi.dbg,NULL);    /* enable debugging            */

 
 r = cmd_execute (&csp);                           /* execute command             */

 return (r);
}

/******************************************************************************/
/* Private                                                                dat */
/******************************************************************************/
/* cmd_dat_gen(void):                                                         */
/** get options for data generation and execute command CMD_DAT_GEN
 */

static SCI_COMMAND (cmd_dat_gen)
{
 PSX_CSP csp;
 char t [1000];
 int r;
 BOOL b;

 cmd_csp_reset (&csp);                 // reset command specification
 csp.cmd = CMD_DAT_GEN;

 INT_OPT_GET (NULL,"f:s",csp.ctx.dat.gen.fsp,"missing format specification",-1);
 INT_OPT_GET (NULL,"o:s",csp.ctx.dat.gen.out,"missing output file specification",-1);
 INT_OPT_GET (NULL,"n:n",&csp.ctx.dat.gen.num,"missing record number",-1);

 r = cmd_execute (&csp);

 return (r);
}

/******************************************************************************/
/* Private                                                                dbs */
/******************************************************************************/
/* cmd_dbs_gen(void):                                                         */
/** get options or database generation and execute command CMD_DBS_GEN
 */

static SCI_COMMAND (cmd_dbs_gen)
{
 PSX_CSP csp;
 char t [1000];
 int r;
 BOOL b;

 cmd_csp_reset (&csp);
 csp.cmd = CMD_DBS_GEN;

 INT_OPT_GET (NULL,"s:s",csp.ctx.dbs.gen.sch,"missing schema specification",-1);

 if (!sci_opt_get ("d:b",&b,NULL))
  b = FALSE;

 csp.ctx.dbs.gen.del = b;

 r = cmd_execute (&csp);

 return (r);
}

/******************************************************************************/
/* cmd_dbs_sts(void):                                                         */
/** get options for database status query ad execute command CMD_DBS_STS
 */

static SCI_COMMAND (cmd_dbs_sts)
{
 PSX_CSP csp;
 int r;

 cmd_csp_reset (&csp);
 csp.cmd = CMD_DBS_STS;

 r = cmd_execute (&csp);

 return (r);
}

/******************************************************************************/
/* Private                                                                dbs */
/******************************************************************************/
/* cmd_cfg(void):                                                             */
/** get options for configuration and execute command CMD_CFG
 */

static SCI_COMMAND (cmd_cfg)
{
 PSX_CSP csp;
 PSX_STR t;
 MTM mtm;
 int r;

 cmd_csp_reset (&csp);                           /* reset command specification */
 csp.cmd = CMD_CFG;

 if (sci_opt_get ("s:b",&csp.ctx.cfg.sch,NULL))  /* get schema                  */
  ;

 if (sci_opt_get ("m:b",&csp.ctx.cfg.mat,NULL))  /* matching                    */
 {
  if (!sci_opt_get ("q:s",t,NULL))               /* filter (KSXO)               */
  {
   psx_put ("missing filter specification");
   return (-1);
  }

  if (!mti_mtm_dec (&mtm,t))
  {
   psx_put ("invalid filter specification");
   return (-1);
  }

  if (!mti_mtm_enc (mtm,t))
   printf ("error\n");
  else
   printf ("mtm test: %s\n",t);

  csp.ctx.cfg.mtm = mtm;
 }

 r = cmd_execute (&csp);

 return (r);
}

/******************************************************************************/
/* Private                                                               Main */
/******************************************************************************/
/** without options and parameters psx can not be executed, give an error message
 */
 
SCI_COMMAND (cmd_nil)
{
 SYS_ARG a;

 sys_arg_get (&a);

 if (a.c == 1)
  psx_put ("missing command line\n");
 else
  psx_put ("invalid command line\n");

 return (0);
}

/******************************************************************************/
/* Private                                                               Main */
/******************************************************************************/
/** print version number
 */
 
SCI_COMMAND (cmd_ver)
{
 SYS_STR num,tsp,bld;

 ver_get (num,tsp,bld);
 printf ("%s:%s\n\n",tsp,bld);

 return (0);
}

/******************************************************************************/

/** check validity of PID
 */
 
SCI_COMMAND (cmd_pid_chk)
{
  PSX_CSP csp;
  PSX_STR t;
  int r;

 cmd_csp_reset (&csp);                            /* reset command specification */
 csp.cmd = CMD_PID_CHK;

 sci_opt_get ("b:b",&csp.ctx.pid.chk.bat,NULL);
 
 if(csp.ctx.pid.chk.bat)   // batch mode
 {
   if(!sci_opt_get ("i:s",csp.ctx.pid.chk.idf,""))      /*  input data file      */   /*:BG: d->i*/
   {
     psx_put("missing input file!");
     return(-1);
   }
 }
 else
  if(!sci_opt_get ("P:s",csp.ctx.pid.chk.inp,NULL))     /* input PID             */
  {
    psx_put("missing input PID!");
    return(-1);
  }

 sci_opt_get ("o:s",csp.ctx.pid.chk.odf,"");      /*  output data file           */
  
 sci_opt_get ("p:n",&csp.ctx.pid.chk.pos,"0");    /* get start position          */

 
    
 // execute command chk:
 r = cmd_execute(&csp);

 return(r);
}



/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/

/******************************************************************************/
/* Interface Definitions                                                      */
/******************************************************************************/

SCI_INTERFACE (psx_int)
{
 { 0		,0	,0,	"psx <cmd> {<par>} {<opt>}\n\n"	\
				"              -c:<file>   configuration"	},

 { "gen"	,cmd_dbs_gen,0,"generate database\n"				\
				"-s:<file>     schema specification\n"		\
				"-d            delete database"			},
 
 { "enc"	,cmd_ctn_enc,0,	"encipher\n"					\
				"-f:<file>     input format specification\n"	\
				"-i:<file>     input data file\n"		\
				"-o:<file>     output data file\n"		\
				"-u:<file>     output format specification\n"	\
				"-p:<num>      start position\n"		\
				"-e:[raw|hex]  encoding specification\n"	\
				"-b            batch mode processing\n"		\
				"-h            print headings\n"			\
                "-d:\"<sym>\"  record item delimiter"	},     /*:BG: added 25.02.05*/
 /* command not working properly yet
 { "dec"	,cmd_ctn_dec,0,	"decipher\n"					\
				"-f:<file>     input format specification\n"	\
				"-i:<file>     input data file\n"		\
				"-o:<file>     output data file\n"		\
				"-e:[raw|hex]  encoding specification\n"	\
				"-b            batch mode processing\n"		\
				"-h            print headings"			},

 */
        
 /* command not working properly yet
 { "cvt"	,cmd_ctn_cvt,0,	"convert\n"					\
				"-f:[f:]<str>  format specification\n"		\
				"-u:<file>     output format file\n"		\
				"-i:<file>     input file\n"			\
				"-o:<file>     output file\n"			\
				"-e:<e>:<e>    encoding"			},
  */

 { "req"	,cmd_pid_req,0,	"request pid\n"					\
				"-f:<file>     input format specification\n"	\
				"-i:<file>     input data file\n"		\
				"-t:<file>     output trace file\n"		\
				"-s:[+|-]      sureness\n"			\
				"-b            batch mode\n"			\
				"-F            force\n"				\
				"-p:<num>      start position\n"		\
                "-d:\"<sym>\"  record item delimiter"	},   /*:BG:added 25.02.05*/

 { "chk" ,cmd_pid_chk,0, "check pid\n"       \
        "-P:<PID>      input PID\n" \
        "-i:<file>     input data file\n" \
        "-p:<num>      start position\n" \
        "-b            batch mode\n" \
        "-o:<file>     output file"       },  

 { "gui"	,cmd_gui_cgi,0, "graphical user interface\n"			\
				"-s            enable system panel\n"		\
				"-d            enable debugging"		},

 { "cfg"	,cmd_cfg,0,"show configuration\n"				\
				"-s            dump schema\n"			\
				"-m            dump matching\n"			\
				"-q:<str>      filter (KSXO)"			},

 /* command should not be visible for end user
 { "dum"	,cmd_dat_gen,0,"generate test data\n"			\
				"-f:file       format specification\n"	\
				"-o:file       output file\n"		\
				"-n:num        number of records"	},
  */
 { "sts"	,cmd_dbs_sts,0,"generate status summary"			}, 
 { "ver"	,cmd_ver,0,"version"					},
 { "hlp"	,sci_hlp,0,"help"					},
 { NULL		,cmd_nil,0,""						},
 { 0,0,0,0 }
};

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


