/******************************************************************************/
/* psx-lst.c                               List Interface                     */
/******************************************************************************/
/** @file psx-lst.c List Interface - Source Code File
 * Definitions and functions implementing a concatenated list. A list node
 * consists of two pointers to other list nodes and one pointer to some contents
 * without a specified datatype.
 * Provided functions are:
 * - creation of a list
 * - deletion of a list
 * - clear contents
 * - freeing allocated memory
 * - putting a node into list
 * - seeking a special node
 * - getting an element from a specified position from the list
 * - counting the list nodes
 */
 
#include "psx.h"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

SYS_STRUCTURE (LST_NODE)
{
 LST_NODE * p;
 LST_NODE * s;
 PTR        ptr;
};

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** create a new list node
 * @param n pointer to new list node
 */

static BOOL lst_nod_create (LST_NODE ** n)
{
 LST_NODE * t;

 if (n == NULL)
  return (FALSE);

 if ((t = SYS_ALLOCATE (LST_NODE)) == NULL)
  return (FALSE);

 memset (t,0,sizeof (*t));

 *n = t;

 return (TRUE);
}

/******************************************************************************/
/** delete list node n
 * @param n pointer to list node
 */

static BOOL lst_nod_delete (LST_NODE ** n)
{
 if (n == NULL)
  return (FALSE);

 if (*n == NULL)
  return (FALSE);

 free (*n);
 *n = NULL;

 return (TRUE);
}

/******************************************************************************/
/** insert node n between node h and node t
 * @param h pointer to pointer to list node (head)
 * @param t pointer to pointer to list node to be inserted
 * @param n pointer to list node
 */

static BOOL lst_integrate (LST_NODE ** h,LST_NODE ** t,LST_NODE * n)
{
 if (h == NULL || t == NULL || n == NULL)
  return (FALSE);

 if (n -> p)
  n -> p -> s = n;
 else
  *h = n;

 if (n -> s)
  n -> s -> p = n;
 else
  *t = n;

 return (TRUE);
}


/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** allocate structure space, intitialize
 * @param l pointer to pointer to list node structure
 */

BOOL lst_create (SSI_LST ** l)
{
 SSI_LST * t;

 if (l == NULL)
  return (FALSE);

 if ((t = SYS_ALLOCATE (SSI_LST)) == NULL)
  return (FALSE);

 memset (t,0,sizeof (*t));

 *l = t;                 // set new node

 return (TRUE);
}

/******************************************************************************/
/** free structure space, free data space
 * @param l pointer to list node structure
 */

BOOL lst_delete (SSI_LST ** l)
{
 BOOL r = TRUE;

 if (l == NULL)
  return (FALSE);

 if (*l == NULL)
  return (FALSE);

 if (!lst_clear (*l))
  r = FALSE;

 free (*l);
 *l = NULL;

 return (r);
}

/******************************************************************************/
/** clear list
 * @param l list node
 */
 
BOOL lst_clear (SSI_LST * l)
{
 LST_NODE * p0,* p1;

 if (l == NULL)
  return (FALSE);

 p0 = (LST_NODE *) l -> h;

 while (p0)
 {
  p1 = p0;
  p0 = p0 -> s;
  free (p1);
 }

 l -> h = NULL;
 l -> t = NULL;
 l -> s = NULL;

 return (TRUE);
}

/******************************************************************************/
/** free allocated memory
 * @param l list node structure
 */
 
BOOL lst_free (SSI_LST * l)
{
 LST_NODE * p;

 if (l == NULL)
  return (FALSE);

 for (p = (LST_NODE *) l -> h;p != NULL;p = p -> s)
  if (p -> ptr)
  {
   free (p -> ptr);
   p -> ptr = NULL;
  }

 lst_clear (l);

 return (TRUE);
}

/******************************************************************************/
/** put element to list
 * @param l list structure
 * @param p pointer to new element
 */
 
BOOL lst_put (SSI_LST * l,PTR p)
{
 LST_NODE * n;

 if (l == NULL || p == NULL)
  return (FALSE);



 if (!lst_nod_create (&n))
  return (FALSE);

 n -> ptr = p;
 n -> p = (LST_NODE *) l -> t;
 n -> s = NULL;

 lst_integrate ((LST_NODE **) &l -> h,(LST_NODE **) &l -> t,n);

 return (TRUE);
}

/******************************************************************************/
/** seek position in list
 * @param l list structure
 * @param pos position in list (LST_POS_HEAD,LST_POS_TAIL,LST_POS_PRED,LST_POS_SUCC)
 */
 
BOOL lst_seek (SSI_LST * l,IDX pos)
{

 LST_NODE * p;

 if (l == NULL)
  return (FALSE);

 switch (pos)
 {
  case LST_POS_HEAD:
   p = (LST_NODE *) l -> h;
   break;

  case LST_POS_TAIL:
   p = (LST_NODE *) l -> t;
   break;

  case LST_POS_PRED:
   p = l -> s ? ((LST_NODE *) (l -> s)) -> p : NULL;
   break;

  case LST_POS_SUCC:
   p = l -> s ? ((LST_NODE *) (l -> s)) -> s : NULL;
   break;

  default:
  {
   IDX i = 0;

   p = (LST_NODE *) l -> h;

   while (i < pos && p != NULL)
   {
    p = p -> s;
    i++;

   }

   if (i != pos)
    p = NULL;
   break;
  }
 }

 l -> s = p;

 if (l -> s == NULL)
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** get element from position pos in list
 * @param l list structure
 * @param pos offset in list
 * @param p pointer to retrieved element
 */

BOOL lst_get (SSI_LST * l,IDX pos,PTR p)
{
 if (l == NULL)
  return (FALSE);

 if (lst_seek (l,pos))
  if (l -> s)
  {
   if (p)
    *((PTR *) p) = ((LST_NODE *)(l -> s)) -> ptr;

   return (TRUE);
  }

 if (p)
  *((PTR *) p) = NULL;

 return (FALSE);
}

/******************************************************************************/
/** count number of list nodes
 * @param l list structure
 * @param n number of list elements
 */
 
BOOL lst_cnt (SSI_LST * l,NUM * n)
{
 LST_NODE * p;
 NUM k;

 if (l == NULL || n == NULL)
  return (FALSE);

 k = 0;

 for (p = (LST_NODE *) l -> h;p;p = p -> s)
  k++;

 *n = k;

 return (TRUE);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


