/******************************************************************************/
/* psx-msi.c                                            Mail System Interface */
/******************************************************************************/
/** @file psx-msi.c Mail System Interface - Source Code File
 * Functions providing a Mailing System being able to send a notification mail
 * to the administrator. E-Mail address is given in the configuration file.
 */
 
#include "psx.h"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** produce temporary mail
 * @param ctx mail context
 * @param fn file name
 */

static BOOL msi_tmp_make (MSI_CTX * ctx,PSX_STR fn)
{
 FILE * f;
 char t [L_tmpnam];                  // temporary filename
 BOOL r;

 if (ctx == NULL || fn == NULL)
  return (FALSE);

 if ((tmpnam (t)) == NULL)           // critical call
  return (FALSE);

 if ((f = fopen (t,"wt")) == NULL)
  return (FALSE);

 if (!str_isnil (ctx -> snd.adr))
  if (!str_isnil (ctx -> snd.name))
   fprintf (f,"From: %s <%s>\n",ctx -> snd.name,ctx -> snd.adr);
  else
   fprintf (f,"From: %s\n",ctx -> snd.adr);

 if (!str_isnil (ctx -> rec.adr))
  if (!str_isnil (ctx -> rec.name))
   fprintf (f,"From: %s <%s>\n",ctx -> rec.name,ctx -> rec.adr);
  else
   fprintf (f,"From: %s\n",ctx -> rec.adr);

 fprintf (f,"Subject: %s\n",ctx -> sub);

 r = TRUE;

 if (fwrite (ctx -> txt -> ptr,ctx -> txt -> len,1,f) != 1)
  r = FALSE;

 fclose (f);

 if (r)
  strcpy (fn,t);

 return (r);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** create new mail configuration
 * @param ctx pointer to mail context
 */

BOOL msi_create (MSI_CTX ** ctx)
{
 PSX_BLK * b = NULL;

 if (ctx == NULL)
  return (FALSE);

 if ((*ctx = (MSI_CTX *) malloc (sizeof (**ctx))) == NULL)
  return (FALSE);

 memset (*ctx,0,sizeof (**ctx));

 if (!blk_create (&b))
 {
  free (ctx);
  return (FALSE);
 }

 (*ctx) -> txt = b;

 return (TRUE);
}

/******************************************************************************/
/** delete mail configuration
 * @param ctx pointer to mail context
 */

BOOL msi_delete (MSI_CTX ** ctx)
{
 if (ctx == NULL)
  return (FALSE);

 if (*ctx == NULL)
  return (FALSE);

 if ((*ctx) -> txt)
  blk_delete (&(*ctx) -> txt);

 free (*ctx);
 *ctx = NULL;

 return (TRUE);
}

/******************************************************************************/
/** set address of sender
 * @param ctx mail context
 * @param ins mail instance
 */

BOOL msi_set_snd (MSI_CTX * ctx,MSI_INS * ins)
{
 if (ctx == NULL || ins == NULL)
  return (FALSE);

 ctx -> snd = *ins;

 return (TRUE);
}

/******************************************************************************/
/** set address of receiver
 * @param ctx mail context
 * @param ins mail instance
 */

BOOL msi_set_rec (MSI_CTX * ctx,MSI_INS * ins)
{
 if (ctx == NULL || ins == NULL)
  return (FALSE);

 ctx -> rec = *ins;

 return (TRUE);
}

/******************************************************************************/
/** set subject
 * @param ctx mail context
 * @param txt subject text
 */

BOOL msi_set_sub (MSI_CTX * ctx,PSX_STR txt)
{
 if (ctx == NULL)
  return (FALSE);

 strcpy (ctx -> sub,txt);

 return (TRUE);
}

/******************************************************************************/
/** generate mail
 * @param ctx mail context
 * @param fmt format string
 * @param ... variable argument list
 */

#define MSI_PUT_MAX 1000

BOOL msi_put (MSI_CTX * ctx,const char * fmt,...)
{
 char * p;
 va_list l;
 int r;

 if (ctx == NULL || fmt == NULL)
  return (FALSE);

 if ((p = (char *) malloc (MSI_PUT_MAX * sizeof (char))) == NULL)
  return (FALSE);

 va_start (l,fmt);
 r = vsnprintf (p,MSI_PUT_MAX,fmt,l);
 va_end (l);

 if (r == -1)
 {
  free (p);
  return (FALSE);
 }


 if (!blk_put_s (ctx -> txt,p)) // put p to context string block
 {
  free (p);
  return (FALSE);
 }

 free (p);

 return (TRUE);
}

/******************************************************************************/
/** send mail with sendmail
 * @param ctx mail context
 */

BOOL msi_send (MSI_CTX * ctx)
{
 PSX_STR f;
 PSX_STR c;
 int r;

 if (ctx == NULL)
  return (FALSE);

 if (!msi_tmp_make (ctx,f))
  return (FALSE);

 sprintf (c,"sendmail -f%s %s < %s",ctx -> snd.adr,ctx -> rec.adr,f);
 r = system (c);

 //psx_put ("msi: %s -> %d",c,r);

 return (TRUE);
}

/******************************************************************************/
/** lookup adresses in cofig and generate mail
 * @param dic dictionary structure
 */

BOOL msi_notify (PSX_DIC * dic)
{
 MSI_CTX * ctx;
 MSI_INS snd,rec;
 PSX_STR sub;

 if (!psx_cfg_get ("mail.src.name",snd.name))
  return (FALSE);

 if (!psx_cfg_get ("mail.src.adr",snd.adr))
  return (FALSE);

 if (!psx_cfg_get ("mail.dst.name",rec.name))
  return (FALSE);

 if (!psx_cfg_get ("mail.dst.adr",rec.adr))
  return (FALSE);

 if (!psx_cfg_get ("mail.sub",sub))
  strcpy (sub,"PSX Notification");

 if (!msi_create (&ctx))
  return (FALSE);

 msi_set_snd (ctx,&snd);  // set sender
 msi_set_rec (ctx,&rec);  // set receiver
 msi_set_sub (ctx,sub);   // set subject

 if (dic)
 {
  char * t;

  if (dic_get (dic,"MSG",&t))
   msi_put (ctx,"Notification: %s",t);

  if (dic_get (dic,"ADR",&t))
   msi_put (ctx,"User address: %s",t);

  if (dic_get (dic,"TSP",&t))
   msi_put (ctx,"Time: %s",t);
 }

 msi_send (ctx);
 msi_delete (&ctx);
 psx_put ("msi notify");

 return (TRUE);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


