/******************************************************************************/
/* psx-mti.inl                             Logical Match Expression Generator */
/******************************************************************************/
/** @file psx-mti.inl Matching Interface: Logical Match Expression Generator - Inline Source Code File
 * Functions and definitions supporting the matching process by implementing the
 * functionaliy of expression creation for SQL statements (WHERE clause).
 */
 
/******************************************************************************/
/* Type                                                               Private */
/******************************************************************************/
/** Matching Context Expression */
typedef struct _MTI_CTX_EXP
{
 PSX_SCH *	sch;             /**< schema     */
 IDX		idx;                 /**< index      */
 PSX_REC *	rec;             /**< record     */
 MTM            mtm;         /**< matchmode  */
 PSX_EXP **	exp;             /**< expression */
} MTI_CTX_EXP;

#define	OPS_ZERO	0x00
#define	OPS_ATR		0x01
#define	OPS_VAL		0x02

/** Matching Operand */
typedef struct _MTI_OPN
{
 IDX  idx;                    /**< index              */
 CMP  cmp;                    /**< component          */
 BYTE ops;                    /**< operand assignment */
} MTI_OPN;

#define OPN_ASSIGN(opn,i,c,s)	\
 opn.idx = i;				\
 opn.cmp = c;				\
 opn.ops = s;

#define MTI_OPT_CMT	1

/******************************************************************************/
/* Private                                                      Operand Level */
/******************************************************************************/
/** generate operand expression for attribute name
 * @param ctx pointer to matching context expression
 * @param idx index
 * @param cmp component
 * @param exp pointer to expression pointer after invocation
 */

static BOOL mti_exp_gen_opn_atr (MTI_CTX_EXP * ctx,IDX idx,CMP cmp,PSX_EXP ** exp)
{
 PSX_STR t;

 if (ctx == NULL || exp == NULL)
  return (FALSE);

 *exp = NULL;

 if (idx == IDX_NIL)
 {
  if (!exp_create (exp))
   return (FALSE);

  exp_assign (*exp,ETP_CON,"NULL");
  return (TRUE);
 }

 if (idx >= ctx -> sch -> fsp -> n)
  return (FALSE);

 if (!sch_atr_sym_get (ctx -> sch,idx,cmp,t)) // get field name
 {
  psx_put_v ("mti: exp_gen_opn_atr: sch_atr_sym_get failed");
  return (FALSE);
 }

 if (!exp_create (exp))
  return (FALSE);

 exp_assign (*exp,ETP_VAR,t);

 return (TRUE);
}

/******************************************************************************/
/** generate operand expression for attribute value
 * @param ctx pointer to expression context
 * @param idx index
 * @param cmp component
 * @param exp expression string
 */

static BOOL mti_exp_gen_opn_val (MTI_CTX_EXP * ctx,IDX idx,CMP cmp,PSX_EXP ** exp)
{
 PSX_STR t,v, temp;

 if (ctx == NULL || exp == NULL)
  return (FALSE);

 *exp = NULL;

 if (idx == IDX_NIL)
 {
  if (!exp_create (exp))
   return (FALSE);

  exp_assign (*exp,ETP_CON,"NULL");
  return (TRUE);
 }

 if (idx >= ctx -> sch -> fsp -> n)
  return (FALSE);
                                                     
 if (!sch_lookup (ctx -> sch,idx,cmp,ctx -> rec,t)) // get input value and put into t
  return (FALSE);

 if (!exp_create (exp))
  return (FALSE);

 if (!str_isnil (t))           // t = value
 {                             
  dbi_val_cvt(t,temp);
  sprintf (v,"'%s'",temp);  
 }
 else
  strcpy (v,"NULL");

 exp_assign (*exp,ETP_CON,v);

 return (TRUE);
}

/******************************************************************************/
/** generate operand expression
 * @param ctx pointer to expression context
 * @param opn pointer to operand
 * @param exp pointer to expression pointer
 */

static BOOL mti_exp_gen_opn (MTI_CTX_EXP * ctx,MTI_OPN * opn,PSX_EXP ** exp)
{
 if (ctx == NULL || exp == NULL)
 {
  psx_put_v ("mti: exp_gen_opn: null failure");
  return (FALSE);
 }

 if (opn == NULL)	
 {
  if (!exp_create (exp))
   return (FALSE);

  exp_assign (*exp,ETP_CON,"NULL");

  return (TRUE);
 }

 switch (opn -> ops)
 {
  case OPS_ATR:
   if (!mti_exp_gen_opn_atr (ctx,opn -> idx,opn -> cmp,exp))
   {
    psx_put_v ("mti: exp_gen_opn: atr failure");
    return (FALSE);
   }
   break;

  case OPS_VAL:
   if (!mti_exp_gen_opn_val (ctx,opn -> idx,opn -> cmp,exp))
   {
    psx_put_v ("mti: exp_gen_opn: val failure");
    return (FALSE);
   }
   break;

  default:
   psx_put_v ("mti: exp_gen_opn: default failure");
   return (FALSE);
 }

 if (*exp == NULL)
 {
  psx_put_v ("mti: exp_gen_opn: exp null failure");
  return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/* Private                                                    Operation Level */
/******************************************************************************/
/** generate relational operation
 * @param ctx pointer to expression context
 * @param rop relational operator identifier
 * @param ops pointer to first operand
 * @param opd pointer to second operand
 * @param exp pointer to pointer to joined expression after invocation
 */
 
static BOOL mti_exp_gen_rop (MTI_CTX_EXP * ctx,BYTE rop,MTI_OPN * ops,MTI_OPN * opd,PSX_EXP ** exp)
{
 PSX_EXP * es,* ed;

 if (ctx == NULL || exp == NULL)
  return (FALSE);

 if (!mti_exp_gen_opn (ctx,ops,&es))        /* generate operand */
 {
  psx_put_v ("mti: exp_gen_rop: error 1");
  return (FALSE);
 }

 if (!mti_exp_gen_opn (ctx,opd,&ed))        /* generate operand */
 {
  psx_put_v ("mti: exp_gen_rop: error 2");
  exp_delete (&es);
  return (FALSE);
 }

 if (!exp_create (exp))                     /* create expression */
 {
  psx_put_v ("mti: exp_gen_rop: error 3");
  exp_delete (&es);
  exp_delete (&ed);
  return (FALSE);
 }

 exp_assign (*exp,ETP_ROP,rop);
 exp_add (*exp,es);
 exp_add (*exp,ed);

 return (TRUE);
}

/******************************************************************************/
/* Private                                                         Data Level */
/******************************************************************************/
/** generate expression for schema item                      
 * @param xs  field index of input field processed                
 * @param xd  field index of database field processed                                                                     
 * @param cs  component of input field                                                         
 * @param cd  component of database field                                                         
 * @param os  operand                                                                     
 * @param od  operand                                                                     
 * @param rop relational operator
 * @param nullAllowed whether NULL value is allowed for field
 * @param exp pointer to expression struct             
 */
 
static BOOL mti_exp_gen_cmp (MTI_CTX_EXP * ctx,IDX xs,IDX xd,CMP cs,CMP cd,BYTE os,BYTE od,BYTE rop,BOOL nullAllowed,PSX_EXP ** exp)
{
 MTI_OPN s,d;
 PSX_EXP * e;

 s.idx = xs;
 s.cmp = cs;
 s.ops = os;

 d.idx = xd;
 d.cmp = cd;
 d.ops = od;

 if (!mti_exp_gen_rop (ctx,rop,&s,&d,&e))
 {
  psx_put_v ("mti: exp_gen_cmp: exp_gen_rop failed");
  return (FALSE);
 }

 if (!nullAllowed)
 {
  PSX_EXP * e1,* e2;

  e1 = e;
  e = NULL;

  if (!mti_exp_gen_rop (ctx,ROP_NE,&s,NULL,&e2))
  {
   psx_put_v ("mti: exp_gen_cmp: exp_gen_rop failed");
   exp_delete (&e1);
   return (FALSE);
  }

  if (!exp_create (&e))
  {
   exp_delete (&e1);
   exp_delete (&e2);
   return (FALSE);
  }

  exp_assign (e,ETP_LOP,LOP_AND);
  exp_add (e,e1);
  exp_add (e,e2);
 }

 *exp = e;

 return (TRUE);
}

/******************************************************************************/
/** generate expression for schema item of type raw
 * @param ctx pointer to matching expression context
 * @param xs index of first expression
 * @param xd index of second expression
 * @param exp pointer to pointer to expression after invocation
 */

static BOOL mti_exp_gen_cmp_r (MTI_CTX_EXP * ctx,IDX xs,IDX xd,PSX_EXP ** exp)
{
 BOOL exact = TEST (ctx -> mtm,MTM_X1) ? 1 : 0;

 if (!mti_exp_gen_cmp (ctx,xs,xd,CMP_NIL,CMP_NIL,OPS_ATR,OPS_VAL,ROP_EQ,TRUE,exp))
 {
  psx_put_v ("mti: exp_gen_cmp_r: exp_gen_cmp failed");
  return (FALSE);
 }

 if (MTI_OPT_CMT)
 {
  PSX_STR t;

  sprintf (t,"exp: raw, exact=%d (ignored): '%s' <-> '%s'",exact,ctx -> sch -> fsp -> itm [xs].sym,ctx -> sch -> fsp -> itm [xd].sym);
  exp_cmt_set (*exp,t);
 }

 return (TRUE);
}

/******************************************************************************/
/** generate expression for sureness
 * @param ctx pointer to matching expression context
 * @param sur sureness (0 or 1)
 * @param exp pointer to pointer to expression after invocation
 */

static BOOL mti_exp_gen_cmp_s (MTI_CTX_EXP * ctx,BOOL sur,PSX_EXP ** exp)
{
 PSX_EXP * s,* d;

 if (!exp_create (&s))
  return (FALSE);

 if (!exp_create (&d))
 {
  exp_delete (&s);
  return (FALSE);
 }

 if (!exp_create (exp))
 {
  exp_delete (&s);
  exp_delete (&d);
  return (FALSE);
 }

 exp_assign (s,ETP_VAR,"sur");
// exp_assign (d,ETP_CON,sur ? "TRUE" : "FALSE");
 exp_assign (d,ETP_CON,sur ? "1" : "0");
 exp_assign (*exp,ETP_ROP,ROP_EQ);
 exp_add (*exp,s);
 exp_add (*exp,d);

 return (TRUE);
}

/******************************************************************************/
/* Private                              Name Compare Expression    Data Level */
/******************************************************************************/
/** generate compare expression for item of type name, exact=0            
 *                                                                            
 *  OR                                                                   <e> 
 *  |                                                                         
 *  +-s.itm_pc=d.itm_pc                                                  <e1> 
 *  |                                                                         
 *  +-s.itm_ph=d.itm_ph                                                  <e2> 
 *
 * @param ctx pointer to matching expression context
 * @param xs index of first expression
 * @param xd index of second expression
 * @param exp pointer to pointer to expression after invocation
 */

static BOOL mti_exp_gen_cmp_n_e0 (MTI_CTX_EXP * ctx,IDX xs,IDX xd,PSX_EXP ** exp)
{
 PSX_EXP * e,* e1,* e2;
 BOOL use_pho_h = TEST (ctx -> sch -> fsp -> itm[xs].tfm, TFM_PHO_H)
 									&& TEST (ctx -> sch -> fsp -> itm[xd].tfm, TFM_PHO_H);
 BOOL use_pho_c = TEST (ctx -> sch -> fsp -> itm[xs].tfm, TFM_PHO_C)
 									&& TEST (ctx -> sch -> fsp -> itm[xd].tfm, TFM_PHO_C);

 /* generate comparision for Cologne phonetic */
 if (use_pho_c)
 {
  if (!mti_exp_gen_cmp (ctx,xs,xd,CMP_PC,CMP_PC,OPS_ATR,OPS_VAL,ROP_EQ,TRUE,&e1))
  {
   psx_put ("mti_exp_gen_cmp_n_e0: mti_exp_gen_cmp failed (1)");
   return (FALSE);
  }
 } else
 {
  e1 = NULL;
 }

 /* generate comparision for Hannover phonetic */
 if (use_pho_h)
 {
  if (!mti_exp_gen_cmp (ctx,xs,xd,CMP_PH,CMP_PH,OPS_ATR,OPS_VAL,ROP_EQ,TRUE,&e2))
  {
   psx_put ("mti_exp_gen_cmp_n_e0: mti_exp_gen_cmp failed (2)");
   exp_delete_n (1,&e1);
   return (FALSE);
  }
 } else
 {
  e2 = NULL;
}

 if (!exp_create (&e))
 {
  exp_delete_n (2,&e1,&e2);
  return (FALSE);
 }

 exp_assign (e,ETP_LOP,LOP_OR);
 if (e1 != NULL)
 {
  exp_add (e,e1);
 }
 if (e2 != NULL)
 {
  exp_add (e,e2);
 }
 *exp = e;

 return (TRUE);
}

/******************************************************************************/
/** generate compare expression for item of type name, exact=1, exp 1     
 *                                                                            
 *  AND                                                                   <e> 
 *  |                                                                         
 *  +-s.itm_c1=d.itm_c1                                                  <e1> 
 *  |                                                                         
 *  +-OR                                                                 <e2> 
 *    |                                                                       
 *    +-s.itm_c2=d.itm_c2                                               <e2a> 
 *    |                                                                       
 *    +-s.itm_c2=NULL                                                   <e2b> 
 *    |                                                                       
 *    +-d.itm_c2=NULL                                                   <e2c> 
 *
 * @param ctx pointer to matching expression context
 * @param xs index of first expression
 * @param xd index of second expression
 * @param exp pointer to pointer to expression after invocation
 */

static BOOL mti_exp_gen_cmp_n_e1_1 (MTI_CTX_EXP * ctx,IDX xs,IDX xd,PSX_EXP ** exp)
{
 PSX_EXP * e,* e1,* e2,* e2a,* e2b,* e2c;

 if (!mti_exp_gen_cmp (ctx,xs,xd,CMP_N1,CMP_N1,OPS_ATR,OPS_VAL,ROP_EQ,TRUE,&e1))
 {
  psx_put_v ("mti: error 1");
  return (FALSE);
 }

 if (!mti_exp_gen_cmp (ctx,xs,xd,CMP_N2,CMP_N2,OPS_ATR,OPS_VAL,ROP_EQ,TRUE,&e2a))
 {
  psx_put_v ("mti: error 2");
  exp_delete_n (1,&e1);
  return (FALSE);
 }

 if (!mti_exp_gen_cmp (ctx,xs,IDX_NIL,CMP_N2,CMP_NIL,OPS_ATR,OPS_VAL,ROP_EQ,TRUE,&e2b))
 {
  psx_put_v ("mti: error 3");
  exp_delete_n (2,&e1,&e2a);
  return (FALSE);
 }

 if (!mti_exp_gen_cmp (ctx,xd,IDX_NIL,CMP_N2,CMP_NIL,OPS_VAL,OPS_VAL,ROP_EQ,TRUE,&e2c))
 {
  psx_put_v ("mti: error 4");
  exp_delete_n (3,&e1,&e2a,&e2b);
  return (FALSE);
 }

 if (!exp_create (&e2))
 {
  psx_put_v ("mti: error 5");
  exp_delete_n (4,&e1,&e2a,&e2b,&e2c);
  return (FALSE);
 }

 exp_assign (e2,ETP_LOP,LOP_OR);
 exp_add (e2,e2a);
 exp_add (e2,e2b);
 exp_add (e2,e2c);

 if (!exp_create (&e))
 {
  psx_put_v ("mti: error 6");
  exp_delete_n (5,&e1,&e2a,&e2b,&e2c,&e2);
  return (FALSE);
 }

 exp_assign (e,ETP_LOP,LOP_AND);
 exp_add (e,e1);
 exp_add (e,e2);

 *exp = e;

 return (TRUE);
}

/******************************************************************************/
/** generate compare expression for item of type name, exact=1, exp 2     
 *                                                                            
 *  AND                                                                   <e> 
 *  |                                                                         
 *  +-s.itm_c2=d.itm_c1                                                  <e1> 
 *  |                                                                         
 *  +-OR                                                                 <e2> 
 *      |                                                                     
 *      +-s.itm_c1=d.itm_c2                                             <e2a> 
 *      |                                                                     
 *      +-d.itm_c2=NULL                                                 <e2b> 
 *
 */

static BOOL mti_exp_gen_cmp_n_e1_2 (MTI_CTX_EXP * ctx,IDX xs,IDX xd,PSX_EXP ** exp)
{
 PSX_EXP * e,* e1,* e2,* e2a,* e2b;

 if (!mti_exp_gen_cmp (ctx,xs,xd,CMP_N2,CMP_N1,OPS_ATR,OPS_VAL,ROP_EQ,FALSE,&e1))
 {
  psx_put ("mti_exp_gen_cmp_n_e1_2: mti_exp_gen_cmp failed (1)");
  return (FALSE);
 }

 if (!mti_exp_gen_cmp (ctx,xs,xd,CMP_N1,CMP_N2,OPS_ATR,OPS_VAL,ROP_EQ,TRUE,&e2a))
 {
  psx_put ("mti_exp_gen_cmp_n_e1_2: mti_exp_gen_cmp failed (2)");
  exp_delete_n (1,&e1);
  return (FALSE);
 }

 if (!mti_exp_gen_cmp (ctx,xd,IDX_NIL,CMP_N2,CMP_NIL,OPS_VAL,OPS_VAL,ROP_EQ,TRUE,&e2b))
 {
  psx_put ("mti_exp_gen_cmp_n_e1_2: mti_exp_gen_cmp failed (3)");
  exp_delete_n (2,&e1,&e2a);
  return (FALSE);
 }

 if (!exp_create (&e2))
 {
  exp_delete_n (3,&e1,&e2a,&e2b);
  return (FALSE);
 }

 exp_assign (e2,ETP_LOP,LOP_OR);
 exp_add (e2,e2a);
 exp_add (e2,e2b);

 if (!exp_create (&e))
 {
  exp_delete_n (4,&e1,&e2a,&e2b,&e2);
  return (FALSE);
 }

 exp_assign (e,ETP_LOP,LOP_AND);
 exp_add (e,e1);
 exp_add (e,e2);

 *exp = e;

 return (TRUE);
}

/******************************************************************************/
/** generate compare expression for item of type name, exact=1, exp 3     
 *                                                                            
 *  AND                                                                   <e> 
 *  |                                                                         
 *  +-s.itm_c1=d.itm_c2                                                  <e1> 
 *  |                                                                         
 *  +-OR                                                                 <e2> 
 *     |
 *      +-s.itm_c2=d.itm_c1                                             <e2a> 
 *      |                                                                     
 *      +-s.itm_c2=NULL                                                 <e2b> 
 *
 *
 * @param ctx pointer to matching expression context
 * @param xs index of first expression
 * @param index of second expression
 * @param exp pointer to pointer to expression after invocation
 */

static BOOL mti_exp_gen_cmp_n_e1_3 (MTI_CTX_EXP * ctx,IDX xs,IDX xd,PSX_EXP ** exp)
{
 PSX_EXP * e,* e1,* e2,* e2a,* e2b;
 /* changed arg nullAllowed to false to prevent match through empty values
    Andreas Borg, 3.1.2008 */
 if (!mti_exp_gen_cmp (ctx,xs,xd,CMP_N1,CMP_N2,OPS_ATR,OPS_VAL,ROP_EQ,FALSE,&e1))
 {
  return (FALSE);
 }

 if (!mti_exp_gen_cmp (ctx,xs,xd,CMP_N2,CMP_N1,OPS_ATR,OPS_VAL,ROP_EQ,TRUE,&e2a))
 {
  exp_delete_n (1,&e1);
  return (FALSE);
 }

 if (!mti_exp_gen_cmp (ctx,xs,IDX_NIL,CMP_N2,CMP_NIL,OPS_ATR,OPS_VAL,ROP_EQ,TRUE,&e2b))
 {
  exp_delete_n (2,&e1,&e2a);
  return (FALSE);
 }

 if (!exp_create (&e2))
 {
  exp_delete_n (3,&e1,&e2a,&e2b);
  return (FALSE);
 }

 exp_assign (e2,ETP_LOP,LOP_OR);
 exp_add (e2,e2a);
 exp_add (e2,e2b);

 if (!exp_create (&e))
 {
  exp_delete_n (4,&e1,&e2a,&e2b,&e2);
  return (FALSE);
 }

 exp_assign (e,ETP_LOP,LOP_AND);
 exp_add (e,e1);
 exp_add (e,e2);

 *exp = e;

 return (TRUE);
}

/******************************************************************************/
/** generate (compare) expression for item of type name, exact=1            
 *                                                                            
 *  OR                                                                        
 *  |                                                                         
 *  +-<exp 1>                                                                 
 *  |                                                                         
 *  +-<exp 2>                                                                 
 *  |                                                                         
 *  +-<exp 3>                                                                 
 *
 * @param ctx pointer to matching expression context
 * @param xs index
 * @param xd index
 * @param exp pointer to pointer to expression after invocation
 */                                                                            

static BOOL mti_exp_gen_cmp_n_e1 (MTI_CTX_EXP * ctx,IDX xs,IDX xd,PSX_EXP ** exp)	// dbg: xs:lname,xd:aname !!!
{
 PSX_EXP * e,* e1,* e2,* e3;

 if (!mti_exp_gen_cmp_n_e1_1 (ctx,xs,xd,&e1))
  return (FALSE);

 if (!mti_exp_gen_cmp_n_e1_2 (ctx,xs,xd,&e2))
 {
  exp_delete_n (1,&e1);
  return (FALSE);
 }

 if (!mti_exp_gen_cmp_n_e1_3 (ctx,xs,xd,&e3))
 {
  exp_delete_n (2,&e1,&e2);
  return (FALSE);
 }

 if (!exp_create (&e))
 {
  exp_delete_n (3,&e1,&e2,&e3);
  return (FALSE);
 }

 exp_assign (e,ETP_LOP,LOP_OR);
 exp_add (e,e1);
 exp_add (e,e2);
 exp_add (e,e3);

 *exp = e;

 return (TRUE);
}

/******************************************************************************/
/** generate compare expression for item of type name
 * @param ctx pointer to matching expression context
 * @param xs index of first expression
 * @param xd index of second expression
 * @param exp pointer to pointer to expression
 */

static BOOL mti_exp_gen_cmp_n (MTI_CTX_EXP * ctx,IDX xs,IDX xd,PSX_EXP ** exp)	// dbg: xs:lname,xd:aname
{
 BOOL exact = TEST (ctx -> mtm,MTM_X1) ? 1 : 0;

 if (exact)
 {
  if (!mti_exp_gen_cmp_n_e1 (ctx,xs,xd,exp))
   return (FALSE);
 }
 else
 {
  if (!mti_exp_gen_cmp_n_e0 (ctx,xs,xd,exp))
   return (FALSE);
 }

 if (MTI_OPT_CMT)
 {
  PSX_STR t;

  sprintf (t,"exp: name, exact=%d: '%s' <-> '%s'",exact,ctx -> sch -> fsp -> itm [xs].sym,ctx -> sch -> fsp -> itm [xd].sym);
  exp_cmt_set (*exp,t);
 }

 return (TRUE);
}

/******************************************************************************/
/* Private                                                        Match Level */
/******************************************************************************/
/** generate (compare) expression for schema item
 *
 *  AND                                                                   <e> 
 *  |                                                                         
 *  +-s.itm=d.itm                                                        <e1> 
 *  |                                                                         
 *  +-s.itm IS NOT NULL                                                  <e2> 
 *
 * @param ctx pointer to matching expression context
 * @param xs index of expression
 * @param xd index of expression
 * @param exp pointer to pointer to expression after invocation
 */

// optional variables are tested for NULL here!

static BOOL mti_exp_gen_itm_cmp (MTI_CTX_EXP * ctx,IDX xs,IDX xd,PSX_EXP ** exp)
{
 PSX_EXP * e;
 BOOL r;

 if (ctx == NULL || exp == NULL)
  return (FALSE);

 *exp = NULL;
 e = NULL;

 switch (ctx -> sch -> fsp -> itm [xs].dtp)
 {
  case DTP_NAME:
  // case DTP_FNAME:

   r = mti_exp_gen_cmp_n (ctx,xs,xd,&e);
   break;

  default:
   r = mti_exp_gen_cmp_r (ctx,xs,xd,&e);
   break;
 }

 if (!r)
  return (FALSE);

 /* if var is optional, add a test for null */

 // if (TEST (ctx -> sch -> fsp -> itm [xs].equ,EQU_OPT))
 if (EQU_CMP (ctx -> sch -> fsp -> itm [xs].equ) == EQU_OPT)
 {
  PSX_EXP *s,* t;

  if (!mti_exp_gen_cmp (ctx,xs,IDX_NIL,CMP_NIL,CMP_NIL,OPS_ATR,OPS_VAL,ROP_NE,TRUE,&s))
  {
   exp_delete (&e);
   return (FALSE);
  }

  if (!exp_create (&t))
  {
   exp_delete (&e);
   exp_delete (&s);
   return (FALSE);
  }

  exp_assign (t,ETP_LOP,LOP_AND);
  exp_add (t,e);
  exp_add (t,s);
  e = t;

 //  printf ("*** exp: %s ***\n",ctx -> sch -> itm [xs].sym);
 }

 *exp = e;

 return (TRUE);
}

/******************************************************************************/
/** generate expression for schema item                                   
 *                                                                            
 * exp:                                                                       
 *                                                                            
 *      OR                                                                <e> 
 *      |                                                                     
 *      +-s.itm[i]=d.itm[i]                                                   
 *      |                                                                     
 *      +-s.itm[i]=d.itm[j1]                                                  
 *      |                                                                     
 *      :                                                                     
 *      |                                                                     
 *      +-s.itm[i]=d.itm[jn]                                                  
 *
 */

static BOOL mti_exp_gen_itm (MTI_CTX_EXP * ctx,PSX_EXP ** exp)
{
 PSX_EXP * ex;
 BYTE xch;

 xch = EQU_XCH (ctx -> sch -> fsp -> itm [ctx -> idx].equ); // get exchange group

 if (!mti_exp_gen_itm_cmp (ctx,ctx -> idx,ctx -> idx,&ex)) // compare field==value 
 {
  return (FALSE);
 }

 if (xch != 0)
 {
  PSX_EXP * e;
  PSX_STR val;
  IDX i;

  if (!exp_create (&e))
  {
   exp_delete (&ex);
   return (FALSE);
  }

  exp_assign (e,ETP_LOP,LOP_OR);
  exp_add (e,ex);

  for (i = 0;i < ctx -> sch -> fsp -> n;i++) // loop over fields
   if (EQU_XCH (ctx -> sch -> fsp -> itm [i].equ) == xch && i != ctx -> idx) // cross-check fields with same exchange group
   {
    PSX_EXP * ei;

    if (!mti_exp_gen_itm_cmp (ctx,ctx -> idx,i,&ei))	// dbg: lname = aname?
    {
     exp_delete_n (1,&e);
     return (FALSE);
    }

    exp_add (e,ei);
    // add expression the other way round to ensure commutativity
    // ei is allocated again in the function, so the old value remains at its place
    // added 3. Jan 2008, Andreas Borg
		if (!sch_lookup(ctx->sch,i,CMP_N1,ctx->rec,val))
		{
			return (FALSE);
		}
		
		if (*val!='\0')
		{
			if (!mti_exp_gen_itm_cmp (ctx,i,ctx -> idx,&ei))	// dbg: aname = lname?
		    {
		     exp_delete_n (1,&e);
		     return (FALSE);
		    }
		
		    exp_add (e,ei);
		}
   }
  *exp = e;
 }
 else
 {
  *exp = ex;
 }


 return (TRUE);
}

/******************************************************************************/
/** evaluates for given equality code if corresponding field should be
 *  considered in sql-query for current status (encoded in match mode mtm)
 *
 * @param mtm matchmode
 * @param equ equality
 */
 
static BOOL mti_exp_consider (MTM mtm,EQU equ)
{
 switch (equ)
 {
  // required, no matter whether exact or similar match is searched for
  case EQU_REQ:
   if (!(TEST (mtm,MTM_X0) || TEST (mtm,MTM_X1)))
    return (FALSE);
   break;
  // optional, only considered when optional fields are searched
  case EQU_OPT:
   if (!TEST (mtm,MTM_O1))
    return (FALSE);
   break;
  // key fields have been processed before ???
  case EQU_KEY:
   return (FALSE);

  default:
   return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/** generate expression for fields
 * @param ctx pointer to matching expression context
 * @param exp pointer to pointer to expression after invocation
 *
 * dsc: for each schema item one expression is generated
 *                                                                            
 * exp:                                                                       
 *                                                                            
 *      AND                                                             <exp> 
 *      |                                                                     
 *      +- <e1>                                                               
 *      |                                                                     
 *      :                                                                     
 *      |                                                                     
 *      +- <en>                                                               
 *                                                                            
 *
 */
 
static BOOL mti_exp_gen_fld (MTI_CTX_EXP * ctx,PSX_EXP ** exp)
{
 if (ctx == NULL)
  return (FALSE);

 if (!exp_create (exp))
  return (FALSE);

 exp_assign (*exp,ETP_LOP,LOP_AND);
 /* iterate all fields specified in fsp=format specification
     ctx -> sch -> fsp -> n : number of fields */
 for (ctx -> idx = 0;ctx -> idx < ctx -> sch -> fsp -> n;ctx -> idx++)
 {
  /* check if current field should be considered in sql-query
     ctx -> mtm: match mode, encodes current status in KSXO notation
     EQU_CMP: required (EQU_REQ 0x02), optional (EQU_OPT 0x03), or key field (EQU_KEY 0x01) */ 
  if (mti_exp_consider (ctx -> mtm,EQU_CMP (ctx -> sch -> fsp -> itm [ctx -> idx].equ)))
  {
   PSX_EXP * e;

   if (!mti_exp_gen_itm (ctx,&e))
   {
    psx_put_v ("mti: exp gen sch s failed");
    return (FALSE);
   }

   if (e)
    exp_add (*exp,e);
  }

 }

 return (TRUE);
}

/******************************************************************************/
/* Private                                                       Schema Level */
/******************************************************************************/
/** validate any key fields for input
 * @param ctx pointer to matching expression context
 */

static BOOL mti_exp_gen_key_validate (MTI_CTX_EXP * ctx)
{
 IDX i;

 if (ctx == NULL)
  return (FALSE);

 for (i = 0;i < ctx -> sch -> fsp -> n;i++)
  if (EQU_CMP (ctx -> sch -> fsp -> itm [i].equ) == EQU_KEY)
  {
   PSX_STR t;

   if (!sch_lookup (ctx -> sch,i,CMP_NIL,ctx -> rec,t))	// get input value
    return (FALSE);

   if (str_isnil (t))
    return (FALSE);
  }

 return (TRUE);
}


/******************************************************************************/
/** generate expression for key condition
 * @param ctx pointer to matching expression context
 * @param exp pointer to pointer to expression after invocation
 */

static BOOL mti_exp_gen_key (MTI_CTX_EXP * ctx,PSX_EXP ** exp)
{
 PSX_EXP * e;
 PSX_STR t;
 BOOL r;

 if (ctx == NULL || exp == NULL)
  return (FALSE);

 *exp = NULL;

 if (mti_exp_gen_key_validate (ctx))
 {
  if (!exp_create (exp))
   return (FALSE);

  exp_assign (*exp,ETP_LOP,LOP_AND);

  for (ctx -> idx = 0;ctx -> idx < ctx -> sch -> fsp -> n;ctx -> idx++)
  {
   if (ctx -> mtm,EQU_CMP (ctx -> sch -> fsp -> itm [ctx -> idx].equ) == EQU_KEY)
   {
    PSX_EXP * e;

    if (!mti_exp_gen_itm (ctx,&e))
    {
     psx_put_v ("mti: exp gen sch s failed");
     return (FALSE);
    }

    if (e)
     exp_add (*exp,e);
   }
  }
  r = TRUE;
  LOG (MOD,"> key complete -> key expression");
 }
 else
 {
  if (!exp_create (&e))
   return (FALSE);

  exp_assign (e,ETP_CON,"FALSE");
  *exp = e;
  LOG (MOD,"> key incomplete -> false expression");
  r = TRUE;
 }


 return (r);
}

/******************************************************************************/
/** generate expression for substitution condition   <<<< under construction >>>>
 * @param ctx pointer to matching expression context
 * @param exp pointer to pointer to expression after invocation
 */

static BOOL mti_exp_gen_sub (MTI_CTX_EXP * ctx,PSX_EXP ** exp)
{
 return (FALSE);
}

/******************************************************************************/
/** generate match expression for schema                                  
 *                                                                            
 * exp:                                                                       
 *                                                                            
 *      AND                                                               <e> 
 *      |                                                                     
 *      +-sur=<sur>                                                      <e1> 
 *      |                                                                     
 *      +-exp [key]                                                      <e2> 
 *      |                                                                     
 *      +-exp [s<sur>]                                                   <e3> 
 *
 * @param ctx pointer to matching expression context                                                                             
 */

static BOOL mti_exp_gen_run (MTI_CTX_EXP * ctx)
{
 PSX_EXP * e,* e1,* e2,* e3, * es;

 /* create substitution condition */
 /* is done in pdi_mat_sql_gen() */

 /* create sureness condition */

 if
 (
  (TEST (ctx -> mtm,MTM_S1) && !TEST (ctx -> mtm,MTM_S0))
  ||
  (TEST (ctx -> mtm,MTM_S0) && !TEST (ctx -> mtm,MTM_S1))
 )
 {
  BOOL s = TEST (ctx -> mtm,MTM_S1);

  if (!mti_exp_gen_cmp_s (ctx,s,&e1)) 
   return (FALSE);
 }
 else
  e1 = NULL;
  
 /* create key condition */

 if (TEST (ctx -> mtm,MTM_K1))
 {
  if (!mti_exp_gen_key (ctx,&e2))
   exp_delete_n (1,&e1);
 }
 else
  e2 = NULL;

 /* create field condition */

 //if (!TEST (ctx -> mtm,MTM_K1))		// if key then only key
 //{
 
  if (!mti_exp_gen_fld (ctx,&e3))
  {
   exp_delete_n (2,&e1,&e2);
   return (FALSE);
  }
  
 //}
 //else
 // e3 = NULL;

 /* pack if useful */

 if (!exp_pack (&e,ETP_LOP,LOP_AND,3,e1,e2,e3))
 {
  exp_delete_n (3,&e1,&e2,&e3);
  return (FALSE);
 }

 *ctx -> exp = e;

 return (TRUE);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


