/******************************************************************************/
/* psx-pdi-mat.inl                         PID Database Interface - Matching  */
/******************************************************************************/
/** @file psx-pdi-mat.inl PID Database Interface: Matching - Inline Source Code File
 * Functions implementing the interface for matching (matching is provided by
 * the mti module).
 * This module provides
 * - SQL statement generation for matching
 * - evaluation of matching
 * - message creation for logging
 * - initialisation of the matching context
 * - starting the request matching
 */


/**************************************/
/*                                    */
/**************************************/

static char * pdi_mat_sql_sel = "SELECT pid,sur FROM rec";

/******************************************************************************/
/*                                                                            */


/******************************************************************************/
/** generate sql statement for matching
 * @param ctx pointer to matching context
 * @param mtm matchmode
 */

static BOOL pdi_mat_sql_gen (CTX_MAT * ctx,MTM mtm)
{
 PSX_STR exp_null;

 if (ctx == NULL)
  return (FALSE);

 if (ctx -> cnd)
 {
  free (ctx -> cnd);
  ctx -> cnd = NULL;
 }

 if (ctx -> sql)
 {
  free (ctx -> sql);
  ctx -> sql = NULL;
 }
                                 
 if (!mti_exp_gen_sql (ctx -> sch,ctx -> rec,mtm,&ctx -> cnd)) // generate condition   
  return (FALSE);

 if ((ctx -> sql = (char *) malloc (strlen (ctx -> cnd) + strlen (pdi_mat_sql_sel) + 100)) == NULL)
  return (FALSE);

 if (!dbi_exp_isnull ("rec.sub",exp_null))
  return (FALSE);
 // create sql statement
 strcpy (ctx -> sql,pdi_mat_sql_sel);
 strcat (ctx -> sql," WHERE ");
 //strcat (ctx -> sql,"rec.sub ISNULL");
 strcat (ctx -> sql,exp_null);  // rec.sub IS NULL (do not select PIDs that got substituted)
 strcat (ctx -> sql," AND ");
 strcat (ctx -> sql,ctx -> cnd); //condition
 //strcat (ctx -> sql,";");

 return (TRUE);
}

/******************************************************************************/
/** evaluate matching
 * @param mtr BYTE pointer to match result. Takes one of the following values:
 *		  <ul>
 *		    <li> MTR_ZERO: An error occured
 *			<li> MTR_X0: No matching records were found.
 *			<li> MTR_X1: Exactly one matching record was found
 *			<li> MTR_G1: More than one matching record was found
 *		  <ul>
 * @param ctx Pointer to matching context
 *
 * @returns FALSE if an error occured, TRUE otherwise
 */

static BOOL pdi_mat_evl (CTX_MAT * ctx,BYTE * mtr)
{
 char * t,pid [1000];
 BOOL sur;

 if (ctx == NULL || mtr == NULL)
  return (FALSE);

 *mtr = MTR_ZERO;
 strcpy (pid,"");

 if (!dbi_fetch (pdi.dbi,0,NULL))	/* get first item */
 {
  *mtr = MTR_X0;	// failure gives result MTR_X0
  return (TRUE);
 }
 /* if this line is reached, at least one matching record could be found */

 if (!dbi_fetch (pdi.dbi,0,&t))   /* get PID */
  return (FALSE);

 strcpy (pid,t);

 if (!dbi_fetch (pdi.dbi,1,&t)) /* get sureness */
  return (FALSE);

 if (strcmp (t,"1") == 0)// strcmp (t, "t");
  sur = TRUE;
 else
  sur = FALSE;

 if (!dbi_fetch (pdi.dbi,0,NULL))	/* get next item */
 {
  *mtr = MTR_X1;  // failure gives result MTR_X1
  strcpy (ctx -> pid,pid);
  ctx -> mrs = sur; /* matched record sureness */
  return (TRUE);
 }

 *mtr = MTR_G1;

 return (TRUE);
}


/******************************************************************************/
/** create message for logging
 * @param mtm matchmode
 * @param mtr result of matching
 */

static BOOL mat_sel_log (BYTE mtm,BYTE mtr)
{
 PSX_STR s_mtm,s_mtr,msg;

 mti_mtm_enc (mtm,s_mtm);

 switch (mtr)
 {
  case MTR_X0: strcpy (s_mtr,"0"); break;
  case MTR_X1: strcpy (s_mtr,"1"); break;
  case MTR_G1: strcpy (s_mtr,">1"); break;
  default:     strcpy (s_mtr,"?"); break;
 }

 sprintf (msg,"SEL: %s -> %s",s_mtm,s_mtr);

// psx_put ("%s",msg);

 return (TRUE);
}

/******************************************************************************/
/** select statement for match procedure
 * @param ctx pointer to matching context
 * @param mtm matchmode
 */

static BYTE mat_sel (CTX_MAT * ctx,BYTE mtm)
{
 BYTE mtr;

 if (ctx == NULL)
  return (MTR_ZERO);

 if (!pdi_mat_sql_gen (ctx,mtm))       // generate sql statement for matching
  return (MTR_ZERO);

 assert (ctx -> sql != NULL);

 // LOG (MOD,"performing select:\n%s",ctx -> sql);

 if (!dbi_execute (pdi.dbi,ctx -> sql))
  return (MTR_ZERO);

 if (!pdi_mat_evl (ctx,&mtr))          // evaluate result
  return (MTR_ZERO);

 //JM: wenn record update mode = else begin select * from rec where pid = ctx.pid

 if (ctx -> cnd)	// condition
 {
  free (ctx -> cnd);
  ctx -> cnd = NULL;
 }

 if (ctx -> sql)
 {
  free (ctx -> sql);
  ctx -> sql = NULL;
 }

#if 1

 mat_sel_log (mtm,mtr);               // generate log message

#endif

 return (mtr);
}

/******************************************************************************/
/*                                                                            */
/******************************************************************************/
/** initialize matching context
 * @param ctx pointer to matching context
 * @param pointer to request structure
 * @param rec pointer to record structure
 */
 
static BOOL pdi_mat_init (CTX_MAT * ctx,PDI_REQ * req,PSX_REC * rec)
{
 if (ctx == NULL || rec == NULL)
  return (FALSE);

 memset (ctx,0,sizeof (*ctx));

 if (!pdi_sch_get (&ctx -> sch))
 {
  return (FALSE);
 }

 //exp_create (&ctx -> exp);

 ctx -> rec = rec;
 ctx -> req = req;
 ctx -> mrs = FALSE;
 ctx -> sel = mat_sel;      // mat_sel is the name of a function

 return (TRUE);
}

/******************************************************************************/
/** exit matching
 * @param ctx pointer to matching context
 */

static BOOL pdi_mat_exit (CTX_MAT * ctx)
{
 BOOL r = TRUE;

 if (ctx == NULL)
  return (FALSE);

 if (ctx -> sch)
  ctx -> sch = NULL;

 ctx -> rec = NULL;

 if (ctx -> cnd)
 {
  free (ctx -> cnd);
  ctx -> cnd = NULL;
 }

 if (ctx -> sql)
 {
  free (ctx -> sql);
  ctx -> sql = NULL;
 }

 //exp_delete (&ctx -> exp);

 return (r);
}

/******************************************************************************/
/** perform a matching for a request
 * @param req pointer to request
 */

static BOOL pdi_match (PDI_REQ * req)
{
 CTX_MAT ctx;
 BOOL r = TRUE;
 PSX_STR t;
 
 LOG (MOD,"matching start");

 if (req == NULL)
  return (FALSE);

// req -> mat = MAT_ZERO;

 SYS_DBG_MEM_BEGIN

if (!pdi_mat_init (&ctx,req,req -> rec))  // initialize matching
  return (FALSE);

 if (!mti_execute (&ctx))                  // execute matching
  r = FALSE;

// req -> mat = ctx.mat;

 if (!pdi_mat_exit (&ctx))                 // exit matching
  r = FALSE;

 SYS_DBG_MEM_END

 if (r)
  memcpy (req -> pid,ctx.pid,8);

 req -> mrs = ctx.mrs;                     // store matched record sureness
 req -> rsl = ctx.rsl;                     // store result 

// mti_mat_sym_get (req -> mat,t);
 assert (pdi.sch != NULL);

 LOG (MOD,"matching stop, %s, rsl = %s",r ? "succeed" : "failure",pdi.sch -> rsp -> itm [req -> rsl].sym);

 return (r);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


