/******************************************************************************/
/* psx-pdi-rgs.inl                       Pseudonymisation Database Interface  */
/******************************************************************************/
/** @file psx-pdi-rgs.inl PID Database Interface: Registry Interface - Inline Source Code File
 * Functions to support registry handling like
 * - retrieving a list of all fields
 * - database index creation
 * - schema handling
 * - creation of the table rec and request
 */


/******************************************************************************/
/** get list of fields from registry
 * @param sch Pointer to schema structure
 * @param fld_lst Pointer to string list, contains comma separated list of fields after invocation 
 */
static BOOL pdi_rgs_get_fld_lst (PSX_SCH * sch,PSX_STL * fld_lst)
{
 IDX i;
 PSX_STL * stl;

 if (sch == NULL)
  return (FALSE);
 
 if (fld_lst == NULL)
	return (FALSE);

 if (!stl_create (&stl))         // create string list
	return (FALSE);

 if (!sch_sym_get_stl (sch,stl)) // get stringlist
 {
  stl_delete (&stl);
  return (FALSE);
 }


 for (i = 0;i < stl -> n;i++)
 {
  PSX_STR t;
  
#ifdef WIN32
  sprintf (t,"f_%s VARCHAR(255)",stl -> str [i]); // BG: VARCHAR berall gltig?
#else
  sprintf (t,"f_%s TEXT",stl -> str [i]);        // JM: warum nicht immer varchar? text entspricht 64 KB!
#endif

  if (i < stl -> n - 1) // add comma except after last field
	strcat (t,",");

  if (!stl_add (fld_lst,t))
	return (FALSE);
 }

 stl_delete (&stl);

 return (TRUE);
}

/******************************************************************************/

/******************************************************************************/
/** get list of fields from registry, append "_o"
 * @param sch pointer to schema structure
 * @param str String pointer, string contains comma separated list of fields with appedix "old" after invocation 
 */
 
static BOOL pdi_rgs_get_fld_lst_old (PSX_SCH * sch,PSX_STL * fld_lst)
{
 IDX i;
 PSX_STL * stl;

 if (sch == NULL)
  return (FALSE);
 
 if (fld_lst == NULL)
	return (FALSE);

 if (!stl_create (&stl))         // create string list
	return (FALSE);

 if (!sch_sym_get_stl (sch,stl)) // get stringlist
 {
  stl_delete (&stl);
  return (FALSE);
 }


 for (i = 0;i < stl -> n;i++)
 {
  PSX_STR t;
  
#ifdef WIN32
  sprintf (t,"f_%s_o VARCHAR(255)",stl -> str [i]); // BG: VARCHAR berall gltig?
#else
  sprintf (t,"f_%s_o TEXT",stl -> str [i]);        // JM: warum nicht immer varchar? text entspricht 64 KB!
#endif

  if (i < stl -> n - 1) // add comma except after last field
	strcat (t,",");

  if (!stl_add (fld_lst,t))
	return (FALSE);
 }

 stl_delete (&stl);

 return (TRUE);
}

/******************************************************************************/
/** save schema and set schema configuration
 * @param sch pointer to schema
 */
 
static BOOL pdi_rgs_sch (PSX_SCH * sch)
{
 IDX idx,i;
 PSX_STR s;
 BOOL r = TRUE;

 if (sch == NULL)
  return (FALSE);

 if (!sch_save (sch,pdi.dbi,&idx))
  return (FALSE);

 sprintf (s,"%d",idx);

 if (!pdi_cfg_set ("sch",s))
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** create index in database
 * @param stl pointer to string list with field names
 */

static BOOL pdi_rgs_rec_idx (PSX_STL * stl)
{
 BOOL r = TRUE;
 IDX i;

 if (stl == NULL)
  return (FALSE);

 // create index on sureness table
 if (!dbi_execute (pdi.dbi,"CREATE INDEX idx_rec_sur ON rec (sur)"))
  r = FALSE;

 for (i = 0;i < stl -> n;i++)
 {
  char sql [1000];

  // create index on other fields
  sprintf (sql,"CREATE INDEX idx_rec_f_%s ON rec (f_%s)",stl -> str [i],stl -> str [i]);

  if (!dbi_execute (pdi.dbi,sql))  
   r = FALSE;
 }

 return (r);
}

/******************************************************************************/
/** create table rec with fields from registry
 * @param sch pointer to schema structure
 */
 
static BOOL pdi_rgs_rec (PSX_SCH * sch)
{
 BOOL r = TRUE;
 IDX i;
 //char *  sql,* s;         //JM: PSX_STR
 //PSX_LSTR sql ,s;
 PSX_STL * sql;
 char * sql_str;
 
 if (sch == NULL)
  return (FALSE);

 if (!stl_create(&sql))
	 return (FALSE);

 if (!stl_add (sql,"CREATE TABLE rec (pid CHAR(8) PRIMARY KEY,sub CHAR(8) REFERENCES rec(pid),sur INT,"))
	 return (FALSE);

 if (!pdi_rgs_get_fld_lst (sch,sql))  // get list of fields from registry (appended to sql)
  return (FALSE);
 
 if (!stl_add(sql,")"))
	 return (FALSE);

 if (!stl_cat(sql,NULL,&sql_str))
	 return (FALSE);

 if (!dbi_execute (pdi.dbi,sql_str))    
  r = FALSE;

 if (PDI_CFG_INDEXES == 1)
 {
  PSX_STL * stl;

  if (!stl_create (&stl))           // create string list
   return (FALSE);

  if (!sch_sym_get_stl (sch,stl))   // get string list elements from schema
  {
   stl_delete (&stl);
   return (FALSE);
  }

  if (!pdi_rgs_rec_idx (stl))       // create index on database fields
   r = FALSE;

  stl_delete (&stl);
  stl_delete(&sql);
  free (sql_str);
 }

 return (r);
}

/******************************************************************************/
/** create database table request with fields from registry
 * @param sch pointer to schema structure
 */
 
static BOOL pdi_rgs_req (PSX_SCH * sch)
{
 BOOL r = TRUE;
 IDX i;
 char * sql_str;
 PSX_STL * sql;

 if (sch == NULL)
  return (FALSE);

 if (!stl_create(&sql))
	return (FALSE);

 if (!stl_add (sql,"CREATE TABLE req (idx INT PRIMARY KEY,pid CHAR(8),tsp TEXT,adr TEXT, prt TEXT,hst TEXT,usr TEXT,sur INT, result TEXT,mode TEXT,upd INT,")) //JM: added 'result TEXT,mode TEXT,upd INT'
	return (FALSE);

 if (!pdi_rgs_get_fld_lst (sch,sql))     // add list of fields from registry
  return (FALSE);

 if (!stl_add (sql, ","))
	 return (FALSE);

 //JM: added for old values
 if (!pdi_rgs_get_fld_lst_old (sch,sql))     // add list of fields from registry
  return (FALSE);

 if (!stl_add(sql,")"))
	 return (FALSE);

 if (!stl_cat(sql,NULL,&sql_str))
	 return (FALSE);

 if (!dbi_execute (pdi.dbi,sql_str))
  r = FALSE;

 //if (PDI_CFG_INDEXES == 1)
 // if (!pdi_rgs_rec_idx (stl))
 //  r = FALSE;
  stl_delete(&sql);
  free (sql_str);

 return (r);
}

/******************************************************************************/





/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


