/******************************************************************************/
/* psx-pdi.c                              Pseudonymization Database Interface */
/******************************************************************************/
/** @file psx-pdi.c
 * PID Database Interface - SourceCode File
 * Functions to support the PID generation by providing the following
 * funcionalities:
 * - configuration of the PDI Context
 * - creation and deletion of a database
 * - opening and closing of a connection to the database
 * - registering the schema in database
 * - retrieval of the format specification from database
 * - handling PID request and index
 * - invoke PID generation and handle results
 * - retrieval of a PID from database for matching
 */
 
#include "psx.h"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

#define MOD PSX_MOD_PDI
#define	PDI_CFG_INDEXES	1

#include "psx-pdi.inl"
#include "psx-pdi-rgs.inl"
#include "psx-pdi-mat.inl"

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** configure PDI
 * @param cfg pointer to PDI configuration structure
 */

BOOL pdi_cfg (PDI_CFG * cfg)
{
 PDI_CFG c;
 PSX_STR t;

 if (cfg)
  c = *cfg;
 else
 {
  PSX_STR t;

  memset (&c,0,sizeof (c));
  
  // initialize configuration struct
  if (psx_cfg_get ("pid.k1",t))
   c.k1 = atoi (t);  
  if (psx_cfg_get ("pid.k2",t))
   c.k2 = atoi (t);
  if (psx_cfg_get ("pid.k3",t))
   c.k3 = atoi (t);
  if (psx_cfg_get ("pid.rw",t))
   c.rw = atoi (t);
 }
      
 k1       = c.k1;
 k2       = c.k2;
 k3       = c.k3;
 rndwidth = c.rw;

 sprintf (t,"k1=%d,k2=%d,k3=%d,rw=%d",c.k1,c.k2,c.k3,c.rw);
 LOG (MOD,"configuration\n%s",t);

 return (TRUE);
}

/******************************************************************************/
/** create database
 */

BOOL pdi_create (void)
{
 DBI * dbi;
 int i,n;
 BOOL r;
 PSX_STR dbs;

 LOG (MOD,"create database");

 if (pdi.dbi != NULL)
  return (FALSE);

 // 17-01-05
 if (!psx_cfg_get (PSX_CFG_DBS,dbs))  // get database configuration string
  return (FALSE);

 if (!dbi_create (dbs))               // create database
 {
  psx_put ("pdi: create database failed");
  return (FALSE);
 }

 if (!dbi_open (&dbi,dbs))            // connect to database
 {
  LOG (MOD,"database open error [%s]",dbs);
  return (FALSE);
 }

 n = sizeof (sql_gen) / sizeof (sql_gen [0]);

 r = TRUE;

 for (i = 0;i < n;i++)
 {
  if (!dbi_execute (dbi,sql_gen [i]))  // execute sql command
  {
   psx_put ("SQL ERROR");
   r = FALSE;
  }

  //  dbi_clear (dbi);
 }

 for (i = 0;i < CMP_N;i++)
 {
  PSX_STR s,t;

  if (!cpi_cmp_enc (i,t))
   return (FALSE);

  sprintf (s,"INSERT INTO ctp (sym) VALUES ('%s')",t);

  if (!dbi_execute (dbi,s))
  {
   psx_put ("SQL ERROR");
   r = FALSE;
  }

  // dbi_clear (dbi);
 }

 if (!dbi_close (&dbi))                 // close connection to database
  return (FALSE);

 pdi.sts |= PDI_STS_INIT;

 if (!pdi_open ())                      // open database
 {
  pdi.sts &= ~PDI_STS_INIT;
  return (FALSE);
 }

 pdi.sts &= ~PDI_STS_INIT;

 if (!pdi_pix_set (0))                 // set PID Index
 {
  psx_put ("PIX ERROR");
  r = FALSE;
 }

 if (!pdi_cfg_set ("sch",""))          // set configuration for sch
  r = FALSE;

 if (!pdi_close ())
  r = FALSE;

 LOG (MOD,"create database: result: %s",r ? "ok" : "failed");

 return (r);
}

/******************************************************************************/
/** delete database
 */

BOOL pdi_delete (void)
{
 BOOL r = TRUE;

 PSX_STR dbs;

 LOG (MOD,"delete database");

 if (!psx_cfg_get (PSX_CFG_DBS,dbs))
  return (FALSE);

 if (!dbi_delete (dbs))
  r = FALSE;

 LOG (MOD,"delete database: result: %s",r ? "ok" : "failed");

 return (r);
}

/******************************************************************************/
/** open database
 */

BOOL pdi_open (void)
{
 PSX_STR dbs;

 LOG (MOD,"open database");

 if (!psx_cfg_get (PSX_CFG_DBS,dbs))  // get dbs string
  return (FALSE);

 if (pdi.dbi != NULL)
 {
  assert (pdi.sem > 0);
  pdi.sem++;
  return (TRUE);
 }

 if (!dbi_open (&pdi.dbi,dbs)) // PDI_DBS))
 {
  LOG (MOD,"database open error [%s]",dbs);
  return (FALSE);
 }

 if (!sch_create (&pdi.sch))       // create schema
 {
  dbi_close (&pdi.dbi);
  LOG (MOD,"database open error [%s]",dbs);
  return (FALSE);
 }

 if (!stl_create (&pdi.atr))
 {
  sch_delete (&pdi.sch);
  dbi_close (&pdi.dbi);
  LOG (MOD,"database open error [%s]",dbs);
  return (FALSE);
 }

 if (!pdi_sch_load (pdi.sch))
 {
   
  if (!TEST (pdi.sts,PDI_STS_INIT)) 
  {
   psx_put ("pdi: schema load error");
   dbi_close (&pdi.dbi);
   LOG (MOD,"database open error [%s]",dbs);
   return (FALSE);
  }
 }

 sch_sym_get_stl (pdi.sch,pdi.atr);

 pdi.sem++;

 LOG (MOD,"open database: result: ok, semaphore=%d",pdi.sem);

 return (TRUE);
}

/******************************************************************************/
/** close database
 */

BOOL pdi_close (void)
{
 BOOL r = TRUE;

 LOG (MOD,"close database");

 if (pdi.dbi == NULL)
  return (FALSE);

 assert (pdi.sem > 0);          

 if (--pdi.sem == 0)            
 {
  if (!dbi_close (&pdi.dbi))
   r = FALSE;

  if (pdi.sch)
   sch_delete (&pdi.sch);

  if (pdi.atr)
   stl_delete (&pdi.atr);
 }

 LOG (MOD,"close database: result: %s, semaphore=%d",r ? "ok" : "failed",pdi.sem);

 return (r);
}

/******************************************************************************/
/** register schema
 * @param sch pointer to schema structure
 */

BOOL pdi_register (PSX_SCH * sch)
{
 IDX idx,i;
 PSX_STR s,sql;
 BOOL r = TRUE;

 if (sch == NULL)
  return (FALSE);

 pdi.sts |= PDI_STS_INIT;                

 if (!pdi_open ())
 {
  pdi.sts &= ~PDI_STS_INIT;
  psx_put ("database open error");
  return (FALSE);
 }

 pdi.sts &= ~PDI_STS_INIT;

 if (pdi.dbi == NULL)
  return (FALSE);

 /*****************/

 if (sch -> fsp -> n == 0)
 {
  pdi_close ();
  psx_put ("pdi: schema has no fields");
  return (FALSE);
 }

 /*****************/

 if (!pdi_rgs_sch (sch))                 
  r = FALSE;
 else
  if (!pdi_rgs_rec (sch))
   r = FALSE;
  else
   if (!pdi_rgs_req (sch))
    r = FALSE;

 /*****************/

 if (!pdi_cfg_set ("req",""))
  r = FALSE;

 pdi_close ();

 return (r);
}

/******************************************************************************/
/** retrieve schema from pdi structure
 * @param pointer to schema pointer
 */

BOOL pdi_sch_get (PSX_SCH ** sch)
{

 if (sch == NULL)
  return (FALSE);

 if (pdi.sch == NULL)
  return (FALSE);

 *sch = pdi.sch;

 return (TRUE);
}

/******************************************************************************/
/** get request
 * @param time specification
 */

BOOL pdi_req_get (PSX_STR tsp)
{
 char * t;

 if (tsp == NULL)
  return (FALSE);

 if (!pdi_cfg_get ("req",&t)) 
  return (FALSE);

 if (t == NULL)
  return (FALSE);

 strcpy (tsp,t);

 return (TRUE);
}

/******************************************************************************/
/** get formatspecification from database
 * @param pointer to format
 */

BOOL pdi_fmt_get (PSX_FMT * fmt)
{
 char * t;
 PSX_STR sql;

ASSERT (FALSE);

 if (fmt == NULL)
  return (FALSE);

 if (!fsp_clr (fmt))
  return (FALSE);

 if (!pdi_cfg_get ("sch",&t))
  return (FALSE);

 sprintf (sql,"SELECT idx,sym,lbl,dtp,mini,maxi,tfm,equ FROM itm WHERE (sch='%s') ORDER BY idx",t);

 if (!dbi_execute (pdi.dbi,sql))
  return (FALSE);

 while (dbi_fetch (pdi.dbi,0,NULL))
 {
  FMT_ITM itm;

  if (!dbi_fetch (pdi.dbi,1,&t))
   return (FALSE);

  strcpy (itm.sym,t);

  itm.pos = 0;
  itm.len = 0;

  if (!dbi_fetch (pdi.dbi,2,&t))
   return (FALSE);

  strcpy (itm.lbl,t);

  if (!dbi_fetch (pdi.dbi,3,&t))
   return (FALSE);

  if (!fsp_dtp_set (&itm.dtp,t))
   return (FALSE);

  if (!dbi_fetch (pdi.dbi,4,&t))
   return (FALSE);

  itm.min = atoi (t);

  if (!dbi_fetch (pdi.dbi,5,&t))
   return (FALSE);

  itm.max = atoi (t);


  if (!dbi_fetch (pdi.dbi,6,&t))
   return (FALSE);

  if (!fsp_tfm_dec (&itm.tfm,t))
   return (FALSE);

  if (!dbi_fetch (pdi.dbi,7,&t))
   return (FALSE);

  if (!fsp_equ_dec (&itm.equ,t))
   return (FALSE);

  if (!fsp_add (fmt,&itm))
   return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/** get PID index
 * @param pix PID index
 */

BOOL pdi_pix_get (PIX * pix)
{
 char * t;
 char * e = NULL;
 long int x;

 if (pix == NULL)
  return (FALSE);

 //if (!dbi_lookup (pdi.dbi,"cfg","val","atr='pix'",&t))
 if (!pdi_cfg_get ("pix",&t))
 {
  if (!pdi_pix_set (0))
    return (FALSE);

  *pix = 0;

  return (TRUE);
 }

 assert (t != NULL);

 x = strtol (t,&e,16);

 if (e != t + strlen (t))
  return (FALSE);

 *pix = x;

 return (TRUE);
}

/******************************************************************************/
/** insert or update PID index (used to create PID) in table cfg
 * @param pix PID index
 */

BOOL pdi_pix_set (PIX pix)
{
 PSX_STR t;

 sprintf (t,"%08X",pix);

 //if (!dbi_update (pdi.dbi,"cfg","atr='pix'","val",t))
 if (!pdi_cfg_set ("pix",t))
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** generate PID
 * @param pid pseudonymisation identifier
 * @param pix pseudonymisation identifier index
 */

BOOL pdi_pid_make (PID pid,PIX pix)
{
 PSX_STR t;
 NUM k,b;
 IDX i,j;
 char * pom;

 if (pid == NULL)
  return (FALSE);

 pom = PIDgen (pix);

 assert (pom != NULL && strlen (pom) == 8);
 memcpy (pid,pom,8);

 return (TRUE);
}

/******************************************************************************/
/** lookup PID
 * @param rec pointer to PSX record, contains values after invocationi
 * @param pid personal identifier
 */

BOOL pdi_lookup (PSX_REC * rec,PSX_STR pid)
{
 PSX_STL * stl;
 PSX_STR   fld;
 IDX       i;
 BOOL      r = TRUE;
 PSX_STR   sql;
//JM:
 char *p1, *p2;
 int len,j;
 //end
 
 pid [8] = 0;

 if (/*rec == NULL || */pid == NULL) /* allow existence test only */
  return (FALSE);

 ASSERT (pdi.sch != NULL);

 LOG (MOD,"lookup [pid='%s']",pid != NULL ? pid : "NULL");

 if (!stl_create (&stl))
  return (FALSE);

 //LOG (MOD,"lookup 1");

 if (!sch_sym_get_stl (pdi.sch,stl))
 {
  stl_delete (&stl);
  return (FALSE);
 }

 strcpy (fld,"");

 // get all field names and store comma seperated list in string fld
 for (i = 0;i < stl -> n;i++)  
 {
   PSX_STR s;

   if (i > 0)
     strcat (fld,",");

   sprintf (s,"f_%s",stl -> str [i]); // "f_"  is prefix of field names in database
   strcat (fld,s);
 }

 // select fields from rec where pid = pid
 sprintf (sql,"SELECT %s FROM rec WHERE pid='%s'",fld,pid);

 //LOG (MOD,"lookup 2, pid='%s'",pid);

 if (!dbi_execute (pdi.dbi,sql))
 {
  stl_delete (&stl);
  return (FALSE);
 }

 if (!dbi_fetch (pdi.dbi,0,NULL))
 {
  stl_delete (&stl);
  return (FALSE);
 }

 //LOG (MOD,"lookup 3");

 if (rec)
 {
    for (i = 0;i < stl -> n;i++)
    {
       PSX_STR atr;
       char * t;

       if (!dbi_fetch (pdi.dbi,i,&t))
      {
           r = FALSE;
          break;
      }

      //JM: clean up values from database, because NULL values get messed up  under Windows
      // (Windows XP, MySQL)   
      len = strlen(t);

      for (j = 0,p1 = p2 = t;j < len;j++)
      {                
         if ( (*p1 >= ' ' && *p1 <= '~') || *p1 == ''  || *p1 == '' || *p1 == '' || *p1 == '' || *p1 == '' || *p1 == '' || *p1 == '')      // ASCII character set is acceptable
         {  
            *p2 = *p1;
            p2++;
         }
         p1++;
      }

      *p2 ='\0';
     //JM: end
  
      sprintf (atr,"%s",stl -> str [i]);
      rec_add1 (rec,atr,t);
    } // end for
 }   // end if (rec)
 
 return (TRUE);
}

/******************************************************************************/
/** PID generation
 * @param req pointer to PDI request structure
 */

BOOL pdi_gen (PDI_REQ * req)
{
 PSX_STR t;

 if (pdi.dbi == NULL || req == NULL)
  return (FALSE);

 if (!pdi_pix_get (&req -> pix))
  return (FALSE);

 if (!pdi_pid_make (req -> pid,req -> pix))
  return (FALSE);

 memcpy (t,req -> pid,8);
 t [8] = 0;

 return (TRUE);
}

/******************************************************************************/
/** request pid
 * @param req pointer to request struct
 * @return TRUE, if no system error
 */

BOOL pdi_req (PDI_REQ * req)
{
 PSX_STR t;
 PSX_STR sql; 

 BOOL r = TRUE,m;     
 PRM prm;


 LOG (MOD,"request");

 memset (req -> pid,0,sizeof (req -> pid));
 req -> pix = 0;

 if (pdi.dbi == NULL || req == NULL)
  return (FALSE);

// 17-01-05
 if(!psx_cfg_get(PSX_CFG_SQL_BEGIN, sql))		// get configuration for BEGIN TRANSACTION statement
	 strcpy(sql, "BEGIN TRANSACTION");

 // The following two statements have to be removed for MS Access
 if (!dbi_execute (pdi.dbi,sql))  
  return (FALSE);

 //dbi_clear (pdi.dbi); // 2002-06-28

 SYS_DBG_MEM_BEGIN
 m = pdi_match (req);      // match procedure 
 SYS_DBG_MEM_END

 LOG (MOD,"req: %s",m ? "ok" : "failed");

 if (!m)
 {
  pdi_req_log (req,NULL);  // log request: write into table 'req'
  return (FALSE);
 }

 prm = pdi.sch -> rsp -> itm [req -> rsl].prm;

 if (req -> frc)	// force!
 {
  psx_put ("force!");
  prm = PRM_GEN;
 }

 req -> prm = prm;

 switch (prm)  // check result of matching
 {
  case PRM_ZERO:
   LOG (MOD,"prm: PRM_ZERO");
   break;

  case PRM_GET:		// retrieve existing PID
   LOG (MOD,"prm: PRM_GET");

   if (!pdi_req_r (req))	// match has set req -> pid, update table rec
    r = FALSE;   
   break;

  case PRM_GEN:    // generate new PID
   LOG (MOD,"prm: PRM_GEN");

   if (!pdi_req_g (req))
    r = FALSE;
   break;

  default:
   LOG (MOD,"prm: ???");
   return (FALSE);
 }
 
 if (r) 
 {
	 if(!psx_cfg_get (PSX_CFG_SQL_COMMIT,sql))      // get configuration for COMMIT statement
	 	 strcpy(sql,"COMMIT TRANSACTION");
     
	 // The following two statements have to be removed for MS Access
	 if (!dbi_execute (pdi.dbi,sql))        
	 	r = FALSE;   
 }
 else
 {
    
	if(!psx_cfg_get (PSX_CFG_SQL_ABORT,sql))       // get configuration for ABORT statement
	 	 strcpy(sql,"ABORT TRANSACTION");    

	// The following statement have to be removed for MS Access
	dbi_execute (pdi.dbi,sql);
 }

 pdi_req_log (req,req -> pid);   // log request: write into table 'req'

 //dbi_clear (pdi.dbi); // 2002-06-28

 LOG (MOD,"pdi req: result: %s",r ? "success" : "failed");
                               
 return (r);
}

/******************************************************************************/
/** get PID index and fetch respective information from database
 * @param sts pointer to PDI status structure
 */
 
BOOL pdi_sts (PDI_STS * sts)
{
 char * t;

 if (sts == NULL)
  return (FALSE);

 memset (sts,0,sizeof (*sts));

 if (pdi.dbi == NULL)
  return (FALSE);

 /*****************/

 if (!pdi_pix_get (&sts -> pix))
  sts -> pix = IDX_NIL;

 /*****************/

 if (dbi_execute (pdi.dbi,"SELECT COUNT (pid) FROM rec"))
  if (dbi_fetch (pdi.dbi,0,NULL))
   if (dbi_fetch (pdi.dbi,0,&t))
    sts -> num_rec = atoi (t);

 /*****************/

 if (dbi_execute (pdi.dbi,"SELECT COUNT (idx) FROM req"))
  if (dbi_fetch (pdi.dbi,0,NULL))
   if (dbi_fetch (pdi.dbi,0,&t))
    sts -> num_req = atoi (t);

 /*****************/

 if (dbi_execute (pdi.dbi,"SELECT COUNT (idx) FROM req WHERE pid IS NOT NULL"))
  if (dbi_fetch (pdi.dbi,0,NULL))
   if (dbi_fetch (pdi.dbi,0,&t))
    sts -> num_req_pid = atoi (t);

 /*****************/
 if (!pdi_req_get (sts -> tsp_req))
  strcpy (sts -> tsp_req,"?");

 /*****************/

 return (TRUE);
}

/******************************************************************************/
/** get host
 * @param string pointer to hostname string
 */
 
BOOL pdi_hst_get (char ** h)
{
 PSX_BLK * blk;
 PSX_STL * stl;
 IDX idx;
 PSX_STR fls,sql;

 if (h == NULL)
  return (FALSE);

 if (!stl_create (&stl))
  return (FALSE);

 sch_get_fld (pdi.sch,fls);

 if (!sch_sym_get_stl (pdi.sch,stl))
  return (FALSE);

 if (!blk_create (&blk))
 {
  stl_delete (&stl);
  return (FALSE);
 }

 sprintf (sql,"SELECT pid,tsp,%s FROM req ORDER BY tsp DESC",fls);
 if (!dbi_execute (pdi.dbi,sql))
 {
  stl_delete (&stl);
  blk_delete (&blk);
  return (FALSE);
 }

 /*******/

 idx = 0;

 blk_put_s (blk,"<font face='courier' size='1'>");
 blk_put_s (blk,"<table border = '1'>");

 while (dbi_fetch (pdi.dbi,0,NULL)) // && idx++ < 1)
 {
  IDX i;
  PSX_STR pid,tsp;
  char * t, itm [5000];

  blk_put_s (blk,"<tr>");

  if (!dbi_fetch (pdi.dbi,0,&t))
   break;

  strcpy (pid,t);

  if (!dbi_fetch (pdi.dbi,1,&t))
   break;

  strcpy (tsp,t);

  blk_put_s (blk,"<td>");
  blk_put_s (blk,"<font face='courier' size='1'>");
  blk_put_s (blk,tsp);
  blk_put_s (blk,"</font>");
  blk_put_s (blk,"</td>");

  blk_put_s (blk,"<td>");
  blk_put_s (blk,"<font face='courier' size='1'>");
  blk_put_s (blk,pid);
  blk_put_s (blk,"</font>");
  blk_put_s (blk,"</td>");

  blk_put_s (blk,"<td>");
  blk_put_s (blk,"<font face='courier' size='1'>");

  for (i = 0;i < stl -> n;i++)
  {
   PSX_STR l;

   sprintf (l,"%s: ",stl -> str [i]);
   blk_put_s (blk,l);

   if (dbi_fetch (pdi.dbi,2 + i,&t))
    blk_put_s (blk,t);
   else
    blk_put_s (blk,"?");
   if (i + 1 < stl -> n)
    blk_put_s (blk,", ");
  }

  blk_put_s (blk,"</font>");
  blk_put_s (blk,"</td>");


  blk_put_s (blk,"</tr>");
 }

 blk_put_s (blk,"</table>");
 blk_put_s (blk,"</font>");

 /*******/

 if (!blk_get_s (blk,h))
 {
  stl_delete (&stl);
  blk_delete (&blk);
  return (FALSE);
 }

 stl_delete (&stl);

 if (!blk_delete (&blk))
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** get DBI structure from PDI structure
 * @param dbi pointer to DBI pointer
 */
 
BOOL pdi_get_dbi (DBI ** dbi)
{
 if (dbi == NULL)
  return (FALSE);

 if (pdi.dbi == NULL)
  return (FALSE);

 *dbi = pdi.dbi;
 
 return (TRUE);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


