/******************************************************************************/
/* psx-pdi.inl                                         PID Database Interface */
/******************************************************************************/
/** @file psx-pdi.inl PID Database Interface - Inline Source Code File
 * Definitions and functions supporting the PID generation and database handling.
 * - Definition of SQL statement templates
 * - database operations like writing the request into the table 'req' 
 */
 
/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

/**************************************/
/* PID Construction                   */
/**************************************/

#define PID_CHR_BIT	6				/* bit / chr */
#define PID_LEN_CHR	8				/* chr / pid */
#define PID_LEN_BIT	(PID_LEN_CHR * PID_CHR_BIT)	/* chr / pid */
#define PIX_CHR_BIT	6				/* bit / chr */

static const char * pid_chr = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ";

/**************************************/
/* PDI Status                         */
/**************************************/

#define PDI_STS_ZERO	0x00
#define PDI_STS_INIT	0x01

/**************************************/
/* PDI Status                         */
/**************************************/

/**Database and schema information.
 * Holds specification items, such as attribute names and matching procedure.
 */
static struct _PDI
{
 DBI     * dbi; /**< Database connection. */
 int       sem;
 PSX_SCH * sch; /**< Schema specification */
 /** Attribute names.
  * String list holding the names for components and phonetic 
  * transformations of fields. Typical values are lname_C1,...,lname_PC,...
  */
 PSX_STL * atr;
 BYTE      sts;
} 
/** Global Variable for database interface.
 * Stores Schema specification, database connection
 * and field names.
 */
pdi =
{
 NULL,
 0,
 PDI_STS_ZERO
};

/**************************************/
/* SQL                                */
/**************************************/

#ifdef WIN32
static char * sql_gen [] =
{
 "CREATE TABLE cfg (atr VARCHAR(64) PRIMARY KEY,val TEXT)",
 "CREATE TABLE ctp (sym VARCHAR(64) PRIMARY KEY)",
 "CREATE TABLE sch (idx INT PRIMARY KEY,sym TEXT NOT NULL)",
 "CREATE TABLE fld (idx INT PRIMARY KEY,sch INT REFERENCES sch (idx),sym TEXT NOT NULL,dtp TEXT,lbl TEXT,pos INT,len INT,mini INT,maxi INT,tfm TEXT,cps TEXT,equ TEXT)",
 "CREATE TABLE rsp (idx INT PRIMARY KEY,sch INT REFERENCES sch (idx),sym TEXT NOT NULL,prm CHAR,rum CHAR,msg TEXT,ntf TEXT)",
 "CREATE TABLE sts (idx INT PRIMARY KEY,sch INT REFERENCES sch (idx),sym TEXT NOT NULL,sps TEXT,tg0 TEXT,tg1 TEXT,tgm TEXT)",
// "CREATE TABLE sch_cmp (idx INT PRIMARY KEY,itm INT REFERENCES itm (idx),ctp TEXT REFERENCES ctp (sym))"
// "CREATE TABLE req (idx INT PRIMARY KEY,pid TEXT,tsp TEXT,adr TEXT,prt TEXT,hst TEXT,usr TEXT)"
};
#else
static PSX_STR sql_gen [] =
{
 "CREATE TABLE cfg (atr TEXT PRIMARY KEY,val TEXT)",
 "CREATE TABLE ctp (sym TEXT PRIMARY KEY)",
 "CREATE TABLE sch (idx INT PRIMARY KEY,sym TEXT NOT NULL)",
 "CREATE TABLE fld (idx INT PRIMARY KEY,sch INT REFERENCES sch (idx),sym TEXT NOT NULL,dtp TEXT,lbl TEXT,pos INT,len INT,mini INT,maxi INT,tfm TEXT,cps TEXT,equ TEXT)",
 "CREATE TABLE rsp (idx INT PRIMARY KEY,sch INT REFERENCES sch (idx),sym TEXT NOT NULL,prm CHAR,rum CHAR,msg TEXT,ntf TEXT)",
 "CREATE TABLE sts (idx INT PRIMARY KEY,sch INT REFERENCES sch (idx),sym TEXT NOT NULL,sps TEXT,tg0 TEXT,tg1 TEXT,tgm TEXT)",
// "CREATE TABLE sch_cmp (idx INT PRIMARY KEY,itm INT REFERENCES itm (idx),ctp TEXT REFERENCES ctp (sym))"
// "CREATE TABLE req (idx INT PRIMARY KEY,pid TEXT,tsp TEXT,adr TEXT,prt TEXT,hst TEXT,usr TEXT)"
};
#endif

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** get values from database table cfg
 * @param atr string pointer to attribute string
 * @param val pointer to string pointer to value string after invocation
 */
 
static BOOL pdi_cfg_get (const char * atr,char ** val)
{
 char * t,s [1000];

 if (atr == NULL)
  return (FALSE);

 sprintf (s,"atr='%s'",atr);

 if (!dbi_lookup (pdi.dbi,"cfg","val",s,&t))
  return (FALSE);

 if (val)
  *val = t;

 return (TRUE);
}

/******************************************************************************/
/** insert or update value into table cfg
 * @param atr string pointer to attribute
 * @param val string pointer to value string
 */
 
static BOOL pdi_cfg_set (const char * atr,const char * val)
{
 char * t,s [1000];

 if (atr == NULL)
  return (FALSE);

 sprintf (s,"atr='%s'",atr);

 if (dbi_lookup (pdi.dbi,"cfg","val",s,&t))
 {
  char v [1000];

  sprintf (v,"'%s'",val);

  if (!dbi_update (pdi.dbi,"cfg",s,"val",v))
   return (FALSE);
 }
 else
 {
  char v [1000];

  sprintf (v,"'%s','%s'",atr,val);

  if (!dbi_insert (pdi.dbi,"cfg","atr,val",v))
   return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** make value from source string
 * @param src pointer to source string
 */
 
static BOOL pdi_val_make (char * src)
{
 char s [1000],t [1000];

 if (src == NULL)
  return (FALSE);

 strcpy (s,src);
 str_trim (s);

 if (!str_isnil (s))
  sprintf (t,"'%s'",s);
 else
  strcpy (t,"NULL");

 strcpy (src,t);

 return (TRUE);
}

/******************************************************************************/
/** get schema definition (result, fields, states) from database and store in sch
 * @param sch pointer to schema
 */
 
static BOOL pdi_sch_load (PSX_SCH * sch)
{
 char * t;
 PSX_STR sql;
 IDX idx;

 if (sch == NULL)
  return (FALSE);

 if (!pdi_cfg_get ("sch",&t))   // get schema index from database table cfg
  return (FALSE);

 if (str_isnil (t))
  return (FALSE);

 idx = atoi (t);

 if (!sch_load (sch,pdi.dbi,idx))  // get field, result, state definitions from database (schema: idx)
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** insert new record into table 'rec'
 * @param pid PID of new record
 * @param rec pointer to new record
 * @param sur sureness specification of new record
 */
 
static BOOL pdi_insert (PID pid,PSX_REC * rec,BOOL sur)
{
 PSX_SCH * sch;
 PSX_LSTR sql,atr,val,s;
 BOOL r = FALSE;

 if (pid == NULL || rec == NULL)
  return (FALSE);

 if (pdi_sch_get (&sch))
  if (sch_sql_get_av (sch,rec,atr,val))
  {
   strncpy (s,pid,8);
   s [8] = 0;
   sprintf (sql,"INSERT INTO rec (pid,sur,%s) VALUES ('%s',%s,%s)",atr,s,sur ? "1" : "0",val);

   if (dbi_execute (pdi.dbi,sql))
    r = TRUE;
  }

 return (r);
}

/******************************************************************************/
/** update rec table (after match if sureness of new record is 1)
 * @param pid PSX_STR, PID
 * @param r_old PSX_REC *, pointer to old record
 * @param r_new PSX_REC *, pointer to new record
 * @param sur BOOL, sureness specification f new record
 * @param mrs BOOL, while matching retrieved sureness
 */
 
static BOOL pdi_update_process (PSX_STR pid,PSX_REC * r_old,PSX_REC * r_new,BOOL sur,BOOL mrs)
{
 IDX i;
 char sql [10000];
 PSX_STR sql_trn;
 PSX_STR t; 

 if (pid == NULL || r_old == NULL || r_new == NULL)
  return (FALSE);

 assert (pdi.sch != NULL);
 assert (pdi.atr != NULL);

 LOG (MOD,"UPDATE");

 strcpy (sql,"UPDATE rec SET ");
 
 LOG (MOD,"update: sureness");
 if (sur || mrs)                          // input is sure or matching retrieved is sure
  strcat (sql,"sur=1"); 
 else 
   strcat(sql, "sur=0");
 
 for (i = 0;i < pdi.atr -> n;i++)
 {
  PSX_STR atr;
  char * v_o,* v_n, *temp;           // old value, new value

  sprintf (atr,"%s",pdi.atr -> str [i]); // "f_"

  if (!rec_get (r_old,atr,&v_o))     // get old values
   continue;

  if (!rec_get (r_new,atr,&v_n))     // get new values
   continue;

  if (str_isnil (v_n))
   continue;   
  
  temp = v_n;
  dbi_val_cvt(temp,v_n);             // escape apostrophes
       
  if (strcmp (v_o,v_n) == 0)         // just update where difference exists
   continue;

  if (!sur)
  {
   if (!str_isnil (v_o))
    continue;
  }

  /* update */
  LOG (MOD,"update: %s",atr);

  sprintf (t,"f_%s='%s'",atr,v_n);
  strcat (sql,",");
  strcat (sql,t);
 } // end for

 // if (f)                     // sql not finished?
//  return (FALSE);
 sprintf (t," WHERE pid='%s'",pid);
 strcat (sql,t);

 if(!psx_cfg_get(PSX_CFG_SQL_BEGIN, sql_trn)) 
	 strcpy(sql_trn, "BEGIN TRANSACTION");

 if (!dbi_execute (pdi.dbi,sql_trn))
  return (FALSE);

 if (!dbi_execute (pdi.dbi,sql))
  return (FALSE);

 if(!psx_cfg_get(PSX_CFG_SQL_COMMIT, sql_trn))
	 strcpy(sql_trn, "COMMIT TRANSACTION");

 if (!dbi_execute (pdi.dbi,sql_trn))
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** invoke update of an existing record
 * @param pid PSX_STR, pseudonymisation identifier
 * @param rec_old PSX_REC *, pointer to old psx record
 * @param rec_new PSX_REC *, pointert to new psx record
 * @param sur BOOL, sureness
 * @param mrs BOOL, matching retrieved sureness 
 */

static BOOL pdi_update (PSX_STR pid,PSX_REC * rec_old,PSX_REC * rec_new,BOOL sur,BOOL mrs)
{
 if (pid == NULL || rec_old == NULL || rec_new == NULL)
  return (FALSE);

 LOG (MOD,"performing update [pid='%s']",pid);

 return (pdi_update_process (pid,rec_old,rec_new,sur,mrs));
}

/******************************************************************************/
/** get time stamp
 * @param tsp PSX_STR, time stamp string
 */
 
static BOOL pdi_tsp_get (PSX_STR tsp)
{
 struct tm * t;
 time_t x;

 if (tsp == NULL)
  return (FALSE);

 x = time (NULL);
 t = localtime (&x);

 if (t == NULL)
 {
  strcpy (tsp,"?");
  return (FALSE);
 }

 sprintf (tsp,"%04d-%02d-%02d-%02d-%02d-%02d",t -> tm_year + 1900,t -> tm_mon + 1,t -> tm_mday,t -> tm_hour,t -> tm_min,t -> tm_sec);

 return (TRUE);
}

/******************************************************************************/
/** request and generate new pid, store new record in database
 * @param req PDI_REQ *, pointer to request structure
 */

static BOOL pdi_req_g (PDI_REQ * req)
{
 if (req == NULL)
  return (FALSE);

 if (!pdi_pix_get (&req -> pix))  // get PID counter from database
  return (FALSE);

 if (req -> pix == 0)
  req -> pix = 1;

 if (!pdi_pid_make (req -> pid,req -> pix))
  return (FALSE);

 psx_put_v ("pdi: insert [%08X]",req -> pix);

 if (!pdi_insert (req -> pid,req -> rec,req -> sur))
  return (FALSE);

 if (!pdi_pix_set (req -> pix + 1))  // increment PID counter in database
  return (FALSE);

#if 0
 // return (smi_insert (req -> rec));
#endif

 return (TRUE);
}

/******************************************************************************/
/** request and retrieve existing pid
 * @param req PDI_REQ *, pointer to request structure
 */
 static BOOL pdi_req_r (PDI_REQ * req)
 {
 RUM rum;
 PSX_STR   pid;
 PSX_REC * rec; 
 
  if (req == NULL)
  return (FALSE);

 LOG (MOD,"update check");

// mti_rum_get (req -> mat,&rum);

 assert (pdi.sch != NULL);
 assert (req -> rsl < pdi.sch -> rsp -> n);

 rum = pdi.sch -> rsp -> itm [req -> rsl].rum;

 //PSX_STR   pid;
 //PSX_REC * rec; 
  //PSX_REC * rec_old;

  memcpy (pid,req -> pid,9);
  rec = req -> rec;

  if (!rec_create (&req -> rec_old))
   return (FALSE);
    
  if (!pdi_lookup (req -> rec_old,pid))    /* get old record from database for comparison */
    return (FALSE);
  
#if 0

  if (!smi_update (rec_old,rec))
  {
   // rec_delete (&rec_old);
   return (FALSE);
  }

#endif
  
 if (/*req -> sur && */ rum == RUM_UPDATE)  
 {   
    if (!pdi_update (pid,req -> rec_old,rec,req -> sur,req -> mrs))  
    {
      LOG (MOD,"update failed");
      return (FALSE);
    }
 }

 return (TRUE);
}

/******************************************************************************/
/** log request, i.e. write request in table 'req'
 * @param req PDI_REQ *, pointer to request structure
 * @param pid char *, pointer pid string
 */

static BOOL pdi_req_log (PDI_REQ * req,char * pid)
{
 IDX idx;
 PSX_STR atr,val,tsp,t,h,sql_atr,sql_val;
 DBI_STR sql;
 PSX_SCH * sch;
 CGI_ACC acc; 
 PSX_STR prm, res_sym, sql_atr_old, sql_val_old, atr_old, val_old, temp_atr;
 BOOL rum = FALSE; 
 int k;

 if (req == NULL)
  return (FALSE);

 if (!pdi_sch_get (&sch))
  return (FALSE);

 pdi_tsp_get (tsp);

 if (!pdi_cfg_set ("req",tsp))
  return (FALSE);

 if (!cgi_acc_get (&acc))
  return (FALSE); 

 if (!str_isnil (pid))
  sprintf (t,"'%s'",pid);
 else
  strcpy (t,"NULL");

 if (!dbi_idx_inc (pdi.dbi,"req","idx",&idx))  // get new idx for request
  return (FALSE);

 strcpy (sql_atr,"");
 strcpy (sql_val,""); 
 strcpy (sql_atr_old,"");
 strcpy (sql_val_old,"");

 if (req -> hsr)
  if (sch_sql_get_av (sch,req -> hsr,atr,val)) // get all request attributes and their values from history record
  {
   sprintf (sql_atr,",%s",atr);
   sprintf (sql_val,",%s",val);

    if (req -> rec_old)         // get all already existing attributes and values from old record
    {
       if (sch_sql_get_av(sch, req->rec_old, atr_old, val_old))
      {

        sprintf (sql_val_old, ",%s", val_old);

        k = 0;    
        while (str_tok (atr_old,&k,",",temp_atr))         /* copy text up to next comma into temp */
        {
            strcat (sql_atr_old, ",");
            strcat (sql_atr_old, temp_atr);
            strcat (sql_atr_old, "_o");
        }
      }
    }
  }
  
  strcpy(res_sym, sch->rsp->itm[req->rsl].sym);

  if (sch->rsp->itm[req->rsl].prm == PRM_GET) 
  {
	  strcpy(prm,"get");
  } else if (sch->rsp->itm[req->rsl].prm == PRM_GEN)
  {
	strcpy(prm,"gen");
  }

  if (sch->rsp->itm[req->rsl].rum == RUM_UPDATE) 
	rum = TRUE;

  rec_delete(&req->rec_old);    
  
  sprintf (sql,"INSERT INTO req (idx,pid,tsp,adr,prt,hst,sur,result,mode,upd %s %s) VALUES (%d,%s,'%s','%s','%s','%s',%d,'%s','%s',%d%s%s)",sql_atr,sql_atr_old,idx,t,acc.tsp,acc.adr,acc.prt,acc.hst,req->sur,res_sym,prm,rum,sql_val,sql_val_old);

 if (!dbi_execute (pdi.dbi,sql))
 {
   psx_put(sql);
  return (FALSE);
 }
      
 return (TRUE);
}


/******************************************************************************/
/******************************************************************************/
/******************************************************************************/



