/******************************************************************************/
/* psx-pgi.c (formerly PIDgen.c)                                  PID Generation Interface */
/******************************************************************************/

/*** PIDgen.c **************************************************/
/*                                                             */
/* Functions to support a pseudonymization service             */
/*-------------------------------------------------------------*/
/* Klaus Pommerening, IMSD, Johannes-Gutenberg-Universitaet,   */
/*   Mainz, 3. April 2001                                      */
/*-------------------------------------------------------------*/
/* Version 1.01, 17. April 2005                                */
/*   Corrects fatal error in PID2c that affected PIDcheck only */
/*      (and only in certain cases).                           */
/*   Comments a non-fatal error in NLmix that affects only     */
/*      the specification of the cryptographic transformation. */
/*      This modified transformation is (prpbably) as good as  */
/*      the intended one; because a correction would make all  */
/*      previously generated PIDs invalid, the error must      */
/*      remain as it is.                                       */
/*   Plus correction of some spelling errors in comments.      */
/***************************************************************/
/* Bugfix:                                                     */
/* Corrected checks in PIDcheck to determine type and location */
/* of error. Former check was imcomplete compared to the       */ 
/* mathematical formulation of the algorithm.                  */
/*                                                             */
/* Andreas Borg, 1.Oct.2007                                    */
/***************************************************************/

// changed include directives for integrating PIDgen.c as psx-pgi.c into psx tool
#include "psx.h"
//#include <stdlib.h>
//#include "PIDgen.h"

typedef unsigned long UINT32; /**< 32-bit-integers >= 0          */
typedef unsigned UINT6;       /**< 6-bit-integers >= 0           */
typedef unsigned UINT5;       /**< 5-bit-integers >= 0           */
typedef UINT5 PRECODEWORD[6]; /**< array of six 5-bit-integers   */
typedef UINT5 CODEWORD[8];    /**< array of eight 5-bit-integers
							     internal format for codewords */
static char sigma[] = "0123456789ACDEFGHJKLMNPQRTUVWXYZ";
                                  /**< alphabet for PIDs       */
  /* A codeword is transformed into a PID by replacing each of */
  /*   the 5-bit-integers by the corresponding character of    */
  /*   the alphabet.                                           */
static UINT32 NN   = 1073741824; /* 2^30 = module for calc     */
static UINT30 NN_1 = 1073741823; /* 2^30 - 1                   */

static unsigned long rndfact; /**< least bit for randomization  
                             rndfact = 2^(30-rndwidth)         */
static UINT30 rndlim;     /**< upper limit for random numbers   
                             rndlim = 2^(rndwidth)             */
static int RSET = 0;  /**< set to 1 after first call to rndsetup */

/** Initialize random generator.                                
 * Set global variables rndfact, rndlim.                     
 * ===> Must be called before first use of rndext <===           
 * rndwidth must be set before calling rndsetup.                 
 *
 * Error handling: Values > 12 of rndwith are treated as = 12  */
static void rndsetup()
{
  unsigned long tt;
  if (RSET) return;
  RSET = 1;
  tt = time(NULL);
  srand(tt);                          /* set seed              */

  if (rndwidth > 12) rndwidth = 12;
  rndfact = 1 << (30 - rndwidth);
  rndlim  = 1 << rndwidth;
}

/** Randomize a number                                          
 *                                                             
 * Input: integer x, 0 <= x < rndfact                          
 * Output: first rndwidth bits of x replaced with random bits  
 * Error handling: If x >= rndfact, overflowing bits are       
 *   dropped.                                                  
 * ---> used in encr                                          
 */
static UINT30 rndext(UINT30 x)
{
  UINT30 r;
  unsigned long rr;
  double r1;

  x = x & (rndfact - 1); 
  r1 = ((double) rand() / (RAND_MAX + 1.0));  /* 0 <= r1 < 1    */
  rr = rndlim * r1;
  r = rr*rndfact + x;
  return r;
}

/** Multiply x and y mod 2^30                                  
 * used in encr                                           
 */
static UINT30 mult30(UINT30 x, UINT30 y)
{
  UINT30 z;
  z = x*y;             /* multiply, dropping long int overflow */
  z = z & NN_1;                   /* reduce mod 2^30           */
  return z;
}

/** Rotate x cyclically by 6 bits to the right.                
 *   Bit 29 becomes bit 23, ..., bit 6 becomes bit 0,          
 *   bit 5 becomes bit 29, ..., bit 0 becomes bit 24.          
 * Error handling: If x is not in the required range,          
 *   overflowing bits are dropped.                             
 * ---> used in encr                                           
*/
static UINT30 rot30_6(UINT30 x)
{
  UINT30 y, z;
  y = x & 63;                   /* preserve last 6 bits        */
  y = y << 24;                  /* shift them to the left      */
  z = x >> 6;                   /* remaining bits to the right */
  z = z & 16777215;             /* clear overflowing bits      */
                                /* 16777215 = 2^24 - 1         */
  z = z | y;                    /* prefix preserved 6 bits     */
  return z;
}

/** Nonlinear transform of x.                                  
 *   Split the input x into five 6-bit-chunks [e|d|c|b|a],     
 *   replace a with the quadratic expression                   
 *   a + b*e + c*d mod 2^6.                                    
 *   Because of an error the transformation is                 
 *   a + b*d + c*d mod 2^6 instead.                            
 * This transformation is bijective on 30-bit-integers.        
 * Error handlicng: If x is not in the required range,         
 *   overflowing bits are dropped.                             
 * ---> used in encr
 */
static UINT30 NLmix(UINT30 x)
{
  UINT30 y;
  UINT6  a, b, c, d, e;
  a = x & 63;             /* extract last 6 bits               */
  y = x >> 6;             /* shift remaining bits to the right */
  b = y & 63;             /* extract last 6 bits               */
  y = y >> 6;             /* shift remaining bits to the right */
  c = y & 63;             /* extract last 6 bits               */
  y = y >> 6;             /* shift remaining bits to the right */
  d = y & 63;             /* extract last 6 bits               */
  e = y >> 6;             /* get remaining 6 bits              */
      /*** This should have been y = y >> 6. This error      ***/
      /*** must NEVER be corrected. It changes the intended  ***/
      /*** cryptographic transformation into another one     ***/
      /*** that is perfectly valid for its own, but probably ***/
      /*** somewhat weaker.                                  ***/
  e = y & 63;             /* clear overflowing bits            */
  
  a = (a + b*e + c*d) & 63;    /* quadratic expression mod 2^6 */
  y = x & 1073741760;          /* AND 2^30 - 2^6 =             */
                               /* preserve bits 6 to 29        */
  y = y | a;                   /* and append new 6 bits        */
  return y;
}

/** Permutation of the bits of x.                              
 *   Split the input x into six 5-bit-chunks,                  
 *   permute each chunk with the same fixed permutation        
 * This transformation is bijective on 30-bit-integers.        
 * Error handling: If x is not in the required range,          
 *   overflowing bits are dropped.                             
 * ---> used in encr                                           
 */
static UINT30 bitmix(UINT30 x)
{
  UINT30 p[5], xx[5], yy[5], y;
  int i;

  p[0] = 34636833;    /* 2^25 + 2^20 + 2^15 + 2^10 + 2^5 + 2^0 */
  for (i=1; i <= 4; i++) p[i] = p[i-1] << 1;
  for (i=0; i <= 4; i++) xx[i] = x & p[i];    /* every 5th bit */
  yy[0] = xx[3] >> 3;                         /* permute       */
  yy[1] = xx[0] << 1;
  yy[2] = xx[4] >> 2;
  yy[3] = xx[2] << 1;
  yy[4] = xx[1] << 3;
  y = yy[0] | yy[1] | yy[2] | yy[3] | yy[4];  /* and glue      */
  return y;
}

/** Encrypt x with keys k1, k2, k3, and k4 = k1+k2+k3 mod 2^30.
 *   The encryption consists of 4 rounds;                      
 *   each round consists of                                    
 *   - first rot30_6,                                          
 *   - then NLmix,                                             
 *   - finally multiply with ki mod 2^30.                      
 *   Between rounds 2 and 3 apply bitmix.                      
 * This transformation is bijective on 30-bit-integers,        
 *   if all ki are odd.                                        
 * Error handling: If ki is even, it's replaced with ki + 1.   
 *   If x is not in the required range, overflowing bits are   
 *   dropped.                                                  
 * ---> used in PIDgen                                         
 */
static UINT30 encr(UINT30 x)
{
  UINT30 w, y, z, k4;
  k1 = k1 | 1;                 /* k1 may be even - make it odd */
  k2 = k2 | 1;                 /* k2 may be even - make it odd */
  k3 = k3 | 1;                 /* k3 may be even - make it odd */
  k4 = (k1 + k2 + k3) & NN_1;  /* Key for round 4              */

  y = rot30_6(x);               /* round 1                     */
  w = NLmix(y);
  z = mult30(k1, w);
  y = rot30_6(z);               /* round 2                     */
  w = NLmix(y);
  z = mult30(k2, w);
  w = bitmix(z);                /* permutation                 */
  y = rot30_6(z);               /* round 3                     */
  w = NLmix(y);
  z = mult30(k3, w);
  y = rot30_6(z);               /* round 4                     */
  w = NLmix(y);
  z = mult30(k4, w);
  return z;
}

/** Split a 30-bit integer x into an array p of six 5-bit-integers.
 * Error handling: If x is not in the required range,
 *   overflowing bits are dropped.                   
 * ---> used in PIDgen                               
 */
static void u2pcw(UINT30 x, PRECODEWORD p)
{
  UINT30 y;
  p[5] = x & 31;           /* extract last 5 bits              */
  y = x >> 5;              /* shift remaining bit to the right */
  p[4] = y & 31;           /* extraxt last 5 bits              */
  y = y >> 5;              /* shift remaining bit to the right */
  p[3] = y & 31;           /* extraxt last 5 bits              */
  y = y >> 5;              /* shift remaining bit to the right */
  p[2] = y & 31;           /* extraxt last 5 bits              */
  y = y >> 5;              /* shift remaining bit to the right */
  p[1] = y & 31;           /* extraxt last 5 bits              */
  y = y >> 5;              /* shift remaining bit to the right */
  p[0] = y & 31;           /* extraxt last 5 bits              */
}

/** Arithmetic in the Galois field F_32.
 *   t is a primitive element with t^5 = t^2 + 1.              
 * Multiply the 5 bit input x = (x4, x3, x2, x1, x0) with t^e 
 *   where 0 <= e <= 3;                                        
 *   the algorithm is described in the documentation.          
 * Error handling: If x has more then 5 bits, overflowing bits 
 *   are dropped.                                              
 *   If e is < 0 or > 3, it's treated as 0, i. e. the function 
 *   returns x.                                                
 * ---> used in multf32                                        
 */
static UINT5 mult0f32(UINT5 x, unsigned e)
{
  UINT5 u, v, w, s;
  x = x & 31;                  /* drop overflowing bits        */
  if (e == 0) return x;        /* catch trivial case and       */
  if (e > 3)  return x;        /* unwanted values              */

  u = x >> (5-e);
  v = u << 2;
  w = (x << e) & 31;
  s = (u ^ v) ^ w;
  return s;
}


/** Arithmetic in the Galois field F_32.
 *   t is a primitive element with t^5 = t^2 + 1.              
 * Multiply the 5 bit input x = (x4, x3, x2, x1, x0) with t^e  
 *   where e is an unsigned integer.                           
 * Error handling: If x has more then 5 bits, overflowing bits 
 *   are dropped.                                              
 * ---> used in wsum1, wsum2, and PIDcheck.                    
 */
static UINT5 multf32(UINT5 x, unsigned e)
{
  x = x & 31;                  /* drop overflowing bits        */
  while (e >= 4) {
    x = mult0f32(mult0f32(x,2),2); /* multiply by t^4          */
    e = e-4;
    }
  x = mult0f32(x,e);
  return x;
}

/** Output weighted sum.
 * Output the weighted sum                                     
 *   p[0] + t p[1] + t^2 p[2] + t^3 p[3] + t^4 p[4] + t^5 p[5] 
 *   in F_32.                                                  
 * Not used at all.                                       
 */
static UINT5 wsum(PRECODEWORD p)
{
  UINT5 s;
  unsigned i;
  s = p[0];
  for (i=1; i<=5; i++) s = s ^ multf32(p[i],i);
  return s;
}


/** Output weighted sum.
 * Output the weighted sum                                     
 *  t p[0] + t^2 p[1] + t^3 p[2] + t^4 p[3] + t^5 p[4]         
 *  + t^6 p[5] 
 *   in F_32.                                                  
 * ---> used in encode and PIDcheck.                           
 */
static UINT5 wsum1(PRECODEWORD p)
{
  UINT5 s;
  unsigned i;
  s = 0;
  for (i=0; i<=5; i++) s = s ^ multf32(p[i],i+1);
  return s;
}

/** Output weighted sum.
 * Output the weighted sum                                     
 *  t^2 p[0] + t^4 p[1] + t^6 p[2] + t^8 p[3] + t^10 p[4]      
 *                                                 + t^12 p[5] 
 *   in F_32.                                                  
 * ---> used in encode and PIDcheck.                           
 */
static UINT5 wsum2(PRECODEWORD p)
{
  UINT5 s;
  unsigned i;
  s = 0;
  for (i=0; i<=5; i++) s = s ^ multf32(p[i],2*i+2);
  return s;
}

/** Transform to codeword.
 * Transform an array p of six 5-bit-integers into a codeword, 
 *   consisting of eight 5-bit-integers.                       
 * ---> used in PIDgen                                         
 */
static void encode(PRECODEWORD p, CODEWORD c)
{
  int i;
  for (i=0; i<=5; i++) c[i] = p[i]; /* preserve input elements */
  c[6] = wsum1(p);                     /* weighted sum in F_32 */
  c[7] = wsum2(p);                     /* weighted sum in F_32 */
}

/** Transform a string p into a codeword c.         
 *   If p has length != 8 or contains a character not in the   
 *   alphabet sigma, the function returns 0 and the output     
 *   variable c is meaningless.                                
 *     Note: Lowercase letters are converted to uppercase.     
 *   Otherwise c is the codeword corresponding to p and the    
 *   function returns 1.                                       
 * ---> used in PIDcheck                                       
 */
static int PID2c(char *p, CODEWORD c)
{
  unsigned i;
  if (strlen(p) != 8) return 0;
  for (i=0; i<8; i++) c[i] = 0;
  for (i=0; i<8; i++) {     /* convert chars to 5-bit-integers */
    switch (p[i]) {
      case '0': case '1': case '2': case '3': case '4':
      case '5': case '6': case '7': case '8': case '9':
        c[i] = p[i] - '0';
	break;
      case 'A': case 'a': c[i] = 10;
	break;
      case 'C': case 'D': case 'E': case 'F': case 'G': case 'H':
        c[i] = p[i] - 'C' + 11;
	break;
      case 'c': case 'd': case 'e': case 'f': case 'g': case 'h':
        c[i] = p[i] - 'c' + 11;
	break;
      case 'J': case 'K': case 'L': case 'M': case 'N':
        c[i] = p[i] - 'J' + 17;
	break;
      case 'j': case 'k': case 'l': case 'm': case 'n':
        c[i] = p[i] - 'j' + 17;
	break;
      case 'P': case 'Q': case 'R':
        c[i] = p[i] - 'P' + 22;
	break;
      case 'p': case 'q': case 'r':
        c[i] = p[i] - 'p' + 22;
	break;
      case 'T': case 'U': case 'V': case 'W': case 'X':
      case 'Y': case 'Z':
        c[i] = p[i] - 'T' + 25;
	break;  /*** missing break inserted 17. April 2005 KP ***/
      case 't': case 'u': case 'v': case 'w': case 'x':
      case 'y': case 'z':
        c[i] = p[i] - 't' + 25;
	break;
      default:                       /* invalid character found */
        return 0;
	break;
      }
    }
  return 1;
}

/** Generate PID.
 * Description in psx-pgi.h
 */
char *PIDgen(UINT30 x)
{
  UINT30 y, z;
  unsigned j;               /* loop counter                    */
  PRECODEWORD pcw;          /* intermediate result             */
  CODEWORD cw;              /* intermediate result             */
  static char p[] = "00000000";  /* output variable            */

  if (x == 0) return p;     /* x outside required range        */
  if (x >= NN) return p;    /* x outside required range        */

  rndsetup();
  y = rndext(x);            /* randomize (or not)              */
  z = encr(y);              /* encrypt                         */
  u2pcw(z, pcw);            /* split                           */
  encode(pcw, cw);          /* encode                          */
  
  for (j=0; j<=7; j++) p[j] = sigma[cw[j]]; /* rewrite as PID  */
  return p;
}

/** Check PID.
 * Description in psx-pgi.h
 */
int PIDcheck(char *p, char *a)
{
  CODEWORD c;
  unsigned i, sum6, sum7, test6, test7;
  *a = 'B';
  if (!(PID2c(p, c))) return 0;

  sum6 = wsum1(c);                /* checksum 1                */
  sum7 = wsum2(c);                /* checksum 2                */
  test6 = sum6 ^ c[6];
  test7 = sum7 ^ c[7];
  if (!test6) {                   /* checksum 1 correct        */
    if (!test7) return -1;        /* checksum 2 correct        */
    else {
      *a = sigma[sum7];           /* correct checksum 2        */
      return 8;
      }
    }
  else {
    if (!test7) {                  /* checksum 2 correct        */
      *a = sigma[sum6];            /* correct checksum 1        */
      return 7;
      }
    else {
      if (test7 == multf32(test6,1)) {          /* c[0] wrong   */
        *a = sigma[c[0] ^ multf32(test6,30)];   /* correct char */
	return 1;
        }
      else if (test7 == multf32(test6,2)) {     /* c[1] wrong   */
        *a = sigma[c[1] ^ multf32(test6,29)];   /* correct char */
	return 2;
        }
      else if (test7 == multf32(test6,3)) {     /* c[2] wrong   */
        *a = sigma[c[2] ^ multf32(test6,28)];   /* correct char */
	return 3;
        }
      else if (test7 == multf32(test6,4)) {     /* c[3] wrong   */
        *a = sigma[c[3] ^ multf32(test6,27)];   /* correct char */
	return 4;
        }
      else if (test7 == multf32(test6,5)) {     /* c[4] wrong   */
        *a = sigma[c[4] ^ multf32(test6,26)];   /* correct char */
	return 5;
        }
      else if (test7 == multf32(test6,6)) {     /* c[5] wrong   */
        *a = sigma[c[5] ^ multf32(test6,25)];   /* correct char */
	return 6;
	}
      else if (test7 == test6 &&               /* corrected 1.Oct.2007 */ 
               test7 == (c[6] ^ c[7])) {       /* c[6], c[7] interchanged */
	return 15;
        }
      else if (test7 == multf32(test6,16) &&   /* corrected 1.Oct.2007 */
        (multf32(c[5],6) ^ multf32(c[6],6) ^ c[5] ^ c[6]) == test6) { /* c[5],c[6] int'd */
	return 14;
	}
      else if (test7 == multf32(test6,19) &&   /* corrected 1.Oct.2007 */
        multf32(c[0] ^ c[1],19) == test6) { /* c[0],c[1] int'd */
	return 9;
	}
      else if (test7 == multf32(test6,20) &&  /* corrected 1.Oct.2007 */
        multf32(c[1] ^ c[2],20) == test6) { /* c[1],c[2] int'd */
	return 10;
	}
      else if (test7 == multf32(test6,21) &&  /* corrected 1.Oct.2007 */
        multf32(c[2] ^ c[3],21) == test6) { /* c[2],c[3] int'd */
	return 11;
	}
      else if (test7 == multf32(test6,22) &&  /* corrected 1.Oct.2007 */
        multf32(c[3] ^ c[4],22) == test6) { /* c[3],c[4] int'd */
	return 12;
	}
      else if (test7 == multf32(test6,23) &&  /* corrected 1.Oct.2007 */
        multf32(c[4] ^ c[5],23) == test6) { /* c[4],c[5] int'd */
	return 13;
	}
      else return 0;             /* at least 2 characters wrong */
                                 /* and no simple interchange   */
      }
    }
}

