/******************************************************************************/
/* psx-pho-c.c                             Phonetics Interface - Cologne      */
/******************************************************************************/
/** @file psx-pho-c.c Phonetics Interface: Cologne - Inline Source Code File
 * Procedures for the generation of phonetic codes according to
 * Cologne Phonetic
 *
 * Author: Irene Schmidtmann, Krebsregister Rheinland-Pfalz 
 *         using functions programmed by Michael Pollecker (1992)
 * Date :  29.08.1993 14:55, documentation translated in November 2004
 */ 
/* Transformation in phon_code()[line131]                                     */
/******************************************************************************/
/*                                                                            */
/******************************************************************************/

#include <stdio.h> 
#include <stdlib.h> 
#include <string.h> 
 
#include "psx.h"
#include "psx-pho-c.h"

/* #define _DEBUG */ 
 
/* noch zu verbessern: bei den Formen, die nur einen Teil des Strings 
   phonetisch codieren, sind Umlaute und ihre Aufloesungen nicht dasselbe ! 
   der Umlaut bzw.  sind nur ein Zeichen, die Aufloesung aber 2! 
*/ 

/******************************************************************************/
/*                                                                            */
/******************************************************************************/

/*------------------------------------------------------------------*/ 
/*                                                                  */ 
/* Definition of Transformationtable for Cologne Phonetic           */ 
/*                                                                  */ 
/*------------------------------------------------------------------*/ 
 
/** doppel_src contains combinations of letters, which are to be    
 *  tranformed into one single letter. In the beginnig of a word     
 *  the first letters of each ligature are used, if flag=SOUND is set.
 */
 
static const char *doppel_src[] = 
       {"AE\0","AI\0","AU\0","AY\0","CH\0","CK\0","CZ\0", 
              "DT\0","EI\0","EU\0","EY\0","IE\0","OE\0","OI\0", 
              "OU\0","PH\0","SC\0","SZ\0","TZ\0","UE\0","UO\0" }; 
static const char *doppel_dest = "EEEESKSDEEEEEEUFSSSEU\0"; 
 
/** einzel_src contains single letters to  be transformed into
 *  the correspondend letters in einzel_dest
 */ 

static const char *einzel_src  = "ABCDEFGIJKLMNOPQRSTUVWXYZ\0";
static const char *einzel_dest = "EBKDEFKEEKLMMUBKRSDUFFSESEEES\0"; 
 
/******************************************************************************/
/** transform small letters of a string into capital ones, Umlauts, too
 * @param str pointer to source and destination string
 */
 
char *strup_g (char *str)
{ 
 char *p1; 
 int i; 
 
 p1 = str; 
 
 for (i = 0; i < strlen(str); i++, p1++) 
 { 
  if (*p1 >= 'a' && *p1 <= 'z') 
  { 
   *p1 = *p1 - ('a' - 'A'); 
  } 
  else if (*p1 ==''){
        *p1 = ''; 
      } 
      else if (*p1 ==''){ 
        *p1 = ''; 
      } 
      else if (*p1 ==''){ 
        *p1 = ''; 
      } 
      else 
      { 
        ;     
      } 
   } 
   return (str); 
}
 
/******************************************************************************/
/** change small into capital letters and remove special characters, blanks,
 *  ligatures, etc. from str (german Umlauts and  are not being handled as
 *  special characters!).
 */
char *cleanstr_g (char *str) 
{ 
 char *p1, *p2;
 int i, len;
 
 len = strlen(str);
 
 strup_g(str);               /* change str to capital letters    */

 for (i = 0,p1 = p2 = str;i < len;i++)
 {
  if (*p1 >= 'A' && *p1 <= 'Z' || *p1 == '' || *p1 == '' || *p1 == '' || *p1 == '' )	/* Buchstabe ? */
  {
   if (*p1 != *(p2-1))	/* ligature ? */
   {
    *p2 = *p1;
    p2++;
   }
  }
  p1++;
 }

 *p2 ='\0';
 
 return (str);
} 
 
/******************************************************************************/
/**
 * @param str string pointer to source string
 * @param len length of str, '0' for original length
 * @param flag specifies, if original Cologne Phonetic coding or modified
 *             Cologne Phonetic coding is being used (values can be either
 *             KOELN for the original or SOUND for the modified algorithm
 */
char *phon_code (char  *str, int len, int flag)
{ 
  char *p1, *p2; 
  int i, gefunden = 0;; 
 
/* check parameters */ 
  if (str == NULL || (flag != KOELN && flag != SOUND)){ 
    return (NULL); 
  } 
 
  if (!len){ 
    len = strlen(str);   /* get length of string           */ 
  } 
  else{ 
    str[len] = '\0';     /* close string with binary null  */ 
  } 
 
  /* remove ligatures, special characters etc.             */ 
  cleanstr_g(str); 
 
  /* get new length, maybe it has been changed by cleanstr */ 
  len = strlen(str); 
 
#ifdef _DEBUG 
  //printf("String <%s> wird konvertiert; ", str);
#endif
 
  p1 = p2 = str; 
  if (flag == SOUND){ 
    /* keep first character where necessary, if a ligature is present 
       only the first character will be taken into account, the second
       will be overread                                               */ 
    for (i=0, gefunden=0; i < 21 && !gefunden; i++){ 
      if (!(strncmp(doppel_src[i], p1, 2))){ 
        *(p2++) = *(doppel_src[i]); 
        gefunden++; 
        p1++; 
      } 
    } 
    if (!gefunden){ 
      *(p2++) = *p1; 
    } 
    p1++; 
  } 
 
 
  /* convert string */ 
  while (p1 < str+len){ 
    if (*p1 != 'H'){    /* 'H' is not taken into account */ 
      gefunden = 0; 
      switch (*p1) 
      { 
        case 'A':  case 'C':  case 'D':  case 'E':  case 'I': 
        case 'O':  case 'P':  case 'S':  case 'T':  case 'U': 
                                        /* convert Umlauts and character
                                           combinations (there are 21)  */ 
    
        i = 0; 
        while (i < 21 && !gefunden){ 
          if (!(strncmp (doppel_src[i], p1, 2))){ 
            *p2 = doppel_dest[i]; /* convert character */ 
            gefunden++; 
            p1++; /* jump over character */ 
          } 
          i++; 
        } 
        if (gefunden) 
          break; 
 
        /* convert single character */ 
        default : for (i = 0; i < strlen(einzel_src) && !gefunden; i++){ 
          if (*p1 == einzel_src[i]){ 
            *p2 = einzel_dest[i]; /* convert character */ 
            gefunden++; 
          } 
        } 
      } 
      if (gefunden){ 
        p2++; 
      } 
      else 
        printf("\n\nFEHLER: Buchstabe %c nicht gefunden !\n\n",*p1); 
    } 
    p1++; 
  } 
  *p2 = '\0'; /* close string with binary null */ 
 
#ifdef _DEBUG 
  //printf("Ergebniss : <%s> \n", str);
#endif

  return (str);
}

/******************************************************************************/
/** invoke main function of phonetic code generation
 * @param txt string pointer to source and result string
 */
 
BOOL pho_col (char * txt)
{
 if (phon_code (txt,0,KOELN) == NULL)
  return (FALSE);
 return (TRUE);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


