/******************************************************************************/
/* psx-pmi.c                                         PID Management Interface */
/******************************************************************************/
/** @file psx-pmi.c PID Management Interface - Source Code File
 * Functions providing an PID Management Interface. The Interface consists of
 * a run function for PID request invocation, an initialisation function, an exit
 * function and several functions for PMI structure handling.
 */
 
#include "psx.h"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

/******************************************************************************/
/** Retrieve record.
 * Reads a single record, either from a file with rec_read_s(), or
 * from stdin with rec_input().
 * @param rec pointer to record
 * @param fmt pointer to format specification
 * @param dat pointer to data file
 */
static BOOL pmi_rec_retrieve (PSX_REC * rec,PSX_FMT * fmt,FILE * dat)
{
 if (rec == NULL || fmt == NULL)
  return (FALSE);

 if (dat != NULL)
 {
  psx_put_v ("loading record from file"); // %s",dat);

  if (!rec_read_s (rec,fmt,dat))          // read single record
  {
   printf ("> record load failure\n");
   return (FALSE);
  }
 }
 else									  // read from stdin
 {
  if (!rec_def_fmt (rec,fmt))
   return (FALSE);

  psx_put ("data acquisition\n");

  if (!rec_input (rec,2))
   return (FALSE);

  printf ("\n");
 }

 return (TRUE);
}

/******************************************************************************/
/** retrieve sureness (1 = sure, 0 = not sure)
 * @param sur integer pointer, contains sureness after invocation
 * @return TRUE if valid sureness value was retrieved
 * @return FALSE if sureness could not be retrieved
 */

static BOOL pmi_sur_retrieve (int * sur)
{
 PSX_STR t;

 if (sur == NULL)
  return (FALSE);

 if (!cio_get ("  sure [0,1]?",t))
  return (FALSE);

 str_trim (t);


 if (strcmp (t,"0") == 0)
  *sur = 0;
 else
  if (strcmp (t,"1") == 0)
   *sur = 1;
  else
  {
   psx_put ("invalid sureness specification");
   return (FALSE);
  }

 return (TRUE);
}

/******************************************************************************/
/** print record
 * @param rec pointer to record
 */

static BOOL pmi_rec_print (PSX_REC * rec)
{
 IDX i;

 if (rec == NULL)
  return (FALSE);

 for (i = 0;i < rec -> n;i++)
 {
  PSX_STR t;

  rec_getLblStr (rec,i,t);                        // get label name

  printf ("  %s: '%s'\n",t,rec -> itm [i].val);   // print both, name and value
 }

 printf ("\n");

 return (TRUE);
}

/******************************************************************************/
/** Execute request.
 * Calls psi_req() for execution.
 * @param req pointer to request
 */

static BOOL pmi_req_execute (PSI_REQ * req)
{
 BOOL r;

 if (req == NULL)
  return (FALSE);

 // printf ("pmir: sure=%d\n",req -> sur);



 r = psi_req (req);

 return (r);
}

/******************************************************************************/
/** Run request, single record.
 * <ul>
 *   <li> Get record data with pmi_rec_retrieve()
 *   <li> Convert to ISO format with rec_cvt_din(), if necessary.
 *   <li> Get PID with pmi_req_execute()
 * </ul>
 * @param ctx pointer to pmi context
 */

static BOOL pmi_run_req_s (PMI_CTX * ctx)
{
 PSI_REQ req;
 PSX_STR msg;
 PSX_STR t;
 BOOL r;

 
if (!pmi_rec_retrieve (ctx -> idr,ctx -> sch -> fsp,ctx -> ids))	// get record 
  return (FALSE);
 
 if ( ctx -> ics == ICS_DIN)  
  rec_cvt_din (ctx -> idr);  // convert record: DIN 66003 to ISO 8859-1

 //printf("ctx->sur : %i\n", ctx -> sur); 
 if (ctx -> sur != 0 && ctx -> sur != 1)
  if (!pmi_sur_retrieve (&ctx -> sur))               // get sureness
   return (FALSE);

 memset (&req,0,sizeof (req));

 req.rec = ctx -> idr;                               // set record
 req.sur = (ctx -> sur == 1);                        // set sureness
 req.ctn = ctx ->  cnr;                              // set control number
 //req.ics = ctx -> ics;							// added 25.10.2004

 req.frc = ctx -> frc;                               // set force generation

 r =  pmi_req_execute (&req);                        // execute request

 ctx -> rsl = req.rsl;

 if (!r)
 {
  psx_put ("pid generation failed\n");
  return (FALSE);
 }

 if (psx.cfg.verbose)
 {
  psx_put ("result\n");
  pmi_rec_print (req.ctn);
 }

 if (!pid_get_str (req.pid,t))
  return (FALSE);

 psx_put ("PID: '%s'",t);

 assert (ctx -> rsl < ctx -> sch -> rsp -> n);
 strcpy (msg,ctx -> sch -> rsp -> itm [ctx -> rsl].msg);

 psx_put (msg);

 if (!r)
  psx_put ("pid generation failed\n");

 return (r);
}

/******************************************************************************/
/* modified 03.01.2005: pid retrieval mode should not be known to user */

/** encode pid retrieval mode
 * @param prm pid retrieval mode
 * @param sym contains name of retrieval mode after invocation
 */

static BOOL pmi_prm_enc (PRM prm,PSX_STR sym)
{
 switch (prm)
 {
  case PRM_GET: strcpy (sym,"PID"); break;    // get PID
  case PRM_GEN: strcpy (sym,"PID"); break;    // generate PID
  default:      strcpy (sym,"???"); break;    // no valid prm
 }
 return(TRUE);	// added 23.8.04
}

#define NOT_STEP 10000  //!< number of records after which user should get notification
/*****************************************************************************/
/** run PID request, multiple records, print out result
 * @param ctx pointer to pmi context structure
 */

static BOOL pmi_run_req_m (PMI_CTX * ctx)
{
 NUM n,n_f,n_m;
 PSX_STR t, rst;
 
 if (ctx == NULL)
  return (FALSE);
 // input data record, input format specification, input data stream
 assert (ctx -> idr != NULL && ctx -> ifs != NULL && ctx -> ids != NULL);

 n   = 0;  // number of requests
 n_f = 0;  // number of errors (failures)
 n_m = 0;  // number of matches

 printf ("\n");
 printf ("%05d\r",n);    // "00000" = number of requests
 fflush (stdout);

 tmi_tmg_start (&ctx -> tmg);   // time management, get start time

 if (ctx -> pos != 0)
 {
  IDX i;

  /* skip records if specified by -p argument */
  psx_put ("skipping %d records",ctx -> pos - 1);

  for (i = 1;i < ctx -> pos;i++)
  {
   if (!rec_read (ctx -> idr,ctx -> ifs,ctx -> ids, NULL,""))		// skip records /*:BG:added 0 for delimiter */
                                                                   /* changed to "", otherwise seg fault */
   {
    psx_put ("error while skipping");
    return (FALSE);
   }

  }
 }

 while (rec_read (ctx -> idr,ctx -> ifs,ctx -> ids, rst,ctx->del))  // read next record /*:BG:added ctx->del*/ 
  {
  PSI_REQ req;
  BOOL r;

  printf ("%05d\r",++n);        // print number of request
  fflush (stdout);

  memset (&req,0,sizeof (req));

  // read rest of input line to get sureness specification
  if ( (strchr (rst, '-')) != NULL)
   req.sur = 0;
  else if ( (strchr (rst, '+')) != NULL) 
   req.sur = 1;  
  else
   req.sur = (ctx -> sur == 1);
  
  // set request  
  req.rec = ctx -> idr;
  req.ctn = ctx -> cnr;
  req.frc = ctx -> frc;
  
  if ( TEST (ctx -> ics,ICS_DIN))   
    rec_cvt_din (req.rec);          // convert record: DIN 66003 to ISO 8859-1
  
  r = pmi_req_execute (&req);         // execute request

  if (ctx -> ots)                     // output trace stream
  {
   PSX_STR t,m;

   pmi_prm_enc (req.prm,m);           // encode pid retrieval mode to string

   sprintf (t,"%04d",n);              // n = record number in dataset
   strcat (t,": ");
   strcat (t,m);                      // m = pid retrieval mode
   strcat (t,": ");

   if (r)     
    strcat (t,req.pid);
   else
    strcat (t,"?");
   fprintf (ctx -> ots,"%s\n",t);     // print output trace file
  }

  if (!r)
   n_f ++;                            // count false

  if (req.prm == PRM_GET)             // count PRM_GET
   n_m ++;

  // psx_put ("request: %s",r ? "ok" : "failed");


  /* notifiy user after NOT_STEP new entries */
  if (n % NOT_STEP == 0)  // n modulo 10000
  {
   char t [1000];

   sprintf (t,"There are now %d records in the database.",n);
   //pid_notify (t);
  }
 } // end while

 printf ("%05d\n",n);
 printf ("\n");

 tmi_tmg_stop (&ctx -> tmg);								// stop time
 tmi_tmg_fmt (&ctx -> tmg,t);								// store time in t

 psx_put ("rec: %d, errors: %d, matches: %d, time: %s sec",n,n_f,n_m,t);

 psx_put ("ready");

 return (TRUE);
}

/******************************************************************************/

/******************************************************************************/
/** Run PID request.
 * Calls pmi_run_req_s() for single PID retrieval, pmi_run_req_m() for
 * batch mode.
 * @param ctx pointer to pmi context
 */
static BOOL pmi_run_req (PMI_CTX * ctx)
{
 BOOL r;

 if (ctx == NULL)
  return (FALSE);

 if (!ctx -> bat)
  r = pmi_run_req_s (ctx);			// single record mode
 else
  r = pmi_run_req_m (ctx);			// batch mode, multiple records

 return (r);
}


/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** create a new pmi control structure
 * @param ctx pointer to pointer to pmi context structure
 */

BOOL pmi_create (PMI_CTX ** ctx)
{
 if (ctx == NULL)
  return (FALSE);

 if ((*ctx = (PMI_CTX *) malloc (sizeof (**ctx))) == NULL)
  return (FALSE);

 memset (*ctx,0,sizeof (*ctx));

 return (TRUE);
}

/******************************************************************************/
/** delete pmi structure
 * @param ctx pointer to pointer to ctx context structure
 */

BOOL pmi_delete (PMI_CTX ** ctx)
{
 if (ctx == NULL)
  return (FALSE);

 if (*ctx == NULL)
  return (FALSE);

 free (*ctx);

 return (TRUE);
}

/******************************************************************************/
/** initialize pid management interface:
 * allocate memory for records and specifications, open input and output files

 * load format specification
 * @param ctx pointer to pmi context structure (result)
 * @param cfg pointer to pmi configuration 
 */
 
BOOL pmi_init (PMI_CTX * ctx,PMI_CFG * cfg)
{
 if (ctx == NULL || cfg == NULL)
  return (FALSE);

 memset (ctx,0,sizeof (*ctx));
 strcpy (ctx -> msg,"");

 ctx -> cmd = cfg -> cmd;

 /*input format specification */

 if (1)
 {
  if (!fsp_create (&ctx -> ifs))  // create and allocate memory
   return (FALSE);
 }


 /* input data record */

 if (1)
 {
  if (!rec_create (&ctx -> idr))
   return (FALSE);
 }

 /* output data record */

 if (1)
 {
  if (!rec_create (&ctx -> odr))
   return (FALSE);
 }

 /* control number record */

 if (1)
 {
  if (!rec_create (&ctx -> cnr))
   return (FALSE);
 }

 /* input data specification */

 if (1)
 {

  if (!str_isnil (cfg -> idf))
  {
   if ((ctx -> ids = fopen (cfg -> idf,"rt")) == NULL)
   {
    fprintf (stderr,"file open error: '%s'!\n",cfg -> idf);
    return (FALSE);
   }
  }
  else
   psx_put ("no data file");

 }

 if (1)
 {
  if (!str_isnil (cfg -> otf))
  {
   if ((ctx -> ots = fopen (cfg -> otf,"wt")) == NULL)
   {
    fprintf (stderr,"file open error: '%s'!\n",cfg -> otf);
    return (FALSE);
   }
  }
  else
   ctx -> ots = NULL;
 }

 ctx -> bat = cfg -> bat;
 ctx -> pos = cfg -> pos;
 ctx -> sur = cfg -> sur;
 ctx -> frc = cfg -> frc;
 ctx -> ics = cfg -> ics;
 strcpy(ctx -> del, cfg->del);   /*:BG:added 255.02.05*/

 /* open database */

 if (1)
 {
  if (!pdi_open ())
  {
   psx_put ("database open error");

   return (FALSE);
  }
  
  /* read database schema */
  if (!pdi_sch_get (&ctx -> sch))
   return (FALSE);
 }

 /* get format specification */

 if (1)
 {
  if (!str_isnil (cfg -> iff))   /* check if format file is given */
  {
   psx_put_v ("using format '%s'",cfg -> iff);

   if (!fsp_load (ctx -> ifs,cfg -> iff))    /* load format from file */
   {
    psx_put ("> invalid format specification\n");
    return (FALSE);
   }
  }
  else  /* no format file given: use default format from schema */
  {
   psx_put_v ("using default format");

   if (!fsp_copy (ctx -> ifs,ctx -> sch -> fsp))
    return (FALSE);
  }
 }

 return (TRUE);
}


/******************************************************************************/
/** clear context
 * @param ctx pointer to pmi context
 */

BOOL pmi_exit (PMI_CTX * ctx)
{
 BOOL r;

 if (ctx == NULL)
  return (FALSE);

 if (ctx == NULL)
  return (FALSE);


 strcpy (ctx -> msg,"");

 r = TRUE;

 if (!pdi_close ())

  r = FALSE;

 if (ctx -> ifs) if (!fsp_delete (&ctx -> ifs)) r = FALSE;
 if (ctx -> idr) if (!rec_delete (&ctx -> idr)) r = FALSE;
 if (ctx -> odr) if (!rec_delete (&ctx -> odr)) r = FALSE;
 if (ctx -> cnr) if (!rec_delete (&ctx -> cnr)) r = FALSE;
 if (ctx -> ids) if (fclose (ctx -> ids) == EOF) r = FALSE;
 if (ctx -> ots) if (fclose (ctx -> ots) == EOF) r = FALSE;

 memset (ctx,0,sizeof (*ctx));

 return (r);
}

/******************************************************************************/
/** Run pid retrieval request.
 * Delegates request to pmi_run_req().
 * @param ctx pointer to pmi context
 */

 
BOOL pmi_run (PMI_CTX * ctx)
{
 BOOL r;

 if (ctx == NULL)
  return (FALSE);

 switch (ctx -> cmd)
 {
  case PMI_CMD_REQ:           
   r = pmi_run_req (ctx);
   break;

  default:
   return (FALSE);
 }

 return (r);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/



