/******************************************************************************/
/* psx-psi-chk.inl                         Pseudonymization Service Interface */
/******************************************************************************/
/** @file psx-psi-chk.inl Pseudonymization Service Interface: Check - Inline Source Code File
 * Functions implementing datatype specific checks on input record:
 * - clean string (uppercase, convert accented letters, DIN to ISO 8859-1,
 *   terminate string with binary null)
 * - checking input data record for correctness
 * - checking fields according to datatype (includes other checks)
 */

/***************************************************************************************/
/** get option ValidateHIC from configuration file
 * @return TRUE if Health Insurance Codes should be checked for validity
 * @return FALSE if Health Insurcance Codes should not be checked
 */
 
static BOOL psi_hic_check ()
{
 PSX_STR t;

 if (!psx_cfg_get (PSX_CFG_HIC,t))
  return (FALSE);

 if (strcmp (t,"1") != 0)
  return (FALSE);

 return (TRUE);
}
/***************************************************************************************/
/** uppercase string, remove or convert non-alphabetical (German alphabet) characters,
 * terminate string
 * @param str string pointer
 */
 
static char * psi_cleanstr (char * str)
{
 char *p1, *p2;
 int i, len;

 len = strlen(str);

 str_cvt_accent(str);
 
 strup_g(str);          /* convert string into capital letters */ 

 for (i = 0,p1 = p2 = str;i < len;i++)
 {                                                                                        // delimiting characters stay in string
  if (*p1 >= 'A' && *p1 <= 'Z' || *p1 == '' || *p1 == '' || *p1 == '' || *p1 == '' || txt_isNameDel (*p1))	/* valid character ? */
  {  
    *p2 = *p1;
    p2++;
  }
  p1++;
 }

 *p2 ='\0';

 return (str);
}

/******************************************************************************/
/** get hio code for Health Insurance Organization
 * @param ctx context structure
 * @param hio health insurance organisation name
 */
  
static BOOL psi_req_chk_hio_get (CTX_PSI_REQ * ctx,HIO * hio)
{
 IDX i;
 BOOL t = FALSE;

 if (ctx == NULL || hio == NULL)
  return (FALSE);
 /* iteration of all input fields defined in input format specification       */
 /* gets hio name if data is of type Health Insurance Organization (DTP_HIO)  */
 for (i = 0;i < ctx -> sch -> fsp -> n;i++)
  if (ctx -> sch -> fsp -> itm [i].dtp == DTP_HIO)
   if (!t)
   {
    if (!hfi_hio_dec (hio,ctx -> rec -> itm [i].val))
     return (FALSE);

    t = TRUE;
   }
   else
    return (FALSE); /* only if multiple fields of datatype DTP_HIO occure     */

 return (t);
}

/******************************************************************************/
/** checks for each character in string str to be within bounds min und max
 * @param str pointer to input string
 * @param min lower bound character
 * @param max upper bound character
 */
  
static BOOL psi_req_chk_str (char * str,char min,char max)
{
 int i,n;

 if (str == NULL)
  return (FALSE);

 n = strlen (str);

 for (i = 0;i < n;i++)
  if (!(str [i] >= min && str [i] <= max))
   return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** check input fields according to datatype
 * @param ctx pointer to request context
 * @param idx index of field
 */
  
static BOOL psi_req_chk_dtp (CTX_PSI_REQ * ctx,IDX idx) // char * str,DTP dtp)
{
 DTP dtp;
 char * str;

 if (ctx == NULL)
  return (FALSE);

 str = ctx -> rec -> itm [idx].val;
 dtp = ctx -> sch -> fsp -> itm [idx].dtp;

 switch (dtp)
 {
  case DTP_NAME:  /* name */
  {
   psi_cleanstr (str);
   break;
  }

  case DTP_DD:    /* day */
  {
   int k;

   if (!psi_req_chk_str (str,'0','9'))
    return (FALSE);

   k = atoi (str);

   if (!(k >= 1 && k <= 31))
    return (FALSE);

   sprintf (str,"%d",k);
   break;
  }

  case DTP_DM:    /* month */
  {
   int k;

   if (!psi_req_chk_str (str,'0','9')) // assert, that input only consists of numbers
    return (FALSE);

   k = atoi (str);

   if (!(k >= 1 && k <= 12))
    return (FALSE);

   sprintf (str,"%d",k);
   break;
  }
   /* :BG: begin */
  case DTP_DATE:
  {
  /* TODO: check format of string, yyyy-mm-dd should be correct.*/
    int year,month,day;
    char *myYear;
    char *myMonth;
    char *myDay;
	

    //char *myhelp;
    //my first attempt, manually
    //myhelp = strchr(str,"-");
    //myYear = str - myhelp;
    //printf("\npsx-psi-chk.inl:180:myYear=%s\n",myYear);

    /* yyyy-mm-dd */
	PSX_STR tempstr;
	strcpy(tempstr, str);

    myYear = strtok(tempstr,"-");                /* first token in str seperated by '-'                  */
    myMonth = strtok(NULL,"-");          /* second token in str seperated by '-'            */
    myDay = strtok(NULL,"-");             /* third and last token in str seperated by '-' */
    
    if (!psi_req_chk_str (myYear,'0','9'))         /* assert, that yearstring only consists of numbers */
      return (FALSE);
    if (!psi_req_chk_str (myMonth,'0','9'))      /* assert, that yearstring only consists of numbers */
      return (FALSE);
    if (!psi_req_chk_str (myDay,'0','9'))          /* assert, that yearstring only consists of numbers */
      return (FALSE);

    if((year = atoi(myYear))<1000)                 /*string to integer, get year as a number*/
      return(FALSE);                                              /*:BG:errormessage??*/
    if((month = atoi(myMonth))>12)
      return(FALSE);                         
    if((day = atoi(myDay))>31)
      return(FALSE);
    
    /* now we have day, month and year in numbers and can do with it whatever we want */
    
    break;
  }
  /*:BG: end*/
  case DTP_DY:    /* year */
  {
   int k;

   if (!psi_req_chk_str (str,'0','9')) // assert, that input only consists of numbers
    return (FALSE);

   k = atoi (str);

   if (!(k >= 1000)) //  && k <= ))
    return (FALSE);

   sprintf (str,"%d",k);
   break;
  }

  case DTP_HIC:   /* Health Insurance Code */
  {
   HIO hio;
   HIV hiv;

   if (!psi_hic_check ())  /* check whether validation of hic is wanted */
    break;
    
   if (!psi_req_chk_hio_get (ctx,&hio))   /* get hio code                      */
    return (TRUE);        /* no hio code found so assume that hic is correct   */

   if (!hfi_check (hio,str,&hiv))         /* check health insurance code       */
   {
     LOG (MOD,"invalid health insurance code");
     return (FALSE);
   }
   break;
  }

  case DTP_SEX:    /* sex */
  {                                                         /*:BG: added possibility of unknown sex n or N (2005-02-22) */
   if (!(strcmp (str,"M") == 0 || strcmp (str,"m") == 0 || strcmp (str,"F") == 0 || strcmp (str,"f") == 0 || strcmp(str,"N") ==0 || strcmp(str,"n") == 0))
    return (FALSE);

   str_toupper (str);
   break;
  }

  case DTP_TEXT: 
  {
     str_cvt_accent(str);
     strup_g(str);          /* convert string into capital letters */      /*JM: not for KPOH!! backward compability!*/
   //psi_cleanstr (str);
   break;
  }

  default:  /* text */               
  {
   //psi_cleanstr (str);
   break;
  }

 }

 return (TRUE);
}

/******************************************************************************/
/** check minimum field length
 * @param ctx pointer to request context
 * @param idx index of field
 */
  
static BOOL psi_req_chk_min (CTX_PSI_REQ * ctx,IDX idx)
{
 if (ctx == NULL)
  return (FALSE);

 if (ctx -> sch -> fsp -> itm [idx].min == 0)
  return (TRUE);

 if (strlen (ctx -> rec -> itm [idx].val) < ctx -> sch -> fsp -> itm [idx].min)
  return (FALSE);
  


 return (TRUE);
}

/******************************************************************************/
/** check maximum field length
 * @param ctx pointer to request context
 * @param idx index
 */
  
static BOOL psi_req_chk_max (CTX_PSI_REQ * ctx,IDX idx)
{
 if (ctx == NULL)
  return (FALSE);

 if (ctx -> sch -> fsp -> itm [idx].max == 0)
  return (TRUE);

 if (strlen (ctx -> rec -> itm [idx].val) > ctx -> sch -> fsp -> itm [idx].max)
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** check expression, if context sceham formatspecification item equals EQU_REQ
 *  ther must be a value in context record item with same index
 * @param ctx pointer to request context
 * @param idx index of field
 */
 
static BOOL psi_req_chk_exs (CTX_PSI_REQ * ctx,IDX idx)
{
 if (ctx == NULL)
  return (FALSE);

 if (EQU_CMP (ctx -> sch -> fsp -> itm [idx].equ) == EQU_REQ)
  if (str_isnil (ctx -> rec -> itm [idx].val))
   return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** check input record (= request) for correctness
 * @param ctx pointer to request context structure
 */
  
static BOOL psi_req_chk (CTX_PSI_REQ * ctx,BYTE * sts)
{
 IDX i;

 *sts = PSI_STS_ZERO;

 LOG (MOD,"checking entry conditions");

 for (i = 0;i < ctx -> sch -> fsp -> n;i++)
 {
  LOG (MOD,"> checking item %d: %s -> %s",i,ctx -> rec -> itm [i].atr,ctx -> rec -> itm [i].val);

  // existence

  LOG (MOD,"check existence [%s]: [%d,%d]",ctx -> sch -> fsp -> itm [i].sym,ctx -> sch -> fsp -> itm [i].min,ctx -> sch -> fsp -> itm [i].max);

  if (!psi_req_chk_exs (ctx,i))
   {
    char len [1000];

    *sts = PSI_STS_INCOMPLETE;
    dic_put (ctx -> dic,"FLD",ctx -> sch -> fsp -> itm [i].lbl);
    txt_construct (ctx -> req -> msg,"msg.req.inc","The field '$FLD' must not be empty.",ctx -> dic);
    LOG (MOD,"check existence failed");
    return (FALSE);
   }

  // length

  LOG (MOD,"check length [%s]: [%d,%d]",ctx -> sch -> fsp -> itm [i].sym,ctx -> sch -> fsp -> itm [i].min,ctx -> sch -> fsp -> itm [i].max);

  if (!psi_req_chk_min (ctx,i))  // check minimum input size
   {
    char len [1000];

    sprintf (len,"%d",ctx -> sch -> fsp -> itm [i].min);
    *sts = PSI_STS_NAMELENGTH;
    dic_put (ctx -> dic,"FLD",ctx -> sch -> fsp -> itm [i].lbl);
    dic_put (ctx -> dic,"LEN",len);
    txt_construct (ctx -> req -> msg,"msg.req.min","The length of the field '$FLD' must be greater than $LEN.",ctx -> dic);
    LOG (MOD,"check min length failed");
    return (FALSE);
   }

  if (!psi_req_chk_max (ctx,i))
   {
    char len [1000];

    sprintf (len,"%d",ctx -> sch -> fsp -> itm [i].max);
    *sts = PSI_STS_NAMELENGTH;
    dic_put (ctx -> dic,"FLD",ctx -> sch -> fsp -> itm [i].lbl);
    dic_put (ctx -> dic,"LEN",len);
    txt_construct (ctx -> req -> msg,"msg.req.max","The length of the field '$FLD' must not be greater than $LEN.",ctx -> dic);
    LOG (MOD,"check max length failed");
    return (FALSE);
   }

  // data type

  LOG (MOD,"check data type");
  //printf("%s\n", ctx -> rec ->itm[i].val);
  if (!str_isnil (ctx -> rec -> itm [i].val))

   if (!psi_req_chk_dtp (ctx,i)) //ctx -> rec -> itm [i].val,ctx -> sch -> fsp -> itm [i].dtp))
   {
    *sts = PSI_STS_INVALID;
    dic_put (ctx -> dic,"FLD",ctx -> sch -> fsp -> itm [i].lbl);
    txt_construct (ctx -> req -> msg,"msg.req.inv","The content of the field '$FLD' is invalid.",ctx -> dic);
    LOG (MOD,"check data type failed");
    return (FALSE);
   }

  LOG (MOD,"> resulting item: '%s'",ctx -> rec -> itm [i].val);
 }

 return (TRUE);
}


/******************************************************************************/
/******************************************************************************/
/******************************************************************************/



