/******************************************************************************/
/* psx-psi.c                               Pseudonymization Service Interface */
/******************************************************************************/
/** @file psx-psi.c Pseudonymization Service Interface - Source Code File
 * The Pseudonymization Service Interface provides the handling of the PID
 * request, that includes functions for initialisation of the interface
 * structure, exiting the interface and running the PID request (containing
 * the preparation like checks and result handling).
 */
 
#include "psx.h"
//#include "psx-psi.inl" 

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

#define MOD PSX_MOD_PSI

/******************************************************************************/
/** context pseudinymisation service interface request                        */
typedef struct _CTX_PSI_REQ
{
 PSI_REQ * req;       /**< PSI request structure       */
 PSX_REC * rec;       /**< record                      */
 BOOL      sur;       /**< sureness                    */
 PSX_SCH * sch;       /**< schema                      */
 PSX_REC * ctn;       /**< control number              */
 PSX_REC * hsr;       /**< history record (plain)      */        
 PID       pid;       /**< pseudonymisation identifier */
 PIX       pix;       /**< PID index                   */
 IDX       rsl;       /**< result                      */
 BYTE      sts;       /**< status                      */
 PSX_DIC * dic;       /**< dictionary                  */
} CTX_PSI_REQ;

#include "psx-psi-chk.inl" // implements datatype specific checks on input record

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** make history
 * @param ctx CTX_PSI_REQ *, pointer to psi request context
 * @param hsr PSX_REC **, pointer to pointer to
 */
 
static BOOL psi_req_hsr_make (CTX_PSI_REQ * ctx,PSX_REC ** hsr)
{
 PSX_REC * r;
 PSX_FSP * f;
 IDX       i;

 if (!fsp_create (&f)) // create new format specification
  return (FALSE);

 if (!fsp_copy (f,ctx -> sch -> fsp))
 {
  fsp_delete (&f);
  return (FALSE);
 }

 if (!rec_create (&r))
 {
  fsp_delete (&f);
  return (FALSE);
 }

 for (i = 0;i < f -> n;i++)
  f -> itm [i].tfm &= ~(TFM_CTN | TFM_ENC);

 if (!cpi_gen (ctx -> rec,r,f,CPE_HEX)) 
 {
  LOG (MOD,"psi_req_hsr_make failed");
  return (FALSE);
 }

 fsp_delete (&f);
 *hsr = r;

 return (TRUE);
}

/******************************************************************************/
/** initialize request context
 * @param ctx pointer to request context
 * @param req pointer to request structure
 */
 
static BOOL psi_req_init (CTX_PSI_REQ * ctx,PSI_REQ * req)
{
 if (ctx == NULL || req == NULL)
  return (FALSE);

 memset (ctx,0,sizeof (*ctx));

 if (req -> rec == NULL)
  return (FALSE);

 ctx -> req = req;

 if (!pdi_sch_get (&ctx -> sch))    // get schema
  return (FALSE);

 ctx -> rec = req -> rec;           // get record

 if (!rec_create (&ctx -> ctn))     // create record
  return (FALSE);

 if (!dic_create (&ctx -> dic))     // create dictionary
  return (FALSE);

 ctx -> sur = req -> sur;           // get sureness
 //ctx -> ics = req -> ics;

 return (TRUE);
}

/******************************************************************************/
/** exit request
 * @param ctx pointer to request context
 */
 
static BOOL psi_req_exit (CTX_PSI_REQ * ctx)
{
 if (ctx == NULL)
  return (FALSE);

 if (ctx -> ctn)              // get controlnumber
  rec_delete (&ctx -> ctn);

 if (ctx -> dic)              // get dictionary
  dic_delete (&ctx -> dic);   

 return (TRUE);
}

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** run request
 */
 
static BOOL psi_req_run (CTX_PSI_REQ * ctx)
{
 PDI_REQ req;
 BYTE sts;
 PSX_STR t;
 BOOL hst;

 if (ctx == NULL)
  return (FALSE);

 hst = FALSE;

 if (psx_cfg_get (PSX_CFG_HST,t))  /* get history configuration */
  if (strcmp (t,"1") == 0)
   hst = TRUE;

 if (!rec_map_fmt (ctx -> rec,ctx -> sch -> fsp))	/* ensure our format */
  return (FALSE);

 ctx -> sts = PSI_STS_ZERO;

 if (!psi_req_chk (ctx,&sts))
 {
  ctx -> sts = sts;
  return (FALSE);
 }
 /* encipher input data  */
 if (!cpi_gen (ctx -> rec,ctx -> ctn,ctx -> sch -> fsp,CPE_HEX)) 
 {
  LOG (MOD,"psi_req_run failed");
  return (FALSE);
 }

 memset (&req,0,sizeof (req));
 req.rec = ctx -> ctn;
 req.sur = ctx -> sur;
 req.frc = ctx -> req -> frc;

 if (hst)
 {
  if (!psi_req_hsr_make (ctx,&req.hsr))           /* make history */
  {
   LOG (MOD,"psi_req_run failed (hsr)");
   return (FALSE);
  }
 }

 if (!pdi_req (&req))
 {
  ctx -> pix = 0;
  memset (ctx -> pid,0,sizeof (ctx -> pid));
  LOG (MOD,"pdi_req failed, psi_req_run fails");
  return (FALSE);
 }

 ctx -> pix = req.pix;                             /* get PID index  */
 memcpy (ctx -> pid,req.pid,sizeof (ctx -> pid));
 ctx -> req -> prm = req.prm;
 ctx -> rsl = req.rsl;

 if (req.hsr)
  rec_delete (&req.hsr);

 return (TRUE);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** do PID request (invoke pid_req_run)
 */
 
BOOL psi_req (PSI_REQ * req)
{
 CTX_PSI_REQ ctx;
 BOOL r = TRUE;

 if (req == NULL)
  return (FALSE);

 req -> sts = PSI_STS_ZERO;
 strcpy (req -> msg,"");

 if (!pdi_open ())
  return (FALSE);

 if (!psi_req_init (&ctx,req))
 {
  pdi_close ();
  return (FALSE);
 }

 if (psi_req_run (&ctx))                // execute request
 {

  if (req -> ctn)                       // request content (field names and values)
  {
   rec_clear (req -> ctn);
   if (!rec_copy (req -> ctn,ctx.ctn))
    r = FALSE;
  }

  req -> pix = ctx.pix;
  req -> rsl = ctx.rsl;

  if (req -> pid)
   memcpy (req -> pid,ctx.pid,8);
 }
 else
 {
  req -> sts = ctx.sts;
  r = FALSE;
 }

 if (!psi_req_exit (&ctx))
  r = FALSE;

 if (!pdi_close ())
  r = FALSE;

 return (r);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** get PID string
 * @param pid PID formatted as type PID
 * @param str PID string after invokation
 */
 
BOOL pid_get_str (PID pid,PSX_STR str)
{
 NUM n;

 if (pid == NULL || str == NULL)
  return (FALSE);

 n = 8; // sizeof (pid);

 strncpy (str,pid,n);
 str [n] = 0;

 return (TRUE);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


