/******************************************************************************/
/* psx-rec.c                         Record Management Interface              */
/******************************************************************************/
/** @file psx-rec.c Record Management Interface - Source Code File
 * Functions implementing the management of the record datatype. A record
 * contains data and additional information, like the label of a data field,
 * the rec module provides functions for the following record operations:
 * - create record
 * - delete record
 * - clear an item
 * - add definition to new record after clearing the old values
 * - map record into another format
 * - fill record rec with fields and values
 * - add new record item
 * - set specified record item value from source string
 * - set value
 * - get value by attribute
 * - lookup index of the record item
 * - copy record
 * - copy items from fmt to rec
 * - get length of biggest attribute in itemlist
 * - get record item attribute
 * - get random record values
 * - read single record from input file (batchmode, single)
 * - write record to file
 * - write format specification to file
 * - record input (stdin)
 * - print record
 * - convert DIN 66003 record to ISO 8859-1
 */
#include "psx.h"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/


/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** create record
 * @param rec PSX_REC **, pointer to pointer psx record
 */

BOOL rec_create (PSX_REC ** rec)
{
 if (rec == NULL)
  return (FALSE);

 if ((*rec = (PSX_REC *) malloc (sizeof (**rec))) == NULL)
  return (FALSE);

 memset (*rec,0,sizeof (**rec));

 return (TRUE);
}

/******************************************************************************/
/** delete record
 * @param rec PSX_REC **, pointer to pointer psx record
 */

BOOL rec_delete (PSX_REC ** rec)
{
 if (rec == NULL)
  return (FALSE);

 rec_clear (*rec);

 free (*rec);
 *rec = NULL;

 return (TRUE);
}

/******************************************************************************/
/** clear rec->itm
 * @param rec PSX_REC *, pointer to psx record
 */

BOOL rec_clear (PSX_REC * rec)
{
 if (rec == NULL)
  return (FALSE);

 if (rec -> n > 0)
 {
  int i;

  assert (rec -> itm != NULL);

  free (rec -> itm);
  rec -> itm = NULL;
 }

 assert (rec -> itm == NULL);

 rec -> n = 0;

 return (TRUE);
}

/******************************************************************************/
/** add definition to new record after clearing the old values
 * @param rec PSX_REC *, pointer to psx record
 * @param def PSX_REC_DEF *, pointer to record definition
 */

BOOL rec_define (PSX_REC * rec,PSX_REC_DEF * def)
{
 int i;

 if (rec == NULL || def == NULL)
  return (FALSE);

 if (!rec_clear (rec))			// clear item
  return (FALSE);

 i = 0;

 while (def [i].lbl != 0)		// go through labels
 {
  if (!rec_add1 (rec,def [i].lbl,NULL))
   return (FALSE);

  i++;
 }

 return (TRUE);
}

/******************************************************************************/
/** clear record and add items as specified in format fmt (no values yet)
 * @param rec PSX_REC *, pointer to record
 * @param fmt PSX_FSP *, pointer to format specification
 */
 
BOOL rec_def_fmt (PSX_REC * rec,PSX_FSP * fmt)
{
 int i;

 if (rec == NULL || fmt == NULL)
  return (FALSE);

 if (!rec_clear (rec))
  return (FALSE);

 for (i = 0;i < fmt -> n;i++)
 {
  if (!rec_add1 (rec,fmt -> itm [i].sym,NULL))	// add item to record, value = NULL
   return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/** map record into another format
 * @param rec pointer to psx record structure
 * @param fmt pointer to format specification
 */

BOOL rec_map_fmt (PSX_REC * rec,PSX_FSP * fmt)
{
 PSX_REC * t;
 BOOL r = TRUE;
 IDX i;

 if (rec == NULL || fmt == NULL)
  return (FALSE);

 if (!rec_create (&t))               // create new record t
  return (FALSE);

 if (!rec_copy (t,rec))              // copy rec to new record
 {
  rec_delete (&t);
  return (FALSE);
 }

 if (!rec_clear (rec))               // clear rec
 {
  rec_delete (&t);
  return (FALSE);
 }

 if (!rec_def_fmt (rec,fmt))        // define new format
 {
  rec_delete (&t);
  return (FALSE);
 }

 for (i = 0;i < t -> n;i++)         // every existing item
 {
  IDX k;

  if (rec_get_idx (rec,t -> itm [i].atr,&k))
   if (!rec_set (rec,k,t -> itm [i].val))
    return (FALSE);
 }


 if (!rec_delete (&t))
  r = FALSE;

 return (r);
}

/******************************************************************************/
/** fill record rec with fields and values specified in fmt and dic
 * @param rec PSX_REC *, pointer to (empty) record to be filled
 * @param fmt PSX_FSP *, pointer to format specification
 * @param dic PSX_DIC *, pointer to dictionary containing values of fields
 * @param pfx char *, prefix of field names (e.g. "fld_")
 */
  
BOOL rec_fetch (PSX_REC * rec,PSX_FSP * fmt,PSX_DIC * dic,const char * pfx)
{
 IDX i;
 PSX_STR p;
 BOOL r = TRUE;

 if (fmt == NULL || dic == NULL)
  return (FALSE);

 if (pfx)          // prefix
  strcpy (p,pfx);
 else
  strcpy (p,"");

 // iterates over all fields specified in fmt
 for (i = 0;i < fmt -> n;i++)
 {
  PSX_STR f;
  char * t;

  sprintf (f,"%s%s",p,fmt -> itm [i].sym); // e.g. f="fld_lname"

  if (!dic_get (dic,f,&t))               // get value for field f
  {
   psx_log_m("gui", "Could not find value for field %s in cgi parameter list.", f);
   r = FALSE;
//   break;
  }

  if (!rec_add1 (rec,fmt -> itm [i].sym,t)) // add item to record rec
  {                                                         
   r = FALSE;
   break;
  }
 } // end for

 return (r);
}

/******************************************************************************/
/** add new record item (rec->itm), with specified values
 * @param rec pointer to psx record structure
 * @param atr pointer to string, name of attribute
 * @param val pointer to string, value
 */

BOOL rec_add1 (PSX_REC * rec,const char * atr,const char * val) //BYTE * src,char * lbl)
{
 PSX_REC_ITM * t;
 int k, l;

 if (rec == NULL)
  return (FALSE);

 k = rec -> n;	// number of recorditems

 if ((t = (PSX_REC_ITM *) realloc (rec -> itm,(k + 1) * sizeof (PSX_REC_ITM))) == NULL)
  return (FALSE);

 rec -> itm = t;

 if (atr)
  strcpy (rec -> itm [k].atr,atr); 	// add new atr
 else
  strcpy (rec -> itm [k].atr,"");
  
 if (val)
 {
  if ( (l = strlen (val)) >= SYS_STR_MAX )
    return (FALSE);
  strncpy (rec -> itm [k].val,val,l);	// add new value
  rec -> itm[k].val[l] = '\0';
 }
 else
  strcpy (rec -> itm [k].val,"");

 rec -> n++;                        // increase number of items

 return (TRUE);
}

/******************************************************************************/
/** set specified record item value from source string
 * @param rec pointer to psx record
 * @param idx index of item
 * @param src pointer to string, source string (value)
 */

BOOL rec_set (PSX_REC * rec,int idx,BYTE * src)
{
 if (rec == NULL || src == NULL)
  return (FALSE);

 if (idx < 0 || idx >= rec -> n)
  return (FALSE);

 strcpy (rec -> itm [idx].val,src);        // copy source string to item value

 return (TRUE);
}

/******************************************************************************/
/** set value of rec
 * @param rec pointer to record
 * @param atr pointer to string, attribute
 * @param fmt pointer to string, format
 * @param ... parameter list
 */

BOOL rec_putf (PSX_REC * rec,const char * atr,const char * fmt,...)
{
 va_list l;
 char t [4000];
 IDX i;

 if (rec == NULL || atr == NULL || fmt == NULL)
  return (FALSE);

 va_start (l,fmt);
 vsprintf (t,fmt,l);
 va_end (l);

 if (!rec_get_idx (rec,atr,&i))		// get index of record
  return (FALSE);

 strcpy (rec -> itm [i].val,t);		// set value

 return (TRUE);
}

/******************************************************************************/
/** get value by attribute
 * @param rec pointer to psx record structure
 * @param atr pointer to string, attribute
 * @param val pointer to pointer to string, value
 */

BOOL rec_get (PSX_REC * rec,char * atr,char ** val)
{
 IDX i;

 if (rec == NULL || atr == NULL)
  return (FALSE);

 for (i = 0;i < rec -> n;i++)
  if (strcmp (rec -> itm [i].atr,atr) == 0)
  {
   if (val)
    *val = rec -> itm [i].val;
   return (TRUE);
  }

 return (FALSE);
}

/******************************************************************************/
/** lookup index of the record item where atr appears in
 * @param rec pointer to psx record structure
 * @param atr pointer to string, attribute
 * @param pointer to index, index of attribute
 */

BOOL rec_get_idx (PSX_REC * rec,const char * atr,IDX * idx)
{
 IDX i;

 if (rec == NULL || atr == NULL)
  return (FALSE);

 for (i = 0;i < rec -> n;i++)
  if (strcmp (rec -> itm [i].atr,atr) == 0)  // lookup the right atr
  {
   if (idx)
    *idx = i;                                // set index
   return (TRUE);
  }

 return (FALSE);
}

/******************************************************************************/
/** copy record src to rec
 * @param rec pointer to psx record structure, result
 * @param pointer to psx record structure, source
 */

BOOL rec_copy (PSX_REC * rec,PSX_REC * src)
{
 IDX i;

 if (rec == NULL || src == NULL)
  return (FALSE);

 for (i = 0;i < src -> n;i++)
  if (!rec_add1 (rec,src -> itm [i].atr,src -> itm [i].val))
   return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** copy items from fmt to rec
 * @param rec pointer to record
 * @param fmt pointer to format specification
 */

BOOL rec_fmt_cpy (PSX_REC * rec,PSX_FSP * fmt)
{
 IDX i;

 if (rec == NULL || fmt == NULL)
  return (FALSE);

 for (i = 0;i < fmt -> n;i++)
  if (!rec_add1 (rec,"",fmt -> itm [i].sym))
   return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** get length of biggest atr in itemlist
 * @param rec pointer to psx record structure
 * @param n pointer to integer, length of longest attribute in itemlist
 */

BOOL rec_getLblMax (PSX_REC * rec,int * n)
{
 int i,k;

 if (rec == NULL || n == NULL)
  return (FALSE);

 k = 0;

 for (i = 0;i < rec -> n;i++)
 {
  int l;

  if ((l = strlen (rec -> itm [i].atr)) > k)  // store length of biggest atr
   k = l;
 }

 *n = k;

 return (TRUE);
}

/******************************************************************************/
/** get record item attribute
 * @param rec pointer to psx record structure
 * @param idx index of record item attribute
 * @param s psx string in which record item attribute is stored
 */

BOOL rec_getLblStr (PSX_REC * rec,int idx,PSX_STR s)
{
 int n,k,i;

 if (idx < 0 || idx >= rec -> n)
  return (FALSE);

 rec_getLblMax (rec,&n);    // get size of biggest itm

 strcpy (s,rec -> itm [idx].atr);
 k = strlen (s);

 for (i = k;i < n;i++)      // append blanks to s
  strcat (s," ");

 return (TRUE);
}

/******************************************************************************/
/** get random record values
 * @param rec pointer to psx record structure
 * @param fsp pointer to format specification
 */

BOOL rec_gen_rnd (PSX_REC * rec,PSX_FSP * fsp)
{
 IDX i;

 if (rec == NULL || fsp == NULL)
  return (FALSE);

 if (!rec_def_fmt (rec,fsp))            // define format specification in rec
  return (FALSE);

 for (i = 0;i < fsp -> n;i++)
 {
  PSX_STR t;

  switch (fsp -> itm [i].dtp)           // check datatype
  {
   case DTP_TEXT:                       // text
   {
    if (!rnd_get_str (t,fsp -> itm [i].len)) // get random String
     return (FALSE);
    break;
   }

   case DTP_DATE:                       // date
   {
    NUM dy,dm,dd;

    rnd_get_num (&dy,1000,3000);        // get random number
    rnd_get_num (&dm,1,12);
    rnd_get_num (&dd,1,30);
    sprintf (t,"%04d-%02d-%02d",dy,dm,dd);
    break;
   }

   case DTP_NAME:                            // name
   {
    if (!rnd_get_str (t,fsp -> itm [i].len)) // get random string
     return (FALSE);
    break;
   }

   case DTP_DD:                             // day
   {
    NUM k;

    rnd_get_num (&k,1,30);
    sprintf (t,"%02d",k);
    break;
   }

   case DTP_DM:                             // month	
   {
    NUM k;

    rnd_get_num (&k,1,12);
    sprintf (t,"%02d",k);
    break;
   }

   case DTP_DY:                             // year
   {
    NUM k;

    rnd_get_num (&k,1000,3000);
    sprintf (t,"%04d",k);
    break;
   }

   default:
   {
    sprintf (t,"?");
    break;
   }
  }

  if (!rec_putf (rec,fsp -> itm [i].sym,"%s",t))
   return (FALSE);

 }

 return (TRUE);
}

/******************************************************************************/
/** read one record from input file (one record per line)
 * @param rec record
 * @param fmt format specification
 * @param f input file
 * @param rst rest of input line
 * @param del delimiter,e.g. for comma seperated values del = ","
 */
/*:BG:TODO:this function has to be changed, maybe in two with a switch or if(){}else{}:strtok(string,delimiter)*/
/*:BG:TODO:get option value with sci_opt_get("d:s",pointer,default);*/
BOOL rec_read (PSX_REC * rec,PSX_FSP * fmt,FILE * f, PSX_STR rst, PSX_STR del)
{
 int i,len;
 char line [1000];
   int k = 0; /*:BG:*/

 if (rec == NULL || fmt == NULL || f == NULL)
  return (FALSE);

 rec_clear (rec);

 if (!rec_def_fmt (rec,fmt))       /* construct record items according to fmt */
  return (FALSE);

 if (fgets (line,1000,f) == NULL) /* get one line from file f */
  return (FALSE);

 len = strlen (line);

 if (line [len - 1] == '\n')
  line [len - 1] = '\0';
  
  /*:BG:begin*/
  if(strcmp(del,"")!=0)       /*look, if option is set*/
  {
    char * hlp, * hlp2;//= (char*) malloc(len);
    char *lineptr;
    char * trec = line;

    lineptr = line;              /* get pointer to character array*/   

    for(i=0;i<(fmt->n -1);i++)
    {
      hlp = strchr(lineptr,del[0]);   /* get pointer to first appearance of del in string line */
	        
	  if(hlp == NULL)
      {				  /* make sure, the delimiter is correct for this record */
        //return(FALSE);
        //trec = ""; may cause error: writing into string constant
	    *trec='\0';
	  } else
      {
       k = hlp - lineptr;                /* found number of characters before del in string line */
       strncpy( trec ,lineptr, k); 
       lineptr +=k+1;             /* cut off front */
       trec[k] = 0;
	  }
       //printf("\npsx-rec.c:626:hlp=%s\nlineptr=%s\nirec=%s\n", hlp, lineptr,irec);      

     
       if (!rec_set (rec,i,trec))     /* get record value    */
         return (FALSE);
     }   // end for
     
     //free(hlp);
     
     /* last item */     
     if(  hlp2 = strchr(lineptr,del[0])   )
     {
             k = hlp2 - lineptr;                /* found number of characters before del in string line */
             strncpy(trec,lineptr,k);
             //lineptr +=k+1;             /* cut off front */
             trec[k] = 0;
            //printf("\npsx-rec.c:626:hlp=%s\nlineptr=%s\nirec=%s\n", hlp, lineptr,irec);

            if (!rec_set (rec,i,trec))     /* get record value    */
              return (FALSE);
            strcpy(rst,hlp2);
     }
     else  {
        if (!rec_set (rec,i,lineptr))     /* get last value    */
           return (FALSE);
     }
         
   } else
   {    // no delimiter set, use defined field lengthes
   /*:BG:end*/
  
    for (i = 0;i < fmt -> n;i++)
    {
      char kontrollstring [1000];
      char * str;
      int l;

      l = strlen (line);

      str = line + ((fmt -> itm [i].pos) - 1); // start position of item i
     /* copies len (= max length of item i in str) characters into kontrollstring */
     strncpy (kontrollstring,str,fmt -> itm [i].len);
     kontrollstring [fmt -> itm [i].len] = '\0';	  // actual item

     if (!rec_set (rec,i,kontrollstring))
             return (FALSE);
  }  // end for
  if (rst != NULL)
  strcpy (rst, line + (((fmt -> itm [(fmt -> n) -1].pos) - 1) + (fmt -> itm [(fmt -> n) -1].len) + 1 ) );
 
 } // end else
 
 return (TRUE);
}

/******************************************************************************/
/** write record to file
 * @param rec pointer to record structure
 * @param fsp pointer to format specification
 * @param f pointer to target file
 */

BOOL rec_write (PSX_REC * rec,PSX_FSP * fsp,FILE * f)
{
 NUM n;
 char * str;
 IDX i;

 if (rec == NULL || fsp == NULL || f == NULL)
  return (FALSE);

 if (!fmt_get_len (fsp,&n))
  return (FALSE);

 if ((str = (char *) malloc ((n + 1) * sizeof (char))) == NULL)
  return (FALSE);

 memset (str,' ',n);                       // init

 for (i = 0;i < fsp -> n;i++)
 {
  PSX_STR t;
  char * p;

  if (rec_get (rec,fsp -> itm [i].sym,&p)) // get record value
   strcpy (t,p);
  else
   strcpy (t,"");

  str_pad (t,fsp -> itm [i].len);          // append len chars to t, initialize
  

  memcpy (str + fsp -> itm [i].pos - 1,t,fsp -> itm [i].len);
 }

 str [n] = 0;

 fprintf (f,"%s\n",str);

 free (str);

 return (TRUE);
}

/******************************************************************************/
/** write format specification to file
 * @param fsp pointer to format specification
 * @param f file pointer
 */

BOOL rec_write_m (PSX_FSP * fsp,FILE * f)
{
 NUM n;
 char * str;
 IDX i;

 if (fsp == NULL || f == NULL)
  return (FALSE);

 if (!fmt_get_len (fsp,&n))
  return (FALSE);

 if ((str = (char *) malloc ((n + 1) * sizeof (char))) == NULL)
  return (FALSE);

 memset (str,' ',n);

 for (i = 0;i < fsp -> n;i++)
 {
  PSX_STR t;
  char * p;

  str_gen_c (t,'-',fsp -> itm [i].len);

  switch (fsp -> itm [i].len)
  {
   case 0:	// empty
    break;

   case 1:
    t [0] = '|';
    break;

   default:
    t [0] = '|';
    t [fsp -> itm [i].len - 1] = '|';
    break;

  }

  memcpy (str + fsp -> itm [i].pos - 1,t,fsp -> itm [i].len);
 }

 str [n] = 0;

 fprintf (f,"%s\n",str);

 free (str);

 return (TRUE);
}

/******************************************************************************/
/** read single record from file
 * @param rec pointer to psx record structure
 * @param fsp pointer to format specification
 * @param f pointer so source file
 */

BOOL rec_read_s (PSX_REC * rec,PSX_FSP * fsp,FILE * f)
{
 BOOL r;
 IDX i;

 if (rec == NULL || fsp == NULL || f == NULL)
  return (FALSE);

 rec_clear (rec);

 if (!rec_def_fmt (rec,fsp))
  return (FALSE);

 r = TRUE;

 for (i = 0;i < fsp -> n;i++)
 {
  char t [1000];
  int n;

  if (fgets (t,1000,f) == NULL)		// get one line from file f
  {
   r = FALSE;
   break;
  }

  n = strlen (t);

  if (n > 0)
  {   
   if (t [n - 1] == '\n')
    t [n - 1] = '\0';
  }
  if (!rec_set (rec,i,t))
  {
   r = FALSE;
   break;
  }
 }

 return (TRUE);
}

/******************************************************************************/
/** load record from sourcefile
 * @param rec pointer to psx record structure
 * @param pointer to format specification
 * @param src pointer to source string
 */

BOOL rec_load (PSX_REC * rec,PSX_FSP * fsp,char * src)
{
 FILE * f;
 BOOL r;
 int i;

 if (rec == NULL || fsp == NULL || src == NULL)
  return (FALSE);

 if ((f = fopen (src,"rt")) == NULL)
 {
  fprintf (stderr,"file open error: '%s'!\n",src);
  return (FALSE);
 }

 r = TRUE;

 if (!rec_read_s (rec,fsp,f))
  r = FALSE;

 fclose (f);

 return (r);
}

/******************************************************************************/
/** record input (stdin)
 * @param rec pointer to psx record structure
 * @param idn number of leading space characters, padding
 */

BOOL rec_input (PSX_REC * rec,int idn)
{
 int i;
 char pad [SYS_STR_MAX];

 if (rec == NULL)
  return (FALSE);

 memset (pad,' ',idn); pad [idn] = 0;    // padding

 for (i = 0;i < rec -> n;i++)   // n: number of elements of record rec
 {
  char l [1000],t [1000];
  int n;

  rec_getLblStr (rec,i,l);     // l: label [e.g. 'name', 'aname', ...]
  printf ("%s[%s] > ",pad,l);
  fgets (t,1000,stdin);						// read t from stdin 
  //fscanf (stdin,"%s", t);
  

  if ((n = strlen (t)) > 0)
   if (t [n - 1] == '\n' || t [n - 1] == '\r')
    t [n - 1] = 0;
      
  rec_set (rec,i,t);
 }

 return (TRUE);
}

/******************************************************************************/
/** print record on stdout
 * @param rec pointer to psx record structure
 */

BOOL rec_print (PSX_REC * rec)
{
 int    i;
 char * f;

 if (rec == NULL)
  return (FALSE);

 printf ("REC[%p]\n",rec);

 REC_ITERATE (rec,i,f)
 {
  PSX_STR t;

  rec_getLblStr (rec,i,t);
  printf ("[%s]: [%s]\n",t,f);
 }

 return (TRUE);
}

/******************************************************************************/
/* convert DIN 66003 record (health insurance card) into ISO 8859-1 format */
/** convert DIN 66003 encoded german specific letters into ISO 8859-1 format
 *  @param rec pointer to record struct, ISO 8850-1 converted after invocation
 */

BOOL rec_cvt_din (PSX_REC * rec)
{
 int i;
 char * val;

 if (rec == NULL)
  return (FALSE);
  
 REC_ITERATE (rec,i,val)
 {
    str_cvt_din_grm (val);
 }
  
 return (TRUE);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/



