/******************************************************************************/
/* psx-rgs.c                             Registry Management Interface        */
/******************************************************************************/
/** @file psx-rgs.c Registry Management Interface - Source Code File
 * Functions implementing the interface to the registry file, also called
 * configuration file. The configuration file has to be read first, then the
 * data structures of the psx programme can be filled.
 * For other modules the Registry Management Interface offers the following
 * functionalities:
 * - creation of a new registry structure
 * - deletion of an old registry structure
 * - initialisation of a registry structure
 * - writing of an entry into registry structure
 * - loading a registry file
 * - reading from a registry file
 * - looking up entries in the registry
 * - output of the registry (on stdout) 
 */
#include "psx.h"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

#define DEL '.'           // delimiter
#define RGS_STR_MAX 2048

/******************************************************************************/
/** get line from file
 * @param f pointer to registry source file
 * @param s line to be read
 */

static BOOL rgs_readl (FILE * f,PSX_STR s)
{
 int n;

 assert (f != NULL && s != NULL);

 if (fgets (s,PSX_STR_MAX,f) == NULL)
  return (FALSE);

 if ((n = strlen (s)) > 0)
  if (s [n - 1] == '\n' || s [n - 1] == '\r')
   s [n - 1] = 0;

 return (TRUE);
}

/******************************************************************************/
/** get value from file
 * @param dst pointer to value destination string
 * @param pointer to name string of input registry source file
 */

static BOOL rgs_val_load (char * dst,char * f)
{
 FILE * stm;
 char p [SYS_STR_MAX];
 BOOL r = TRUE;

 if (dst == NULL || f == NULL)
  return (FALSE);

 if ((stm = fopen (f,"rt")) == NULL)
  return (FALSE);

 if (fgets (dst,PSX_STR_MAX,stm) == NULL)
  return (FALSE);

 fclose (stm);

 return (TRUE);
}

/******************************************************************************/
/** get path information
 * @param pth pointer to pointer to string, path name
 * @param sym pointer to string
 */

static BOOL rgs_shift (char ** pth,char * sym) 
{
 char * p;  

 if (pth == NULL || *pth == NULL || sym == NULL)
  return (FALSE);

 if ((*pth) [0] == 0) 
  return (FALSE);

 if ((p = strchr (*pth,DEL)) != NULL)   // pointer to first '.' in *pth
 {
  int l = p - *pth;                     // offset to '.'
  strncpy (sym,*pth,l);                 // copy 1 character from pth to sym
  sym [l] = 0;                          // close string with \0
  *pth += l + 1;
 }
 else
 {
  int l = strlen (*pth);                // full length of string without delimiter
  strcpy (sym,*pth);                    // copy path to sym
  *pth += l;                            // next path
 }

 return (TRUE);
}

/******************************************************************************/
/** trim string, cut off whitespace (front and back) and quotes (single and
 *  double)
 * @param l pointer to string
 */

static BOOL rgs_trim (char * l)
{
 char * p;
 int n,k,i0,i1;

 if (l == NULL)
  return (FALSE);

 n = strlen (l);                             // n is length of string l

 if (n == 0)
  return (TRUE);

 i0 = 0;

 while (isspace (l [i0]) && l [i0] != 0)     // jump over leading spaces
  i0++;                                      // and count them

 i1 = n - 1;                                 // i1 is number of last element in string

 while (isspace (l [i1]) && i1 > 0)          // jump over space from behind
  i1--;                                      // and count them

 if ((l [i0] == '"' && l [i1] == '"') || (l [i0] == '\'' && l [i1] == '\''))
 {  /*         "something"                        'something'             */
  i0++;                                      // count quotes, single and double
  i1--;
 }

 if (i0 > i1)
 {
  l [0] = 0;                                 // string is empty then
  return (TRUE);
 }

 k = i1 - i0 + 1;         // cut off leading spaces, spaces from back and quotes
 memmove (l,l + i0,k);
 l [k] = 0;

 return (TRUE);
}

/******************************************************************************/
/** parse one given line of registry (configuration file)
 * if line contains assignment, attribute and value are stored in a and v
 * @param l pointer to input string line
 */
 
static BOOL rgs_parse (char * l,char * a,char * v)
{
 char * p;
 int k;

 if (l == NULL || a == NULL || v == NULL)
  return (FALSE);

 if ((p = strstr (l,"//")) != NULL) // remove comments
  *p = 0;

 if ((p = strstr (l,"=")) == NULL)  // check that l contains assignment
  return (FALSE);


 k = p - l;
 strncpy (a,l,k);     // copy attribute name into a
 a [k] = 0;

 strcpy (v,p + 1);    // copy value into v

 rgs_trim (a);        // cut off space and quotes
 rgs_trim (v);

 // added: retrieval from file

 if (v [0] == '<')    // registry input is stored in file
 {
  char f [PSX_STR_MAX];

  // added 09-15-2004: avoid buffer overflow if v contains more than 1024 characters
  int n;
  n = strlen (v+1);
  
  if (n <= 1024)
    strncpy (f,v+1,n);
  else
    return (FALSE);
  // end added
  
    //strcpy (f,v+1);
  rgs_trim (f);

  if (!rgs_val_load (v,f))     // get value from file
   return (FALSE);
 }

 if (a [0] == 0 || v [0] == 0)
  return (FALSE);

 return (TRUE);
}


/********************************************************************************/
/** get attribute (name of registry assignment, e.g. 'dbs')
 * @param rgs pointer to registry information
 * @param par number of parameter
 * @param sym pointer to name string
 * @param idx index
 */

static BOOL rgs_atr_put(PSX_RGS * rgs,IDX par,char * sym,IDX * idx)
{
 RGS_ATR * p;
 IDX i;

 if (rgs == NULL || sym == NULL)
  return (FALSE);

 // create new element 
 p = (RGS_ATR *) realloc (rgs -> atr.p,(rgs -> atr.n + 1) * sizeof (RGS_ATR));

 if (p == NULL)
  return (FALSE);
  	
 i = rgs -> atr.n;                    // get number f attribute
 rgs -> atr.p = p;                    // assign memory block

 rgs -> atr.p [i].par = par;
 strcpy (rgs -> atr.p [i].sym,sym);   // copy name to new element

 rgs -> atr.n++;

 if (idx)
  *idx = i;                           // total number of attrib parameters
//printf ("rgs_atr_put: par=%d,sym='%s',idx=%d\n",par,sym,i);

 return (TRUE);
}

/******************************************************************************/
/** find an existing entry or make a new one. There are two ways of usage for
 *  this procedure: if make is 0, sym will be looked up and false is return, if
 *  no entry is found; if make is 1, a new entry will be made, if the lookup
 *  failed
 * @param rgs pointer to registry information
 * @param idx index
 * @param sym pointer to name string
 * @param make TRUE, if new entry is made or FALSE, if just the lookup is important
 */

static BOOL rgs_atr_val_sub (PSX_RGS * rgs,IDX * idx,char * sym,BOOL make)
{
 IDX i;

 if (rgs == NULL || idx == NULL || sym == NULL) 
  return (FALSE);

 for (i = 0;i < rgs -> atr.n;i++)  // lookup all registered attributes
 {
  if (rgs -> atr.p [i].par == *idx && strcmp (rgs -> atr.p [i].sym,sym) == 0)
  {
   *idx = i;                       // sym is already in registry struct
   return (TRUE);
  }
 }

 if (!make)                                // lookup failed
  return (FALSE);

 if (!rgs_atr_put (rgs,*idx,sym,idx))      // put sym to registry struct
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** insert attribute into registry information or just lookup if it is already
 *  inserted
 * @param rgs pointer to registry information
 * @param atr pointer to constant attribute name string
 */

static BOOL rgs_atr_val (PSX_RGS * rgs,const char * atr,IDX * idx,BOOL make)
{
 char * p = (char*)atr,sym [SYS_STR_MAX]; 
 IDX  i;

 i = IDX_NIL;	// IDX_NIL = 0xFFFFFFFF


 while (rgs_shift (&p,sym))                 // p --> atr; initialize sym 
 {                                          // sym: registry symbol
  if (!rgs_atr_val_sub (rgs,&i,sym,make))   // existence check, create a new element
   return (FALSE);
 }

 if (idx)
  *idx = i;

 return (TRUE);
}

/******************************************************************************/
/** put value to registry
 * @param rgs pointer to registry information
 * @param atr index of correct attribute
 * @param val pointer to value string
 */

static BOOL rgs_val_put (PSX_RGS * rgs,IDX atr,char * val)
{
 RGS_VAL * p;
 IDX i;

 if (rgs == NULL || val == NULL)
  return (FALSE);

 if (atr >= rgs -> atr.n)  // atr: index of respective attribute
  return (FALSE);

 p = (RGS_VAL *) realloc (rgs -> val.p,(rgs -> val.n + 1) * sizeof (RGS_VAL));

 if (p == NULL)
  return (FALSE);

 i = rgs -> val.n;
 rgs -> val.p = p;

 rgs -> val.p [i].idx = atr;
 strcpy (rgs -> val.p [i].dat,val);

 rgs -> val.n++;
//printf ("rgs_val_put: atr=%d,val='%s',idx=%d\n",atr,val,i);

 return (TRUE);
}

/******************************************************************************/
/** lookup value corresponding to attribute 
 * @param rgs pointer to registry information
 * @param atr index of corresponding attribute
 * @param idx index of found value
 */

static BOOL rgs_val_get (PSX_RGS * rgs,IDX atr,IDX * idx)

{
 IDX i;

 if (rgs == NULL || idx == NULL)
  return (FALSE);

 i = 0;

 while (i < rgs -> val.n)           // for every value
 {
  if (rgs -> val.p [i].idx == atr)  // compare with number of attribute
  {
   if (idx)
    *idx = i;
   return (TRUE);                   // found
  }
  i++;
 }

 return (FALSE);                    // not found
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** create new registry structure and initialize
 * @param rgs pointer to pointer to registry information structure
 */

BOOL rgs_create (PSX_RGS ** rgs)
{
 PSX_RGS * r;

 if (rgs == NULL)
  return (FALSE);

 if ((r = SYS_ALLOCATE (PSX_RGS)) == NULL)
  return (FALSE);

 r -> atr.p = NULL;
 r -> atr.n = 0;

 r -> val.p = NULL;
 r -> val.n = 0;

 *rgs = r;

 return (TRUE);
}

/******************************************************************************/
/** delete registry structure element
 * @param rgx pointer to pointer to registry information structure
 */

BOOL rgs_delete (PSX_RGS ** rgs)
{
 if (rgs == NULL)
  return (FALSE);

 if (*rgs == NULL)
  return (FALSE);

 if ((*rgs) -> atr.p)
  free ((*rgs) -> atr.p);

 free (*rgs);
 *rgs = NULL;

 return (TRUE);
}

/******************************************************************************/
/** put registry entry to registry information structure
 * @param rgs pointer to registry information
 * @param atr pointer to constant strng, attribute name
 * @param val pointer to value string
 */

BOOL rgs_put (PSX_RGS * rgs,const char * atr,char * val)
{
 IDX i;

 if (rgs == NULL || atr == NULL || val == NULL)
  return (FALSE);

 if (!rgs_atr_val (rgs,atr,&i,TRUE))   // validate attribute
  return (FALSE);

 if (!rgs_val_put (rgs,i,val))         // put value
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** get value by attribute
 * @param rgs pointer to registry information
 * @param atr pointer to constant string, attribute name
 * @param val pointer to value string
 */

BOOL rgs_get (PSX_RGS * rgs,const char * atr,char * val)
{
 IDX i;

 if (rgs == NULL || atr == NULL || val == NULL)
  return (FALSE);

 if (!rgs_atr_val (rgs,atr,&i,FALSE))       // lookup entry (don't make a new one)
  return (FALSE);

 if (i >= rgs -> atr.n)                     // validate found value
  return (FALSE);

 if (!rgs_val_get (rgs,i,&i))               // lookup 
  return (FALSE);

 if (i >= rgs -> val.n)                     // validate again
  return (FALSE);	

 strcpy (val,rgs -> val.p [i].dat);         // get value

 return (TRUE);
}

/******************************************************************************/
/** read configuration file
 * @param rgs pointer to registry information
 * @param stm pointer to registry source file (config file)
 */

BOOL rgs_read (PSX_RGS * rgs,FILE* stm)
{
 char l [RGS_STR_MAX],a [RGS_STR_MAX],v [RGS_STR_MAX]; //l: line from configfile, a: attribute, v: value
 BOOL r = TRUE;

 if (rgs == NULL || stm == NULL)
  return (FALSE);

 // iterate over all lines of configuration file 
 while (rgs_readl (stm,l))       // read line l from file
 {
  strcpy (a,"");
  strcpy (v,"");

  if (!rgs_parse (l,a,v))        // parse registry line l
  {
   //r = FALSE;
   continue;

  }

  if (!rgs_put (rgs,a,v))        // put in structure
  {
   r = FALSE;
   continue;
  }

 }

 return (r);
}

/******************************************************************************/
/* not implemented                                                            */
/*
BOOL rgs_write (PSX_RGS * rgs,FILE * stm)
{
 return (TRUE);
}
*/
/******************************************************************************/
/** load configuration file
 * @param rgs pointer to registry information
 * @param f pointer to file name string
 */

BOOL rgs_load (PSX_RGS * rgs,char * f)   // f= (/path/to/)registry file
{
 FILE * stm;
 char p [SYS_STR_MAX];
 BOOL r = TRUE;

 sprintf (p,"f:r:%s",f);

 if ((stm = fopen (f,"rt")) == NULL)
  return (FALSE);

 if (!rgs_read (rgs,stm))                // read from file
  r = FALSE;

 fclose (stm);

 return (r);
}

/******************************************************************************/
/* not implemented                                                            */
/*
BOOL rgs_save (PSX_RGS * rgs,char * f)
{
 return (FALSE);
}
*/
/******************************************************************************/
/** output on stdout
 * @param rgs pointer to registry information
 */

BOOL rgs_dmp (PSX_RGS * rgs)
{
 IDX i;

 if (rgs == NULL)
  return (FALSE);

 printf ("rgs [%p]:\n",rgs);
 printf ("- atr:\n");

 for (i = 0;i < rgs -> atr.n;i++)
  printf ("  %2d: par: %2d, sym: %s\n",i,rgs -> atr.p [i].par,rgs -> atr.p [i].sym);

 printf ("- val:\n");

 for (i = 0;i < rgs -> val.n;i++)
  printf ("  %2d: atr: %2d, dat: %s\n",i,rgs -> val.p [i].idx,rgs -> val.p [i].dat);

 return (TRUE);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/



