/******************************************************************************/
/* psx-sch.c                                                 Schema Interface */
/******************************************************************************/
/** @file psx-sch.c Schema Interface - Source Code File
 * Functions providing the usage of a database schema. This schema is not only
 * used for the database itself, but by other modules to look up fields and
 * their order.
 */
 
#include "psx.h"

/******************************************************************************/
/*                                                                            */
/******************************************************************************/

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** get component set
 * @param sch PSX_SCH * pointer to psx schema interface structure
 * @param idx IDX index
 * @param cmp CMP cipher component
 * @param sym PSX_STR symbol string
 */

static BOOL sch_sym_get_cps (PSX_SCH * sch,IDX idx,CMP cmp,PSX_STR sym)
{
 CPS cps;
 PSX_STR t;

 if (sch == NULL)
  return (FALSE);

 if (idx >= sch -> fsp -> n)	// n number of specification records
  return (FALSE);

 cpi_cps_map_tfm (&cps,sch -> fsp -> itm [idx].tfm);   // set cps

 fsp_tfm_enc (sch -> fsp -> itm [idx].tfm,t);          // get tfm for t

 // printf ("sch_sym_get_cps: fld='%s',tfm='%s',cmp=%02XH\n",sch -> fsp -> itm [idx].sym,t,cmp);

 switch (cmp)                // cipher component
 {
  case CMP_N1:
  case CMP_N2:
  case CMP_N3:
  case CMP_PC:
  case CMP_PH:
  {
   PSX_STR s1,s2;

   if (!TEST (cps,CPS_CMP (cmp)))
    return (FALSE);

   strcpy (s1,sch -> fsp -> itm [idx].sym);

   if (!cpi_cmp_enc (cmp,s2))
    return (FALSE);

   sprintf (sym,"%s_%s",s1,s2);
   break;
  }

  case CMP_NIL:
   if (cps != 0)
    return (FALSE);

   strcpy (sym,sch -> fsp -> itm [idx].sym);
   break;

  default:
   return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/** add a record
 * @param sch PSX_SCH * pointer to psx schema interface structure
 * @param idx IDX index
 * @param rec PSX_REC * pointer to psx record
 * @param cmp CMP cipher component
 */

static BOOL sch_rec_add_if (PSX_SCH * sch,IDX idx,PSX_REC * rec,CMP cmp)
{
 PSX_STR s,t;

 if (sch == NULL || rec == NULL)
  return (FALSE);

 if (idx >= sch -> fsp -> n)
  return (FALSE);

 if (!sch_sym_get_cps (sch,idx,cmp,s))	// get component set
  return (FALSE);

 sprintf (t,"r_%s",s);					// "r_[component set]"

 rec_add1 (rec,s,t);

 return (TRUE);
}

/******************************************************************************/
/** modify src to be a valid value
 * @param src char * pointer to source string
 */

static BOOL sch_val_make (char * src)
{
 char s [1000],t [1000], temp[1000];
 int n;

 if (src == NULL)
  return (FALSE);

 strcpy (s,src);
 str_trim (s);

 if (!str_isnil (s))
 {
   n = strlen(s);    //
   strncpy(temp,s,n);
   temp[n] = '\0';
   dbi_val_cvt(temp,s); // escape apostrophe 
  sprintf (t,"'%s'",s);
 }
 else
  strcpy (t,"NULL");

 strcpy (src,t);

 return (TRUE);
}

/******************************************************************************/
/** set PID retrieval mode
 * @param prm PRM * pid retrieval mode
 * @param sym PSX_STR symbol string
 */

static BOOL sch_prm_set (PRM * prm,PSX_STR sym) //sym = 0
{
 if (prm == NULL || sym == NULL)
  return (FALSE);

 if (strncmp (sym,"0",1) == 0) 
  *prm = PRM_ZERO;
 else
  if (strncmp (sym,"1",1) == 0)
   *prm = PRM_GET;
  else
   if (strncmp (sym,"*",1) == 0)
    *prm = PRM_GEN;
   else
    return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** get PID retrieval mode
 * @param prm PRM pid retrieval mode (PRM_ZERO, PRM_GET or PRM_GEN)
 * @param sym PSX_STR sym symbol string (result is 0,1 or *)
 */

static BOOL sch_prm_get (PRM prm,PSX_STR sym)
{
 if (sym == NULL)
  return (FALSE);

 switch (prm)
 {
  case PRM_ZERO:
   strcpy (sym,"0");
   break;

  case PRM_GET:
   strcpy (sym,"1");
   break;

  case PRM_GEN:
   strcpy (sym,"*");
   break;

  default:
   return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/** set record update mode
 * @param rum RUM * pointer to record update mode
 * @param sym PSX_STR symbol string (0|1)
 */

static BOOL sch_rum_set (RUM * rum,PSX_STR sym)
{
 if (rum == NULL || sym == NULL)
  return (FALSE);

 if (strncmp (sym,"0",1) == 0)
  *rum = RUM_ZERO;
 else
  if (strncmp (sym,"1",1) == 0)
   *rum = RUM_UPDATE;
  else
   return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** get record update mode
 * @param rum RUM record update mode (RUM_ZERO or RUM_UPDATE)
 * @param sym PSX_STR symbol string (result is 0 or 1)
 */

static BOOL sch_rum_get (RUM rum,PSX_STR sym)
{
 if (sym == NULL)
  return (FALSE);

 switch (rum)
 {
  case RUM_ZERO:
   strcpy (sym,"0");
   break;

  case RUM_UPDATE:
   strcpy (sym,"1");
   break;

  default:
   return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/


/******************************************************************************/
/** schema prologue specification: set statustype and matchmode, decode sym



 * @param stt STT * pointer to status type
 * @param mtm MTM * pointer to match mode
 * @param sym PSX_STR symbol string to be decoded
 */

BOOL sch_sps_dec (STT * stt,MTM * mtm,PSX_STR sym)
{
 if (stt == NULL || mtm == NULL || sym == NULL)
  return (FALSE);

 if (strcmp (sym,"I") == 0)
 {
  *stt = STT_INPUT;
  *mtm = 0;
  return (TRUE);
 }

 *stt = STT_QUERY;

 if (!mti_mtm_dec (mtm,sym))
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** encode schema prologue specification
 * @param stt STT status type
 * @param mtm MTM match mode
 * @param sym PSX_STR symbol string
 */

BOOL sch_sps_enc (STT stt,MTM mtm,PSX_STR sym)
{
 if (sym == NULL)
  return (FALSE);

 switch (stt)
 {

  case STT_INPUT: strcpy (sym,"I"); break;
  case STT_QUERY:
   if (!mti_mtm_enc (mtm,sym))
    return (FALSE);
   break;
  default:
   return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/** set status destination type
 * @param sdt SDT * pointer to status destination type (result is SDT_STS, SDT_RSL
 *                  or SDT_ZERO)
 * @param sym PSX_STR symbol string (S|R)
 */

static BOOL sch_sdt_set (SDT * sdt,PSX_STR sym)
{
 if (sdt == NULL || sym == NULL)
  return (FALSE);

 if (strcmp (sym,"S") == 0)
  *sdt = SDT_STS;
 else
  if (strcmp (sym,"R") == 0)
   *sdt = SDT_RSL;
  else
  {
   *sdt = SDT_ZERO;
   return (FALSE);
  }


 return (TRUE);
}

/******************************************************************************/
/** get status data type
 * @param sdt SDT status data type
 * @param sym PSX_STR symbol string (result is -, S, R or ?)
 */

static BOOL sch_sdt_get (SDT sdt,PSX_STR sym)
{
 if (sym == NULL)
  return (FALSE);

 switch (sdt)
 {
  case SDT_ZERO: strcpy (sym,"-"); break;
  case SDT_STS:  strcpy (sym,"S"); break;
  case SDT_RSL:  strcpy (sym,"R"); break;
  default:       strcpy (sym,"?"); return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/** encode target
 * @param sch PSC_SCH * pointer to psx schema nterface structure
 * @param tgt PSP_TGT * pointer to procedure specificaton target structure
 * @param sym PSX_STR symbol string  (result is Z, psp item symbol or rsp item symbol)
 */

static BOOL sch_tgt_enc (PSX_SCH * sch,PSP_TGT * tgt,PSX_STR sym)
{
 PSX_STR st,ss;

 if (tgt == NULL || sym == NULL)
  return (FALSE);

 strcpy (sym,"?");

 if (!sch_sdt_get (tgt -> sdt,st))			// get st by sdt
  return (FALSE);

 switch (tgt -> sdt)
 {
  case SDT_ZERO:	strcpy (sym,"Z");	break;
  case SDT_STS:		sprintf (sym,"%s:%s",st,sch -> psp -> itm [tgt -> idx].sym);	break;
  case SDT_RSL:		sprintf (sym,"%s:%s",st,sch -> rsp -> itm [tgt -> idx].sym);	break;
  default:	return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/** resolve target
 * @param sch PSX_SCH * pointer to schema interface structure
 * @param tgt PSP_TGT * pointer to target
 */

static BOOL sch_tgt_res (PSX_SCH * sch,PSP_TGT * tgt)
{
 IDX i;



 if (sch == NULL || tgt == NULL)
  return (FALSE);



 switch (tgt -> sdt)
 {
  case SDT_ZERO:
   tgt -> idx = IDX_NIL;
   break;

  case SDT_STS:
   for (i = 0;i < sch -> psp -> n;i++)
    if (strcmp (sch -> psp -> itm [i].sym,tgt -> sym) == 0)
    {
     tgt -> idx = i;
     return (TRUE);
    }

   psx_put ("unresolved [s]: %s\n",tgt -> sym);
   return (FALSE);

  case SDT_RSL:
   for (i = 0;i < sch -> rsp -> n;i++)
    if (strcmp (sch -> rsp -> itm [i].sym,tgt -> sym) == 0)

    {
     tgt -> idx = i;
     return (TRUE);
    }


   psx_put ("unresolved [r]: %s\n",tgt -> sym);
   return (FALSE);

  default:
   return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/** decompose symbol string, set target
 * @param sch PSX_SCH * pointer to schema interface structure
 * @param tgt PSP_TGT * pointer to target
 * @param scm PSX_STR symbol string (Z, :S, :R)
 */

static BOOL sch_tgt_dec (PSX_SCH * sch,PSP_TGT * tgt,PSX_STR sym)
{
 if (tgt == NULL || sym == NULL)
  return (FALSE);

 memset (tgt,0,sizeof (*tgt));



 if (sym [0] == 0)
  return (FALSE);

 if (strcmp (sym,"Z") == 0)
 {
  tgt -> sdt = SDT_ZERO;
  tgt -> idx = IDX_NIL;
  strcpy (tgt -> sym,"");
 }
 else
 {
  if (sym [1] != ':')
   return (FALSE);

  switch (sym [0])
  {
   case 'S':
    tgt -> sdt = SDT_STS;
    break;

   case 'R':
    tgt -> sdt = SDT_RSL;
    break;

   default:
    return (FALSE);
  }

  strcpy (tgt -> sym,sym + 2);
 }

 return (TRUE);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** create new format specification, result specification and procedrue spec.
 * @param sch PSX_SCH ** pointer to pointer to schema interface structure
 */

BOOL sch_create (PSX_SCH ** sch)
{
 if (sch == NULL)
  return (FALSE);

 if ((*sch = (PSX_SCH *) malloc (sizeof (**sch))) == NULL)
  return (FALSE);

 memset (*sch,0,sizeof (**sch));

 if (!fsp_create (&(*sch) -> fsp))
 {
  sch_delete (sch);
  return (FALSE);
 }

 if (!rsp_create (&(*sch) -> rsp))
 {
  sch_delete (sch);
  return (FALSE);
 }

 if (!psp_create (&(*sch) -> psp))
 {
  sch_delete (sch);
  return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/** delete schema after clearing its values
 * @param sch PSX_SCH ** pointer to opitner to schema interface structure
 */

BOOL sch_delete (PSX_SCH ** sch)
{
 if (sch == NULL)
  return (FALSE);

 sch_clear (*sch);	// clear pointer in structure

 free (*sch);
 *sch = NULL;

 return (TRUE);
}

/******************************************************************************/
/** clear all values of schema interface structure
 * @param PSX_SCH * pointer to schema interface structure
 */

BOOL sch_clear (PSX_SCH * sch)
{
 if (sch == NULL)
  return (FALSE);

 if (sch -> fsp)
  fsp_delete (&sch -> fsp);

 if (sch -> rsp)
  rsp_delete (&sch -> rsp);

 if (sch -> psp)
  psp_delete (&sch -> psp);

 return (TRUE);
}

/******************************************************************************/
/** assign format specification in schema interface structure
 * @param sch PSX_SCH * pointer to schema interface structure
 * @param fsp PSX_FMT * pionter to format specification
 */

BOOL sch_assign_fsp (PSX_SCH * sch,PSX_FMT * fsp)
{
 IDX i;

 if (sch == NULL || fsp == NULL)
  return (FALSE);

 for (i = 0;i < fsp -> n;i++)
 {
  FMT_ITM itm;
  IDX idx_itm;
  PSX_STR t,tfm,s_cps,equ,xch;
  CPS cps;

  memset (&itm,0,sizeof (itm));

  itm.idx = IDX_NIL;
  strcpy (itm.sym,fsp -> itm [i].sym);
  strcpy (itm.lbl,fsp -> itm [i].lbl);
  itm.dtp = fsp -> itm [i].dtp;

  itm.min = fsp -> itm [i].min;

  itm.max = fsp -> itm [i].max;

  //fsp_tfm_get (fsp -> itm [i].tfm,tfm);

  itm.tfm = fsp -> itm [i].tfm;

  //cpi_cps_map_tfm (&itm.cps,fsp -> itm [i].tfm);

  //cpi_cps_get (cps,s_cps);

  itm.equ = fsp -> itm [i].equ;

  if (!sch_fsp_add (sch,&itm))
   return (FALSE);
 }



 return (TRUE);
}

/******************************************************************************/
/** add format specification item to schema
 * @param sch PSX_SCH * pointer to schema structure
 * @param itm FSP_ITM * pointer to format specification item
 */

BOOL sch_fsp_add (PSX_SCH * sch,FSP_ITM * itm)
{
 if (sch == NULL || itm == NULL)
  return (FALSE);

 if (!fsp_add (sch -> fsp,itm))
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** add result specification item to sch
 * @param sch PSX_SCH * pointer to schema structure
 * @param itm RSP_ITM * pointer to result specification item
 */

BOOL sch_rsp_add (PSX_SCH * sch,RSP_ITM * itm)
{
 if (sch == NULL || itm == NULL)
  return (FALSE);

 if (!rsp_add (sch -> rsp,itm))
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** add procedure specification to schema
 * @param sch PSX_SCH * pointer to schema structure
 * @param itm PSP_ITM * result specification item
 */


BOOL sch_psp_add (PSX_SCH * sch,PSP_ITM * itm)
{
 if (sch == NULL || itm == NULL)
  return (FALSE);

 if (!psp_add (sch -> psp,itm))
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** get component set and append to "f_"
 * @param PSX_SCH * pointer to schema structure
 * @param idx IDX index of field
 * @param cmp CMP component
 * @param sym PSX_STR symbol string (result is "f_[fieldname]")
 */

BOOL sch_atr_sym_get (PSX_SCH * sch,IDX idx,CMP cmp,PSX_STR sym)
{
 PSX_STR t;

 if (sch == NULL || sym == NULL)
  return (FALSE);

 if (idx >= sch -> fsp -> n)
  return (FALSE);

 if (!sch_sym_get_cps (sch,idx,cmp,t))  // get component set
  return (FALSE);

 sprintf (sym,"f_%s",t);

 return (TRUE);
}

/******************************************************************************/
/** look up value from record
 * @param sch PSX_SCH * pointer to schema interface structure
 * @param idx IDX index of field
 * @param cmp CMP component
 * @param rec PSX_REC * pointer to record
 * @param val PSX_STR value string
 */

BOOL sch_lookup (PSX_SCH * sch,IDX idx,CMP cmp,PSX_REC * rec,PSX_STR val)
{
 PSX_STR t;
 char * p;

 if (sch == NULL || val == NULL)
  return (FALSE);

 if (idx >= sch -> fsp -> n)
  return (FALSE);

 if (!sch_sym_get_cps (sch,idx,cmp,t))		// get component set
  return (FALSE);

 if (!rec_get (rec,t,&p))					// get value p by atr t from record
  return (FALSE);

 if (p == NULL)
  return (FALSE);

 strcpy (val,p);							// p is value

 return (TRUE);
}



/******************************************************************************/
/** get string list
 * @param sch PSX_SCH * pointer to schema interface structure
 * @param idx IDX field index
 * @param stl PSX_STL * pointer to string list
 */

BOOL sch_itm_get_stl (PSX_SCH * sch,IDX idx,PSX_STL * stl)
{

 PSX_STR t;

 if (sch == NULL || stl == NULL)
  return (FALSE);

 if (idx >= sch -> fsp -> n)
  return (FALSE);

 if (sch_sym_get_cps (sch,idx,CMP_NIL,t))	stl_add (stl,t);
 if (sch_sym_get_cps (sch,idx,CMP_N1,t))	stl_add (stl,t);
 if (sch_sym_get_cps (sch,idx,CMP_N2,t))	stl_add (stl,t);
 if (sch_sym_get_cps (sch,idx,CMP_N3,t))	stl_add (stl,t);
 if (sch_sym_get_cps (sch,idx,CMP_PC,t))	stl_add (stl,t);
 if (sch_sym_get_cps (sch,idx,CMP_PH,t))	stl_add (stl,t);

 return (TRUE);
}

/******************************************************************************/
/** get string list for each item of schema 
 * @param sch PSX_SCH * pointer to schema interface structure
 * @param stl PSX_STL * pointer to string list
 */

BOOL sch_sym_get_stl (PSX_SCH * sch,PSX_STL * stl)
{
 IDX i;

 if (sch == NULL || stl == NULL)
  return (FALSE);

 for (i = 0;i < sch -> fsp -> n;i++)		// for each field in format spec 
  sch_itm_get_stl (sch,i,stl);				// get string list = list of database fields

 return (TRUE);
}

/******************************************************************************/
/** get attribute and value
 * @param sch PSX_SCH * pointer to schema interface structure
 * @param rec PSX_REC * pointer to record
 * @param atr PSX_STR attribute string, contains comma separated attribute list after invocation
 * @param val PSX_STR value string, contains comma separated value list after invocation
 */

BOOL sch_sql_get_av (PSX_SCH * sch,PSX_REC * rec,PSX_STR atr,PSX_STR val)
{
 PSX_STL * stl;
 BOOL r = TRUE;
 IDX i;

 if (sch == NULL || rec == NULL || atr == NULL || val == NULL)
  return (FALSE);


 if (!stl_create (&stl))			// create string list
  return (FALSE);

 if (!sch_sym_get_stl (sch,stl))	// get all field names (incl. components) from format spec and store in stl
  r = FALSE;

 strcpy (atr,"");
 strcpy (val,"");

 for (i = 0;i < stl -> n;i++)
 {
  PSX_STR a,v,s;
  char * t;

  sprintf (a,"f_%s",stl -> str [i]);	// f_stl[i]

  if (!rec_get (rec,stl -> str [i],&t))	// get value from record, store in t
   t = "";

  strcpy (s,t);
  
  sch_val_make (s);     // 's'
  sprintf (v,"%s",s);

  if (atr [0] != 0)	strcat (atr,",");
  if (val [0] != 0)	strcat (val,",");

  strcat (atr,a);
  strcat (val,v);
 }

 if (!stl_delete (&stl))
  r = FALSE;

 return (r);
}

/******************************************************************************/
/** get schema values over stringlist to field
 * @param sch PSX_SCH * pointer to schema interface structure
 * @param fld PSX_STR field name
 */

BOOL sch_get_fld (PSX_SCH * sch,PSX_STR fld)
{
 PSX_STL * stl;
 IDX i;

 if (sch == NULL || fld == NULL)
  return (FALSE);

 if (!stl_create (&stl))          // create string list
  return (FALSE);

 if (!sch_sym_get_stl (sch,stl))	// get values from sch
 {
  stl_delete (&stl);
  return (FALSE);
 }

 strcpy (fld,"");

 for (i = 0;i < stl -> n;i++)
 {
  PSX_STR t;

  if (i > 0)
   strcat (fld,",");
  sprintf (t,"f_%s",stl -> str [i]);
  strcat (fld,t);					// append stringlist values to fld
 }

 stl_delete (&stl);

 return (TRUE);
}

/******************************************************************************/
/** add dummy record
 * @param sch PSX_SCH * pointer to schema interface structure
 * @param rec PSX_REC * pointer to record
 */

BOOL sch_rec_dummy (PSX_SCH * sch,PSX_REC * rec)
{
 IDX i;

 if (sch == NULL || rec == NULL)
  return (FALSE);

 for (i = 0;i < sch -> fsp -> n;i++)
 {
  int j;

  sch_rec_add_if (sch,i,rec,CMP_NIL);

  for (j = 0;j < CMP_N;j++)
   sch_rec_add_if (sch,i,rec,j);
 }

 return (TRUE);
}

/******************************************************************************/
/** parse an extern file input by calling spi-parser
 * @param sch PSX_SCH * pointer to schema interface structure
 * @param f PSX_STR file name
 */

BOOL sch_parse (PSX_SCH * sch,PSX_STR f)
{
 PSX_SPI spi;
 SCP scp;

 if (sch == NULL || f == NULL)
  return (FALSE);

 sch_clear (sch);

 scp = SCP_FSP + SCP_RSP + SCP_PSP;

 if (!spi_parse (&spi,scp,f))
  return (FALSE);

 assert (sch -> fsp == NULL);


 assert (sch -> rsp == NULL);
 assert (sch -> psp == NULL);

 sch -> fsp = spi.fsp;
 sch -> rsp = spi.rsp;
 sch -> psp = spi.psp;

 assert (sch -> fsp != NULL);
 assert (sch -> rsp != NULL);
 assert (sch -> psp != NULL);


 return (TRUE);
}


/******************************************************************************/
/** load schema data from database (state, result, and field definitions)
 * @param sch PSX_SCH * pointer to schema interface structure
 * @param dbi DBI * pointer to database interface
 * @param idx IDX schema index
 */
 
BOOL sch_load (PSX_SCH * sch,DBI * dbi,IDX idx)
{
 PSX_STR sql;
 char * t;
 char var[SYS_STR_MAX];
 char val[SYS_STR_MAX];
 IDX i;


 if (sch == NULL || dbi == NULL)
  return (FALSE);

 sprintf (sql,"SELECT idx,sym,dtp,lbl,pos,len,mini,maxi,tfm,cps,equ FROM fld WHERE (sch=%d) ORDER BY idx",idx);

 if (!dbi_execute (dbi,sql))	// execute sql statement
  return (FALSE);

 while (dbi_fetch (dbi,0,NULL))
 {
  FMT_ITM f;


  if (!dbi_fetch (dbi,0,&t))
   return (FALSE);

  f.idx = atoi (t);

  if (!dbi_fetch (dbi,1,&t))
   return (FALSE);

  strcpy (f.sym,t);

  if (!dbi_fetch (dbi,2,&t))
   return (FALSE);

  if (!fsp_dtp_set (&f.dtp,t))
   return (FALSE);

  if (!dbi_fetch (dbi,3,&t))
   return (FALSE);

  strcpy (f.lbl,t);

  if (!dbi_fetch (dbi,4,&t))
   return (FALSE);

  f.pos = atoi (t);

  if (!dbi_fetch (dbi,5,&t))
   return (FALSE);

  f.len = atoi (t);

  if (!dbi_fetch (dbi,6,&t))
   return (FALSE);

  f.min = atoi (t);

  if (!dbi_fetch (dbi,7,&t))
   return (FALSE);

  f.max = atoi (t);

  if (!dbi_fetch (dbi,8,&t))
   return (FALSE);

  if (!fsp_tfm_dec (&f.tfm,t))
   return (FALSE);

  if (!dbi_fetch (dbi,9,&t))
   return (FALSE);

  if (!cpi_cps_set (&f.cps,t))
   return (FALSE);

  if (!dbi_fetch (dbi,10,&t))
   return (FALSE);

  if (!fsp_equ_dec (&f.equ,t))
   return (FALSE);

  if (!sch_fsp_add (sch,&f))
   return (FALSE);
 }

 /*************************************/

 sprintf (sql,"SELECT sym,prm,rum,msg,ntf FROM rsp WHERE (sch=%d) ORDER BY idx",idx);

 if (!dbi_execute (dbi,sql))
  return (FALSE);

 while (dbi_fetch (dbi,0,NULL)) // get all result specifications
 {
  RSP_ITM r;

  if (!dbi_fetch (dbi,0,&t))    // get name of result (sym)
   return (FALSE);

  strcpy (r.sym,t);

  if (!dbi_fetch (dbi,1,&t))    // get pid retrieval mode
   return (FALSE);

  if (!sch_prm_set (&r.prm,t))  // BG:ACCESS:hier steigt er aus
   return (FALSE);

  if (!dbi_fetch (dbi,2,&t))    // get rum (result update mode?)
   return (FALSE);

  if (!sch_rum_set (&r.rum,t))
   return (FALSE);
  /* changed 10/2007: 
   * Result messages can be specified in the configuration file.
   * Messages from configuration file are given precedence
   */      
  sprintf (var, "msg.req.%s", r.sym);
  if (psx_cfg_get(var, val))
  {
   strcpy (r.msg,val);
  }
  else 
  {
   if (!dbi_fetch (dbi,3,&t))    // get message
    return (FALSE);
   
   strcpy (r.msg,t);
  }

  if (!dbi_fetch (dbi,4,&t))     // get ntf
   return (FALSE);

  strcpy (r.not,t);

  if (!sch_rsp_add (sch,&r))
   return (FALSE);
 }

 /*************************************/

 sprintf (sql,"SELECT sym,sps,tg0,tg1,tgm FROM sts WHERE (sch=%d) ORDER BY idx",idx);  // get states from db

 if (!dbi_execute (dbi,sql))
  return (FALSE);

 while (dbi_fetch (dbi,0,NULL))
 {
  PSP_ITM s;

  if (!dbi_fetch (dbi,0,&t))      // get state name (sym)
   return (FALSE);

  strcpy (s.sym,t);

  if (!dbi_fetch (dbi,1,&t))      // get KSXO notation
   return (FALSE);


  if (!sch_sps_dec (&s.pro.stt,&s.pro.mtm,t))
   return (FALSE);

  if (!dbi_fetch (dbi,2,&t))      // get next state if no match
   return (FALSE);

  if (!sch_tgt_dec (sch,&s.epi.tgt_0,t))
   return (FALSE);


  if (!dbi_fetch (dbi,3,&t))      // get next state if one match
   return (FALSE);

  if (!sch_tgt_dec (sch,&s.epi.tgt_1,t))
   return (FALSE);

  if (!dbi_fetch (dbi,4,&t))      // get next state if multiple matches
   return (FALSE);

  if (!sch_tgt_dec (sch,&s.epi.tgt_m,t))
   return (FALSE);

  if (!sch_psp_add (sch,&s))
   return (FALSE);
 }

 for (i = 0;i < sch -> psp -> n;i++)
 {
  if (!sch_tgt_res (sch,&sch -> psp -> itm [i].epi.tgt_0))
   return (FALSE);

  if (!sch_tgt_res (sch,&sch -> psp -> itm [i].epi.tgt_1))
   return (FALSE);



  if (!sch_tgt_res (sch,&sch -> psp -> itm [i].epi.tgt_m))
   return (FALSE);
 }

 // TRACE ("SCH LOAD\n"); sch_print (sch);

 return (TRUE);
}

/******************************************************************************/

/** save new schema (state, result, and field definitions)
 * @param sch PSX_SCH * pointer to schema structure
 * @param dbi DBI * pointer to database interface structure
 * @param idx IDX * pointer to index
 */

BOOL sch_save (PSX_SCH * sch,DBI * dbi,IDX * idx)

{
 IDX i,idx_sch;
 PSX_STR s;

 if (sch == NULL || dbi == NULL || idx == NULL)
  return (FALSE);

 // TRACE ("SCH SAVE\n"); sch_print (sch);

 if (!dbi_idx_inc (dbi,"sch","idx",&idx_sch))
 {
  psx_put ("sch: idx inc failed\n");
  return (FALSE);
 }


 if (idx)
  *idx = idx_sch;

 /*****************/

 sprintf (s,"%d,'%s'",idx_sch,"Schema");

 if (!dbi_insert (dbi,"sch","idx,sym",s))
  return (FALSE);

 /*****************/

 for (i = 0;i < sch -> fsp -> n;i++)
 {
  IDX idx_itm;
  PSX_STR t,s_dtp,s_tfm,s_cps,s_equ;
  CPS cps;

  if (!dbi_idx_inc (dbi,"fld","idx",&idx_itm))
   return (FALSE);

 // printf ("> fmt [%2d]: '%s' -> idx: %d\n",i,fmt -> itm [i].var,idx_itm);

  cpi_cps_map_tfm (&cps,sch -> fsp -> itm [i].tfm);

  fsp_dtp_get (sch -> fsp -> itm [i].dtp,s_dtp);
  fsp_tfm_enc (sch -> fsp -> itm [i].tfm,s_tfm);
  // cpi_cps_get (sch -> fsp -> itm [i].cps,s_cps);
  cpi_cps_get (cps,s_cps);
  fsp_equ_enc (sch -> fsp -> itm [i].equ,s_equ);

  sprintf (t,"%d,%d,'%s','%s','%s',%d,%d,%d,%d,'%s','%s','%s'",
  idx_itm,idx_sch,
  sch -> fsp -> itm [i].sym,
  s_dtp,
  sch -> fsp -> itm [i].lbl,
  sch -> fsp -> itm [i].pos,
  sch -> fsp -> itm [i].len,
  sch -> fsp -> itm [i].min,
  sch -> fsp -> itm [i].max,
  s_tfm,s_cps,s_equ);

  if (!dbi_insert (dbi,"fld","idx,sch,sym,dtp,lbl,pos,len,mini,maxi,tfm,cps,equ",t))
   return (FALSE);
 }

 /*****************/

 for (i = 0;i < sch -> rsp -> n;i++)
 {
  IDX idx_itm;
  PSX_STR t,s_sym,s_prm,s_rum;
  CPS cps;
  char stg_m [0xFFFF],stg_n [0xFFFF];

  if (!dbi_idx_inc (dbi,"rsp","idx",&idx_itm))
   return (FALSE);

  strcpy (s_sym,sch -> rsp -> itm [i].sym);

  sch_prm_get (sch -> rsp -> itm [i].prm,s_prm);
  sch_rum_get (sch -> rsp -> itm [i].rum,s_rum);

  dbi_val_cvt (sch -> rsp -> itm [i].msg,stg_m);  // escape apostrophes
  dbi_val_cvt (sch -> rsp -> itm [i].not,stg_n);
  sprintf (t,"%d,%d,'%s','%s','%s','%s','%s'",idx_itm,idx_sch,s_sym,s_prm,s_rum,stg_m,stg_n);

  if (!dbi_insert (dbi,"rsp","idx,sch,sym,prm,rum,msg,ntf",t))
   return (FALSE);
 }

 /*****************/

 for (i = 0;i < sch -> psp -> n;i++)
 {
  PSX_STR s,t,t0,t1,tm;
  IDX idx_sts;

  if (!dbi_idx_inc (dbi,"sts","idx",&idx_sts))
   return (FALSE);

  if (!sch_sps_enc (sch -> psp -> itm [i].pro.stt,sch -> psp -> itm [i].pro.mtm,s))
   return (FALSE);

  if (!sch_tgt_enc (sch,&sch -> psp -> itm [i].epi.tgt_0,t0))
   return (FALSE);

  if (!sch_tgt_enc (sch,&sch -> psp -> itm [i].epi.tgt_1,t1))
   return (FALSE);

  if (!sch_tgt_enc (sch,&sch -> psp -> itm [i].epi.tgt_m,tm))
   return (FALSE);

  sprintf (t,"%d,%d,'%s','%s','%s','%s','%s'",
           idx_sts,idx_sch,sch -> psp -> itm [i].sym,s,t0,t1,tm);

  if (!dbi_insert (dbi,"sts","idx,sch,sym,sps,tg0,tg1,tgm",t))
   return (FALSE);

 }

 return (TRUE);
}

/******************************************************************************/
/** print status target info
 * @param tgt PSP_TGT * pointer to procedure specification target
 * @param sym char * symbol string
 */

static BOOL sch_print_sts_tgt (PSP_TGT * tgt,const char * sym)
{
 PSX_STR t,s;

 if (!sch_sdt_get (tgt -> sdt,t))
  strcpy (t,"?");


 switch (tgt -> sdt)
 {
  case SDT_ZERO:
   strcpy (s,"Z");
   break;

  case SDT_STS:
  case SDT_RSL:							// result
   sprintf (s,"%s: %s",t,tgt -> sym);	// status destination type
   break;

  default:
   printf ("ERROR\n");
   return (FALSE);
 }

 printf ("   tgt %s: %s\n",sym,s);

 return (TRUE);
}

/*****************************************************************************/
/** print schema on stdout
 * @param sch PSX_SCH * pointer to schema structure
 */
 
BOOL sch_print (PSX_SCH * sch)
{
 IDX i;
 PSX_STR t;

 if (sch == NULL)
  return (FALSE);

 printf ("Schema %p:\n\n",sch);                 // print schema
	
 fmt_print (sch -> fsp);                        // print format spec

 printf ("\nrsl:\n\n");

 for (i = 0;i < sch -> rsp -> n;i++)            // for every result spec itm
 {
  printf (" > %s\n",sch -> rsp -> itm [i].sym); // print result spec item
  sch_prm_get (sch -> rsp -> itm [i].prm,t);	
  printf ("   prm: %s, ",t);                    // print PID retrieval mode
  sch_rum_get (sch -> rsp -> itm [i].rum,t);	

  printf ("   rum: %s, ",t);                    // print record update mode
  printf ("   msg: %s\n",sch -> rsp -> itm [i].msg);
 }                                              // print info message

 printf ("\nsts:\n\n");

 for (i = 0;i < sch -> psp -> n;i++)			// for every procedure spec
 {
  //printf (" > %s [%04XH]\n",sch -> sts.itm [i].sym,sch -> sts.itm [i].pro.sps);

  switch (sch -> psp -> itm [i].pro.stt)		
  {
   case STT_INPUT: printf ("   input sure?\n");				break;
   case STT_QUERY: printf ("   query: %02XH\n",sch -> psp -> itm [i].pro.mtm); break;
   default:        printf ("   ?\n");                                 break;
  }

  sch_print_sts_tgt (&sch -> psp -> itm [i].epi.tgt_0,"0");
  sch_print_sts_tgt (&sch -> psp -> itm [i].epi.tgt_1,"1");
  sch_print_sts_tgt (&sch -> psp -> itm [i].epi.tgt_m,"*");
 }

 return (TRUE);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


