/******************************************************************************/
/* psx-sci.c                                   Standard Commandline Interface */
/******************************************************************************/
/** @file psx-sci.c Standard Commandline Interface - Source Code File
 * Functions providing the management of the commandline input. By starting the
 * psx programme information about the wanted functionality is given. This
 * information is stored in argv[] of the main function and passed on to this
 * interface, which provides functions to count, split or check the options. 
 */
 
#include "psx.h"


/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
 
/** SCI Structure */
static struct
{
 SYS_ARG * a;             /**< argument            */
 IDX       p;             /**< index of parameter  */
 IDX       o;             /**< index of option     */
 SCI_INT * int_active;    /**< active interface    */
} sci = { 0,0,0,0 };

#include "psx-sci.inl"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** recursive for each input textformat
 * @param itf interface
 * @param idn
 */
 
static BOOL sci_hlp_recurse (SCI_INT * itf,int idn)
{
 SYS_STR pad;
 IDX i;
 SCI_TGT tgt;

 if (itf == NULL)
  return (FALSE);

 memset (pad,' ',idn);    // pad = padding
 pad [idn] = 0;

 for (i = 0;sci_itf_get (itf,&i,SCI_ITP_TGT);i++)  // get itf
 {
  SYS_STR sym,t;

  sci_tgt_get_sym (itf [i].sym,sym,NULL);

  sprintf (t,"%s%s   ",pad,sym);
  sys_put ("%>%s",t,itf [i].syn); // print to screen
  //printf ("\n");

  if (!sci_tgt_fetch (itf,i,&tgt))
   continue;

  if (tgt.ttp == SCI_TTP_INT)
  {
   assert (tgt.itf != NULL);

   sci_hlp_recurse (tgt.itf,idn + 6);
   printf ("\n");
  }
 }

 return (TRUE);
}

/******************************************************************************/
/* Public                                                       Configuration */
/******************************************************************************/
/** configure sci structure
 * @param a pointer to system argument structure
 */
 
BOOL sci_cfg (SYS_ARG * a)
{
 sci.a          = a;
 sci.p          = 0;      // number of parameters 
 sci.o          = 0;      // number of options 
 sci.int_active = NULL;

 return (TRUE);
}

/******************************************************************************/
/* Public                                                Parameter Management */
/******************************************************************************/
/** get parameters from argument structure (argv).
 */
 
BOOL sci_par_get (IDX idx,char * s)
{
 IDX i,j;

 if (s == NULL)
  return (FALSE);

 assert (sci.a != NULL);

 j = 0;

 for (i = 0;i < sci.a -> c;i++)
 {
  if (sci.a -> v [i][0] == '-' || j++ < idx)		// '-' help
   continue;

  if (s)
   strcpy (s,sci.a -> v [i]);

  return (TRUE);
 }

 return (FALSE);
}

/******************************************************************************/
/** invoke parameter retrieval.
 * @param s string pointer to parameters after invocation.
 */
 
BOOL sci_par_peek (char * s)
{
 if (!sci_par_retrieve (s,NULL))	// retrieves command name from argv
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** fetch parameters
 * @param s string pointer to parameter string after invocation
 */
 
BOOL sci_par_fetch (char * s)
{
 IDX i;

 assert (sci.a != NULL);

 if (!sci_par_retrieve (s,&i))  // i = number of retrieved parameters
 {
  sci.p = 0;
  return (FALSE);
 }

 sci.p = i;

 return (TRUE);
}

/******************************************************************************/
/** check argv for completeness, from back.
 */
 
BOOL sci_par_back (void)
{
 IDX i;

 assert (sci.a != NULL);

 if (sci.p == 0)
  return (FALSE);

 for (i = sci.p - 1;i > 0;i--)      // backwards
 {
  if (sci.a -> v [i][0] == '-')     // parameter not complete e.g. "cmd -"
   continue;

  sci.p = i;                        

  return (TRUE);
 }

 sci.p = 0;

 return (TRUE);
}

/******************************************************************************/
/** count number of parameters.
 */
 
int sci_par_count (void)
{
 int i,n = 0;

 assert (sci.a != NULL);

 for (i = sci.p + 1;i < sci.a -> c;i++)
  if (sci.a -> v [i][0] != '-')
   n++;

 return (n);
}

/******************************************************************************/
/* Public                                                  Options Management */
/******************************************************************************/
/** fetch option, e.g "-o:f.out"
 * @param s string pointer to option name string after invocation
 * @param v string pointer to option value after invocation
 */
 
BOOL sci_opt_fetch (char * s,char * v)
{
 IDX i;

 assert (sci.a != NULL);

 if (s == NULL)
  return (FALSE);

 for (i = sci.o + 1;i < sci.a -> c;i++)
 {
  if (sci.a -> v [i][0] != '-')
   continue;

  sci.o = i;
  sci_opt_split (sci.a -> v [i] + 1,s,v);   // split option, e.g. "o" "f.out"
  return (TRUE);
 }

 sci.o = 0;

 return (FALSE);
}

/******************************************************************************/
/** find option with specified name
 * @param sym string pointer to option name
 */
 
BOOL sci_opt_switch (char * sym)
{
 SYS_STR v;

 if (sym == NULL)
  return (FALSE);

 if (!sci_opt_find (sym,v))
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** get specified option from commandline or default value and store in ptr
 * @param fmt format (optionname:optiontype) ( "c:s", "v:b", ...)                           
 * @param Pointer to where to store optionvalue                                            
 * @param def default value
 */

BOOL sci_opt_get (const char * fmt,PTR ptr,const char * def)
{
 SYS_STR sym,t,v;
 OTP otp;
 char * val;
 BOOL r = TRUE;

 if (fmt == NULL || ptr == NULL)
  return (FALSE);

 sci_opt_split ((char*)fmt,sym,t);  /* sym= optionname, t= optiontype        */

 if (!sci_opt_type (t,&otp))        /* get numeric code for type of option   */
  return (FALSE);

 if (sci_opt_find (sym,v))	        /* find value for option "sym" if supplied at command line */
  val = v;
 else
 {
  val = def;                        /* otherwise default value is assumed    */
  r = FALSE;
 }

 if (!sci_opt_store (otp,val,ptr))  /* verify and store option                */
  return (FALSE);

 return (r);
}

/******************************************************************************/
/* get option by splitting                                                    */

/* 16.09.2004: commented out, not needed
BOOL sci_opt_peek (SCI_INT_OPT * o,char * sym,PTR p)
{
 IDX    i;
 char * s;

 if (o == NULL || sym == NULL)
  return (FALSE);



 SCI_OPT_ITERATE (i,s)
 {
  SYS_STR s,v;

  sci_opt_split (sci.a -> v [i] + 1,s,v);

  if (!sci_opt_parse_put (o,s,v))
   continue;
 }

 return (TRUE);
}
*/


/******************************************************************************/
/* Public                                                  Control Management */
/******************************************************************************/
// fllt Target mit Werten, Optionen/Argumente werden geprft,
// Rekursion wird durch tgt.ttp == SCI_TTP_PRC unterbrochen -> Procedure gefunden,
// oder die Funktion wird dadurch unterbrochen, dass ein Ausdruck FALSE als Rckgabewert
// erzeugt.
// WICHTIG: tgt.prc (); ist der Aufruf einer relativ beliebigen Funktion im Programm
// mit Hilfe dieses Aufrufs wird die Hauptfunktionalitt von psx sichergestellt

/** fill target with values, options/arguments are checked, recursion is stopped,
 * when a procedure is found (tgt.ttp == SCI_TTP_PRC) or if an error occurs.
 * Important: "tgt.prc();" is an important call of a function in psx. The function
 * responding to this call is specified by the command given by argv.
 * @param itf interface
 * @param r return value of target procedure.
 */
 
BOOL sci_prc (SCI_INT * itf,int * r)
{
 SCI_TGT tgt;
 IDX i;

 if (r)				// if r != 0
  *r = -1;

 if (itf == NULL)
  return (FALSE);

 if (!sci_tgt_get (itf,&tgt)) //  get command name and procedure if defined
  return (FALSE);

 switch (tgt.ttp)             // target.type
 {
  case SCI_TTP_PRC:           //  0x01
  {
   SCI_INT * int_bak;
   int s;

   assert (tgt.prc != NULL);	// assert that function exists
   
   int_bak = sci.int_active;	// save active interface
   sci.int_active = itf;	    // new active interface

   s = tgt.prc ();            // call function, whose name ist stored in tgt.prc

   sci.int_active = int_bak;	// active interface as before function call

   if (r)
    *r = s;
   break;
  }

  case SCI_TTP_INT:		// SCI_INT
  {
   SCI_INT * int_bak;
   int s;

   assert (tgt.itf != NULL);

   int_bak = sci.int_active;
   sci.int_active = itf;

   if (!sci_prc (tgt.itf,&s))	// rekursiver Aufruf, bis prc gefunden oder False erzeugt
    s = -1;

   sci.int_active = int_bak;

   if (r)
    *r = s;
   break;
  }

  default:
   return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/ 
/** print help page
 * @param itf pointer to interface struccture
 */
BOOL sci_hlp_put (SCI_INT * itf)
{
 IDX i;
 char * syn;

 if (itf == NULL)
  if ((itf = sci.int_active) == NULL)
   return (FALSE);

 if (!sci_itf_get (itf,&i,SCI_ITP_SYN))
  return (FALSE);

 syn = itf [i].syn; // = "psx <cmd> {<par>} {<opt>}\n\n"

 assert (syn != NULL);

 printf ("Syntax: %s\n\n",syn);

 sci_hlp_recurse (itf,8);   // print out commands and options

 return (TRUE);
}

/******************************************************************************/ 

SCI_COMMAND (sci_hlp)
{
 if (!sci_hlp_put (NULL))
  return (-1);

 return (0);
}

/******************************************************************************/
/* Public                                                    Input Management */
/******************************************************************************/

BOOL sci_input (char * lbl,char * d)
{
 return (TRUE);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/





